<?php
namespace Highpots\SpamProtection\Admin;

use Highpots\SpamProtection\Helpers\HPSP_Constants;
use Highpots\SpamProtection\Helpers\HPSP_Utils;
use Highpots\SpamProtection\Integrations\HPSP_Form_Integration_Manager;


if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
class HPSP_Settings {

  
    private const OPTION_PREFIX = '_hpsp_';
    private const NONCE_ACTION = 'hpsp_update_settings';
    private const NONCE_NAME = 'hpsp_settings_nonce';
    private const LOGS_PAGE_SLUG = 'hpsp-logs';
    private const SETTINGS_PAGE_SLUG = 'hpsp-settings';
    
    private array $sections = [];
    private array $writing_systems = [];
    private array $default_settings = [];
    private static ?self $instance = null;

    private bool $initialized = false;

    private function __construct() {
        $this->init();
    }

    /**
     * Get singleton instance
     */
    public static function get_instance(): self {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Initialize the settings
     */
    private function init(): void {
        
        // Add plugin action links
        $plugin_basename = HPSP_PLUGIN_BASENAME;
        add_filter("plugin_action_links_{$plugin_basename}", [$this, 'add_plugin_action_links']);
        


   

        $this->initialize_default_settings();
    }


     /**
     * Ensure translations are initialized (lazy loading)
     */
    private function ensure_initialized(): void {
        if ($this->initialized) {
            return;
        }
        
        $this->initialize_writing_systems();
        $this->initialize_setting_sections();
        $this->initialized = true;
    }

    /**
     * Initialize writing systems
     */
    private function initialize_writing_systems(): void {
        $this->writing_systems = [
            'latin' => esc_html__('Latin (A-Z, a-z)',  'highpots-spam-protection'),
            'cyrillic' => esc_html__('Cyrillic (Russian, Bulgarian, etc.)',  'highpots-spam-protection'),
            'arabic' => esc_html__('Arabic',  'highpots-spam-protection'),
            'hebrew' => esc_html__('Hebrew',  'highpots-spam-protection'),
            'armenian' => esc_html__('Armenian',  'highpots-spam-protection'),
            'georgian' => esc_html__('Georgian',  'highpots-spam-protection'),
            'bengali' => esc_html__('Bengali/Bangla',  'highpots-spam-protection'),
            'chinese' => esc_html__('Chinese (Han characters)',  'highpots-spam-protection'),
            'japanese' => esc_html__('Japanese (Hiragana/Katakana)',  'highpots-spam-protection'),
            'korean' => esc_html__('Korean (Hangul)',  'highpots-spam-protection'),
            'greek' => esc_html__('Greek',  'highpots-spam-protection'),
            'devanagari' => esc_html__('Devanagari (Hindi, Sanskrit, etc.)',  'highpots-spam-protection'),
            'thai' => esc_html__('Thai',  'highpots-spam-protection'),
            'tamil' => esc_html__('Tamil',  'highpots-spam-protection'),
            'ethiopic' => esc_html__('Ethiopic (Amharic, Tigrinya)',  'highpots-spam-protection'),
        ];
    }

    /**
     * Initialize default settings
     */
    private function initialize_default_settings(): void {
        $this->default_settings = HPSP_Constants::get_defaults();
    }




    /**
     * Initialize setting sections and fields
     */
    private function initialize_setting_sections(): void {
        // Timing Settings Section
        $timing_section = new HPSP_Setting_Section(
            'timing',
            esc_html__('Time Settings',  'highpots-spam-protection'),
            esc_html__('Time span of how long a user takes to fill out the form All entries in seconds.',  'highpots-spam-protection')
        );

        $timing_section->add_field(new HPSP_Setting_Field(
            HPSP_Constants::get_min_time_id(),
            esc_html__('Minimum Time',  'highpots-spam-protection'),
            'number',
            $this->default_settings[HPSP_Constants::get_min_time_id()],
            esc_html__('The minimum time a user must spend before submitting the form.',  'highpots-spam-protection'),
            [],
            ['min' => 1, 'max' => 300, 'step' => 1],
            [$this, 'validate_min_time']
        ));

        $timing_section->add_field(new HPSP_Setting_Field(
            HPSP_Constants::get_max_time_id(),
            esc_html__('Maximum Time',  'highpots-spam-protection'),
            'number',
            $this->default_settings[HPSP_Constants::get_max_time_id()],
            esc_html__('The maximum time before a form submission expires.',  'highpots-spam-protection'),
            [],
            ['min' => 60, 'max' => 7200, 'step' => 60],
            [$this, 'validate_max_time']
        ));


         $timing_section->add_field(new HPSP_Setting_Field(
            HPSP_Constants::get_token_grace_period_id(),
            esc_html__('Token Grace Period',  'highpots-spam-protection'),
            'number',
            $this->default_settings[HPSP_Constants::get_token_grace_period_id()],
            esc_html__('Browser window - "Reload/refresh page": Spam bots often reload pages if the first input was incorrect. Time windows allowing token reuse for legitimate retries (network failures, double-clicks). Recommended: 30 to 60 seconds.',  'highpots-spam-protection'),
            [],
            ['min' => 0, 'max' => 300, 'step' => 1],
             [$this, 'validate_token_grace_period']

        ));


        $this->sections['timing'] = $timing_section;

        // Rate Limiting Section
        $rate_section = new HPSP_Setting_Section(
            'rate_limiting',
            esc_html__('Limit - Form submission restriction',  'highpots-spam-protection'),
            esc_html__('Configure submission rate limits to prevent rapid-fire spam attacks.',  'highpots-spam-protection'),
            [],
            '',
            esc_html__('Define how many form submissions per user are allowed within a certain time before spam protection is triggered.',  'highpots-spam-protection'),

        );

        $rate_section->add_field(new HPSP_Setting_Field(
            HPSP_Constants::get_enable_rate_limiting_id(),
            esc_html__('Enable Rate Limiting',  'highpots-spam-protection'),
            'checkbox',
            $this->default_settings[HPSP_Constants::get_enable_rate_limiting_id()],
            esc_html__('Enable rate limiting to prevent rapid-fire spam attacks.',  'highpots-spam-protection'),
            [],
            [],
            null,
            '',
            true // render as toggle
        ));

        $rate_section->add_field(new HPSP_Setting_Field(
            HPSP_Constants::get_rate_limit_window_id(),
            esc_html__('Time',  'highpots-spam-protection'),
            'number',
            $this->default_settings[HPSP_Constants::get_rate_limit_window_id()],
            esc_html__('Time window for counting submissions.',  'highpots-spam-protection'),
            [],
            ['min' => 10, 'max' => 3600, 'step' => 10],
            [$this, 'validate_rate_limit_window']
        ));

        $rate_section->add_field(new HPSP_Setting_Field(
            HPSP_Constants::get_rate_limit_max_id(),
            esc_html__('Maximum submissions per time',  'highpots-spam-protection'),
            'number',
            $this->default_settings[HPSP_Constants::get_rate_limit_max_id()],
            esc_html__('Maximum submissions allowed within the time window.',  'highpots-spam-protection'),
            [],
            ['min' => 1, 'max' => 100, 'step' => 1],
            [$this, 'validate_rate_limit_max']
        ));

        $this->sections['rate_limiting'] = $rate_section;

        // Content Filtering Section
        $content_section = new HPSP_Setting_Section(
            'content_filtering',
            esc_html__('Content Filtering',  'highpots-spam-protection'),
            esc_html__('Block submissions containing specific writing systems or characters.',  'highpots-spam-protection'),
            [],
            '',
            esc_html__('Spam attacks often come from abroad and fill out forms in the respective foreign language. Here you can block foreign languages. Note: block those languages from countries with which you have no (business) relations.',  'highpots-spam-protection'),
        );

        $content_section->add_field(new HPSP_Setting_Field(
            HPSP_Constants::get_blocked_writing_systems_id(),
            esc_html__('Blocked Writing Systems',  'highpots-spam-protection'),
            'checkbox_group',
            $this->default_settings[HPSP_Constants::get_blocked_writing_systems_id()],
            esc_html__('Select writing systems to block in form submissions.',  'highpots-spam-protection'),
            $this->writing_systems,
            [],
            [$this, 'validate_blocked_systems']
        ));

        $this->sections['content_filtering'] = $content_section;

        // Logging Section
        $logging_section = new HPSP_Setting_Section(
            'logging',
            esc_html__('Logging & Analytics',  'highpots-spam-protection'),
            esc_html__('Configure spam protection logging and data retention.',  'highpots-spam-protection')
        );

        $logging_section->add_field(new HPSP_Setting_Field(
            HPSP_Constants::get_enable_logging_id(),
            esc_html__('Enable Logging',  'highpots-spam-protection'),
            'checkbox',
            $this->default_settings[HPSP_Constants::get_enable_logging_id()],
            esc_html__('Log spam protection events for analysis and debugging.',  'highpots-spam-protection'),
            [],
            [],
            null,
            esc_html__('Here you can enable whether every successful form submission and every spam attempt should be logged.',  'highpots-spam-protection'),
            true // render as toggle
        ));

        $logging_section->add_field(new HPSP_Setting_Field(
            HPSP_Constants::get_log_retention_days_id(),
            esc_html__('Log Retention (days)',  'highpots-spam-protection'),
            'number',
            $this->default_settings[HPSP_Constants::get_log_retention_days_id()],
            esc_html__('How long to keep log entries before automatic cleanup.',  'highpots-spam-protection'),
            [],
            ['min' => 1, 'max' => 365, 'step' => 1],
            [$this, 'validate_log_retention'],
            esc_html__('Here you can define how long the log should retain form activity. The following information is stored: date, time, the form ID (useful when spam protection is enabled for multiple forms), the reason a form was blocked (suspected spam), the first two segments of the IP address, and a note about the respective spam protection (which spam filter blocked the form submission).',  'highpots-spam-protection')
        ));

        $logging_section->add_field(new HPSP_Setting_Field(
            HPSP_Constants::get_mask_ip_in_logs_id(),
            esc_html__('Mask IP Addresses',  'highpots-spam-protection'),
            'checkbox',
            $this->default_settings[HPSP_Constants::get_mask_ip_in_logs_id()],
            esc_html__('Display IP addresses partially masked (e.g., 192.168.xxx.xxx) in logs for privacy.',  'highpots-spam-protection'),
            [],
            [],
            null,
            '',
            true // render as toggle
        ));

        $this->sections['logging'] = $logging_section;

        // Validation Settings Section
        $validation_section = new HPSP_Setting_Section(
            'validation',
            esc_html__('Additional validation Settings',  'highpots-spam-protection'),
            esc_html__('Configure additional spam validation checks.',  'highpots-spam-protection')
        );

        $validation_section->add_field(new HPSP_Setting_Field(
            HPSP_Constants::get_enable_user_agent_validation_id(),
            esc_html__('Enable User Agent Validation',  'highpots-spam-protection'),
            'checkbox',
            $this->default_settings[HPSP_Constants::get_enable_user_agent_validation_id()],
            esc_html__('Block submissions with suspicious or missing User Agent headers.',  'highpots-spam-protection'),
            [],
            [],
            null,
            esc_html__('A "User-Agent" is a browser (Firefox, Chrome, Edge, etc.). Blocks form submissions if the User-Agent header is missing or looks "suspicious." Background: Every browser sends a "User-Agent" string with HTTP requests (e.g., Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:143.0) Gecko/20100101 Firefox/143.0). Many bots (or primitive scripts) omit this header or fill it with something very simple ("curl/7.68.0", "python-requests/2.22.0"). Result: Such requests can be blocked effectively. Caution: Some legitimate requests (e.g., from specific screen readers or security scanners) may also have unusual user agents. False positives possible.',  'highpots-spam-protection'),
            true // render as toggle
        ));

        $validation_section->add_field(new HPSP_Setting_Field(
            HPSP_Constants::get_enable_referrer_validation_id(),
            esc_html__('Enable Referrer Validation',  'highpots-spam-protection'),
            'checkbox',
            $this->default_settings[HPSP_Constants::get_enable_referrer_validation_id()],
            esc_html__('Block submissions that do not originate from your site.',  'highpots-spam-protection'),
            [],
            [],
            null,
            esc_html__('Blocks submissions that do not originate from your own domain. Normally, when submitting a form, a browser sends the HTTP header Referer. Bots that post directly to the form action URL without first loading the page often have no referrer or an incorrect one. Advantage: Simple protection against "direct POST" spam. Risk: Some legitimate users with aggressive privacy plugins block referrers. Corporate proxies or security gateways sometimes remove the referrer. Can block legitimate submissions if validation is too strict.',  'highpots-spam-protection'),
            true // render as toggle
        ));

        $this->sections['validation'] = $validation_section;

        // Form Plugin Integrations Section
        $integrations_section = new HPSP_Setting_Section(
            'integrations',
            esc_html__('Form Plugin Integrations',  'highpots-spam-protection'),
            esc_html__('Enable or disable spam protection for specific form plugins. When enabled, protection fields are injected server-side using native plugin hooks for better performance.',  'highpots-spam-protection')
        );

        $integrations_section->add_field(new HPSP_Setting_Field(
            HPSP_Constants::get_enable_cf7_integration_id(),
            esc_html__('Contact Form 7',  'highpots-spam-protection'),
            'checkbox',
            $this->default_settings[HPSP_Constants::get_enable_cf7_integration_id()],
            esc_html__('Enable spam protection for Contact Form 7 forms.',  'highpots-spam-protection'),
            [],
            [],
            null,
            '',
            true, // render as toggle
            fn() => HPSP_Utils::is_plugin_active('contact-form-7')
        ));

        $integrations_section->add_field(new HPSP_Setting_Field(
            HPSP_Constants::get_enable_gravity_forms_integration_id(),
            esc_html__('Gravity Forms',  'highpots-spam-protection'),
            'checkbox',
            $this->default_settings[HPSP_Constants::get_enable_gravity_forms_integration_id()],
            esc_html__('Enable spam protection for Gravity Forms.',  'highpots-spam-protection'),
            [],
            [],
            null,
            '',
            true, // render as toggle
            fn() => HPSP_Utils::is_plugin_active('gravityforms')
        ));

        $integrations_section->add_field(new HPSP_Setting_Field(
            HPSP_Constants::get_enable_wpforms_integration_id(),
            esc_html__('WPForms',  'highpots-spam-protection'),
            'checkbox',
            $this->default_settings[HPSP_Constants::get_enable_wpforms_integration_id()],
            esc_html__('Enable spam protection for WPForms.',  'highpots-spam-protection'),
            [],
            [],
            null,
            '',
            true, // render as toggle
            fn() => HPSP_Utils::is_plugin_active('wpforms')
        ));

        $integrations_section->add_field(new HPSP_Setting_Field(
            HPSP_Constants::get_enable_elementor_integration_id(),
            esc_html__('Elementor Forms',  'highpots-spam-protection'),
            'checkbox',
            $this->default_settings[HPSP_Constants::get_enable_elementor_integration_id()],
            esc_html__('Enable spam protection for Elementor forms.',  'highpots-spam-protection'),
            [],
            [],
            null,
            '',
            true, // render as toggle
            fn() => HPSP_Utils::is_plugin_active('elementor')
        ));

        $integrations_section->add_field(new HPSP_Setting_Field(
            HPSP_Constants::get_enable_formidable_integration_id(),
            esc_html__('Formidable Forms',  'highpots-spam-protection'),
            'checkbox',
            $this->default_settings[HPSP_Constants::get_enable_formidable_integration_id()],
            esc_html__('Enable spam protection for Formidable Forms.',  'highpots-spam-protection'),
            [],
            [],
            null,
            '',
            true, // render as toggle
            fn() => HPSP_Utils::is_plugin_active('formidable')
        ));

        $integrations_section->add_field(new HPSP_Setting_Field(
            HPSP_Constants::get_enable_generic_integration_id(),
            esc_html__('Generic Forms (JavaScript Fallback)',  'highpots-spam-protection'),
            'checkbox',
            $this->default_settings[HPSP_Constants::get_enable_generic_integration_id()],
            esc_html__('Protect unknown/generic forms using JavaScript injection. Disable to save bandwidth if you only use supported plugins.',  'highpots-spam-protection'),
            [],
            [],
            null,
            esc_html__('This fallback uses JavaScript to inject protection fields for forms not handled by specific integrations. Disabling this saves ~8-10KB of page weight if you only use supported form plugins.',  'highpots-spam-protection'),
            true, // render as toggle
            fn() => true  // Always show - works with any HTML form
        ));

        $this->sections['integrations'] = $integrations_section;
    }


    /**
     * Add plugin action links
     */
    public function add_plugin_action_links(array $links): array {
        $settings_link = sprintf(
            '<a href="%s">%s</a>',
            admin_url('admin.php?page=' . self::SETTINGS_PAGE_SLUG),
            esc_html__('Settings',  'highpots-spam-protection')
        );

        array_unshift($links, $settings_link);
        return $links;
    }

    /**
     * Get setting value
     */
    public function get_setting(string $key, mixed $default = null): mixed {
        $default_value = $default ?? ($this->default_settings[$key] ?? null);
        return get_option($key, $default_value);
    }

    /**
     * Update setting value
     */
    public function update_setting(string $key, mixed $value): bool {
        return update_option($key, $value);
    }

    /**
     * Get all settings
     */
    public function get_all_settings(): array {
        $settings = [];
        foreach ($this->default_settings as $key => $default_value) {
            $settings[$key] = $this->get_setting($key, $default_value);
        }
        return $settings;
    }

    /**
     * Render the settings page
     */
    public function render_settings_page(): void {
        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.',  'highpots-spam-protection'));
        }

        $this->ensure_initialized();

        $message = '';
        $message_type = '';

        // Handle form submission
        if (isset($_POST[self::NONCE_NAME]) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST[self::NONCE_NAME])), self::NONCE_ACTION)) {
            $result = $this->process_form_submission();
            $message = $result['message'];
            $message_type = $result['type'];
        }

        // Get current settings
        $current_settings = $this->get_all_settings();

        ?>
        <div class="hpsp-container">
            <?php if ($message): ?>
                <div class="hpsp-notice <?php echo esc_attr($message_type); ?> is-dismissible">
                    <p><?php echo esc_html($message); ?></p>
                    <button type="button" class="notice-dismiss">
                        <span class="screen-reader-text"><?php esc_html_e('Dismiss this notice.', 'highpots-spam-protection'); ?></span>
                    </button>
                </div>
            <?php endif; ?>

            <div>&nbsp;</div>

            <form method="post" action="">
                <?php wp_nonce_field(self::NONCE_ACTION, self::NONCE_NAME); ?>
                
                <div class="hpsp-settings-container">
                    <?php foreach ($this->sections as $section): ?>
                        <?php
                        // Check if section has any visible fields
                        $has_visible_fields = false;
                        foreach ($section->fields as $field) {
                            if ($field->should_render()) {
                                $has_visible_fields = true;
                                break;
                            }
                        }

                        // Skip empty sections
                        if (!$has_visible_fields) continue;
                        ?>
                        <div class="hpsp-settings-section">
                            <h2>
                                <?php echo esc_html($section->title); ?>
                                <?php if (!empty($section->title_help_text)): ?>
                                    <span class="hpsp-help-tooltip">
                                        <span class="dashicons dashicons-info hpsp-help-icon"></span>
                                        <span class="hpsp-help-text"><?php echo esc_html($section->title_help_text); ?></span>
                                    </span>
                                <?php endif; ?>
                            </h2>
                            <?php if ($section->description): ?>
                                <p class="description">
                                    <?php echo esc_html($section->description); ?>
                                    <?php if (!empty($section->desc_help_text)): ?>
                                        <span class="hpsp-help-tooltip">
                                            <span class="dashicons dashicons-info hpsp-help-icon"></span>
                                            <span class="hpsp-help-text"><?php echo esc_html($section->desc_help_text); ?></span>
                                        </span>
                                    <?php endif; ?>
                                </p>
                            <?php endif; ?>
                            
                            <table class="form-table hpsp-form-table" role="presentation">
                                <?php foreach ($section->fields as $field): ?>
                                    <?php if (!$field->should_render()) continue; ?>
                                    <?php
                                    // Check if this is a toggle field
                                    $is_toggle = $field->type === 'checkbox' && $field->render_as_toggle;
                                    ?>
                                    <tr valign="top" class="hpsp-field-row <?php echo esc_attr($is_toggle ? 'hpsp-toggle-row' : ''); ?>">
                                        <?php if (!$is_toggle): ?>
                                            <th scope="row" class="hpsp-field-label">
                                                <label for="<?php echo esc_attr($field->id); ?>">
                                                    <?php echo esc_html($field->title); ?>
                                                    <?php if (!empty($field->help_text)): ?>
                                                        <span class="hpsp-help-tooltip">
                                                            <span class="dashicons dashicons-info hpsp-help-icon"></span>
                                                            <span class="hpsp-help-text"><?php echo esc_html($field->help_text); ?></span>
                                                        </span>
                                                    <?php endif; ?>
                                                </label>
                                            </th>
                                        <?php endif; ?>
                                        <td colspan="<?php echo esc_attr($is_toggle ? '2' : '1'); ?>" class="hpsp-field-input">
                                            <?php $this->render_field($field, $current_settings[$field->id] ?? $field->default); ?>
                                            <?php if ($field->description): ?>
                                                <p class="description"><?php echo esc_html($field->description); ?></p>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </table>
                        </div>
                    <?php endforeach; ?>
                </div>

                <?php submit_button(esc_html__('Save Settings',  'highpots-spam-protection'), 'primary', 'submit', true, ['id' => 'hpsp-save-settings']); ?>
            </form>

            <div class="hpsp-settings-info">

                <?php $this->render_plugin_info(); ?>
                <?php $this->render_support_section(); ?>

            </div>
        </div>

        <?php
    }

private function render_plugin_info() {
    ?>
    <section class="hpsp-plugin-info-section">
        <div class="hpsp-info-header">
            <span class="dashicons dashicons-admin-plugins"></span>
            <h3><?php esc_html_e('Plugin Information', 'highpots-spam-protection'); ?></h3>
        </div>

        <div class="hpsp-info-content">
            <!-- Plugin Version -->
            <div class="hpsp-version-info">
                <div class="hpsp-version-item">
                    <span class="dashicons dashicons-info-outline"></span>
                    <div class="hpsp-version-details">
                        <strong><?php esc_html_e('Plugin Name:', 'highpots-spam-protection'); ?></strong>
                        <span><?php echo esc_html(HPSP_PLUGIN_DISPLAY_NAME); ?></span>
                    </div>
                </div>

                <div class="hpsp-version-item">
                    <span class="dashicons dashicons-update"></span>
                    <div class="hpsp-version-details">
                        <strong><?php esc_html_e('Version:', 'highpots-spam-protection'); ?></strong>
                        <span class="hpsp-version-number"><?php echo esc_html(HPSP_PLUGIN_VERSION); ?></span>
                    </div>
                </div>

                <div class="hpsp-version-item">
                    <span class="dashicons dashicons-calendar-alt"></span>
                    <div class="hpsp-version-details">
                        <strong><?php esc_html_e('PHP Version:', 'highpots-spam-protection'); ?></strong>
                        <span><?php echo esc_html(PHP_VERSION); ?></span>
                    </div>
                </div>

                <div class="hpsp-version-item">
                    <span class="dashicons dashicons-wordpress"></span>
                    <div class="hpsp-version-details">
                        <strong><?php esc_html_e('WordPress Version:', 'highpots-spam-protection'); ?></strong>
                        <span><?php echo esc_html(get_bloginfo('version')); ?></span>
                    </div>
                </div>
            </div>

            <div class="hpsp-info-divider"></div>

        </div>
    </section>
    <?php
}



private function render_support_section() {
    ?>
    <section class="hpsp-support-section">
        <div class="hpsp-support-header">
            <span class="dashicons dashicons-sos"></span>
            <h3><?php esc_html_e('Support', 'highpots-spam-protection'); ?></h3>
        </div>

        <div class="hpsp-support-content">
            <p class="hpsp-support-intro">
                <?php esc_html_e('Support is provided directly by the manufacturer.', 'highpots-spam-protection'); ?>
            </p>

            <div class="hpsp-support-grid">
                <div class="hpsp-support-item">
                    <span class="dashicons dashicons-building"></span>
                    <div class="hpsp-support-details">
                        <strong><?php esc_html_e('Manufacturer:', 'highpots-spam-protection'); ?></strong>
                        <span>HighPots</span>
                    </div>
                </div>

                <div class="hpsp-support-item">
                    <span class="dashicons dashicons-admin-site"></span>
                    <div class="hpsp-support-details">
                        <strong><?php esc_html_e('Web:', 'highpots-spam-protection'); ?></strong>
                        <a href="https://highpots.com" target="_blank" rel="noopener noreferrer">
                            highpots.com
                        </a>
                    </div>
                </div>

                <div class="hpsp-support-item hpsp-support-email">
                    <span class="dashicons dashicons-email"></span>
                    <div class="hpsp-support-details">
                        <strong>
                            <?php esc_html_e('Email:', 'highpots-spam-protection'); ?>
                            <span class="hpsp-help-tooltip">
                                <span class="dashicons dashicons-info hpsp-help-icon"></span>
                                <span class="hpsp-help-text"><?php esc_html_e('This email address is connected to a ticketing system. After you send the email, you will receive a ticket number. In general, we resolve support requests within 1 to 2 business days.', 'highpots-spam-protection'); ?></span>
                            </span>
                        </strong>
                        <a href="mailto:support@highpots.com">support@highpots.com</a>
                    </div>
                </div>

                <div class="hpsp-support-item">
                    <span class="dashicons dashicons-phone"></span>
                    <div class="hpsp-support-details">
                        <strong><?php esc_html_e('Phone Germany:', 'highpots-spam-protection'); ?></strong>
                        <a href="tel:+4971619659279">+49 7161 965 927 965</a>
                    </div>
                </div>

                <div class="hpsp-support-item">
                    <span class="dashicons dashicons-phone"></span>
                    <div class="hpsp-support-details">
                        <strong><?php esc_html_e('Phone Switzerland:', 'highpots-spam-protection'); ?></strong>
                        <a href="tel:+41415520189">+41 41 552 01 89</a>
                    </div>
                </div>

                <div class="hpsp-support-item">
                    <span class="dashicons dashicons-whatsapp"></span>
                    <div class="hpsp-support-details">
                        <strong>WhatsApp:</strong>
                        <a href="https://wa.me/4917687459637" target="_blank" rel="noopener noreferrer">
                            +49 176 874 596 37
                        </a>
                    </div>
                </div>

                <div class="hpsp-support-item hpsp-support-form">
                    <span class="dashicons dashicons-format-aside"></span>
                    <div class="hpsp-support-details">
                        <strong>
                            <?php esc_html_e('Support web form:', 'highpots-spam-protection'); ?>
                            <span class="hpsp-help-tooltip">
                                <span class="dashicons dashicons-info hpsp-help-icon"></span>
                                <span class="hpsp-help-text"><?php esc_html_e('Use our web form for detailed support requests. You can attach screenshots and files.', 'highpots-spam-protection'); ?></span>
                            </span>
                        </strong>
                        <a href="https://highpots.com/softwareprodukte/support-softwareprodukte/" target="_blank" rel="noopener noreferrer">
                            <?php esc_html_e('Open support form', 'highpots-spam-protection'); ?>
                            <span class="dashicons dashicons-external"></span>
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </section>
    <?php
}

    /**
     * Render individual field
     */
    private function render_field(HPSP_Setting_Field $field, mixed $value): void {
        switch ($field->type) {
            case 'number':
                $this->render_number_field($field, $value);
                break;
            case 'checkbox':
                $this->render_checkbox_field($field, $value);
                break;
            case 'checkbox_group':
                $this->render_checkbox_group_field($field, $value);
                break;
            case 'select':
                $this->render_select_field($field, $value);
                break;
            case 'text':
            default:
                $this->render_text_field($field, $value);
                break;
        }
    }

    /**
     * Render number field
     */
    private function render_number_field(HPSP_Setting_Field $field, mixed $value): void {
        $attributes = array_merge([
            'type' => 'number',
            'id' => $field->id,
            'name' => $field->id,
            'value' => $value,
            'class' => 'regular-text',
            'data-default' => $field->default
        ], $field->attributes);

        echo '<div class="input-with-clear">';
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Escaped in build_input_tag()
        echo $this->build_input_tag($attributes);
        echo '<span class="clear-input dashicons dashicons-dismiss" title="' . esc_attr__('Reset to default',  'highpots-spam-protection') . '"></span>';
        echo '</div>';
    }

    /**
     * Render text field
     */
    private function render_text_field(HPSP_Setting_Field $field, mixed $value): void {
        $attributes = array_merge([
            'type' => 'text',
            'id' => $field->id,
            'name' => $field->id,
            'value' => $value,
            'class' => 'regular-text'
        ], $field->attributes);

        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Escaped in build_input_tag()
        echo $this->build_input_tag($attributes);
    }

    /**
     * Render checkbox field
     */
    private function render_checkbox_field(HPSP_Setting_Field $field, mixed $value): void {
        $checked = checked($value, true, false);

        if ($field->render_as_toggle) {
            // Render as compact toggle switch with label on right
            echo '<div class="hpsp-toggle-wrapper">';
            echo '<label class="hpsp-toggle-switch">';
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- $checked is from checked() which is safe
            echo '<input type="checkbox" id="' . esc_attr($field->id) . '" name="' . esc_attr($field->id) . '" value="1" ' . $checked . '>';
            echo '<span class="hpsp-toggle-slider"></span>';
            echo '</label>';
            echo '<span class="hpsp-toggle-label">' . esc_html($field->title) . '</span>';

            // Add info icon with tooltip if help_text is available
            if (!empty($field->help_text)) {
                echo '<span class="hpsp-help-tooltip">';
                echo '<span class="dashicons dashicons-info hpsp-help-icon"></span>';
                echo '<span class="hpsp-help-text">' . esc_html($field->help_text) . '</span>';
                echo '</span>';
            } 

            echo '</div>';
        } else {
            // Render as regular checkbox
            echo '<label>';
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- $checked is from checked() which is safe
            echo '<input type="checkbox" id="' . esc_attr($field->id) . '" name="' . esc_attr($field->id) . '" value="1" ' . $checked . '>';
            echo ' ' . esc_html($field->title);
            echo '</label>';
        }
    }

    /**
     * Render checkbox group field
     */
    private function render_checkbox_group_field(HPSP_Setting_Field $field, mixed $value): void {
        $selected_values = is_array($value) ? $value : [];
        
        echo '<div class="checkbox-group">';
        foreach ($field->options as $option_value => $option_label) {
            $checked = checked(in_array($option_value, $selected_values), true, false);
            echo '<label>';
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- $checked is from checked() which is safe
            echo '<input type="checkbox" name="' . esc_attr($field->id) . '[]" value="' . esc_attr($option_value) . '" ' . $checked . '>';
            echo ' ' . esc_html($option_label);
            echo '</label>';
        }
        echo '</div>';
    }

    /**
     * Render select field
     */
    private function render_select_field(HPSP_Setting_Field $field, mixed $value): void {
        echo '<select id="' . esc_attr($field->id) . '" name="' . esc_attr($field->id) . '" class="regular-text">';
        foreach ($field->options as $option_value => $option_label) {
            $selected = selected($value, $option_value, false);
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- $selected is from selected() which is safe
            echo '<option value="' . esc_attr($option_value) . '" ' . $selected . '>' . esc_html($option_label) . '</option>';
        }
        echo '</select>';
    }

    /**
     * Build and render input element with escaped attributes
     *
     * @param array $attributes Array of HTML attributes for the input element
     * @return string Complete HTML input tag with properly escaped attributes
     */
    private function build_input_tag(array $attributes): string {
        $html_attributes = [];
        foreach ($attributes as $key => $value) {
            if ($value !== null && $value !== false) {
                $html_attributes[] = esc_attr($key) . '="' . esc_attr($value) . '"';
            }
        }
        return '<input ' . implode(' ', $html_attributes) . '>';
    }

    /**
     * Process form submission
     */
   private function process_form_submission(): array {
    $errors = [];
    $updated_settings = [];
    
    // Track if logging setting changed
    $old_logging_enabled = $this->get_setting(HPSP_Constants::get_enable_logging_id(), true);

    foreach ($this->sections as $section) {
        foreach ($section->fields as $field) {
              // Special handling for checkboxes
            if ($field->type === 'checkbox') {
                // Checkbox unchecked = not in $_POST = 0, checked = 1
                // Store as integer to avoid WordPress get_option() falsy value issue
                // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in render_settings_page() before calling this method
                $posted_value = isset($_POST[$field->id]) ? 1 : 0;
            } elseif ($field->type === 'checkbox_group') {
                // Checkbox group: not in $_POST = empty array (none selected)
                // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in render_settings_page() before calling this method
                if (isset($_POST[$field->id]) && is_array($_POST[$field->id])) {
                    // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce verified in render_settings_page(), sanitization happens immediately below
                    $posted_value = array_map('sanitize_text_field', wp_unslash($_POST[$field->id]));
                } else {
                    $posted_value = [];
                }
            } else {
                // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in render_settings_page() before calling this method
                if (isset($_POST[$field->id])) {
                    // Sanitize early based on field type
                    // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce verified in render_settings_page(), sanitization happens immediately below based on field type
                    $raw_value = wp_unslash($_POST[$field->id]);
                    if ($field->type === 'number' || $field->type === 'select') {
                        $posted_value = sanitize_text_field($raw_value);
                    } elseif ($field->type === 'textarea') {
                        $posted_value = sanitize_textarea_field($raw_value);
                    } elseif ($field->type === 'multiselect' && is_array($raw_value)) {
                        $posted_value = array_map('sanitize_text_field', $raw_value);
                    } else {
                        $posted_value = sanitize_text_field($raw_value);
                    }
                } else {
                    $posted_value = $field->default;
                }
            }

            // Validate field
            if ($field->validator && is_callable($field->validator)) {
                $validation_result = call_user_func($field->validator, $posted_value, $field);
                if (is_array($validation_result) && isset($validation_result['error'])) {
                    $errors[] = sprintf('%s: %s', $field->title, $validation_result['error']);
                    continue;
                }
                $posted_value = $validation_result['value'] ?? $posted_value;
            }

            $updated_settings[$field->id] = $posted_value;
            $this->update_setting($field->id, $posted_value);
        }
    }

    // Validate timing relationships
    $min_time = $updated_settings[HPSP_Constants::get_min_time_id()] ?? $this->get_setting(HPSP_Constants::get_min_time_id());
    $max_time = $updated_settings[HPSP_Constants::get_max_time_id()] ?? $this->get_setting(HPSP_Constants::get_max_time_id());
    
    if ($min_time >= $max_time) {
        $errors[] = esc_html__('Minimum time must be less than maximum time.', 'highpots-spam-protection');
        // Reset to defaults
        $this->update_setting(HPSP_Constants::get_min_time_id(), $this->default_settings[HPSP_Constants::get_min_time_id()]);
        $this->update_setting(HPSP_Constants::get_max_time_id(), $this->default_settings[HPSP_Constants::get_max_time_id()]);
    }

    // Check if logging setting changed
    $new_logging_enabled = $updated_settings[HPSP_Constants::get_enable_logging_id()] ?? $old_logging_enabled;
    if ($old_logging_enabled !== $new_logging_enabled) {
        // Fire action to notify scheduler
        do_action('hpsp_logging_setting_changed', $new_logging_enabled);
    }

    if (!empty($errors)) {
        return [
            'message' => esc_html__('Settings saved with warnings: ',  'highpots-spam-protection') . implode(' ', $errors),
            'type' => 'notice-warning'
        ];
    }

    return [
        'message' => esc_html__('Settings saved successfully.',  'highpots-spam-protection'),
        'type' => 'notice-success'
    ];
}

    /**
     * Sanitize field value
     */
    public function sanitize_field($value): mixed {
        if (is_array($value)) {
            return array_map('sanitize_text_field', $value);
        }
        return sanitize_text_field($value);
    }

    // Validation methods
    public function validate_min_time($value, $field): array {
        $value = absint($value);
        if ($value < 1) {
            return ['error' => esc_html__('Minimum time must be at least 1 second.',  'highpots-spam-protection')];
        }
        if ($value > 300) {
            return ['error' => esc_html__('Minimum time cannot exceed 300 seconds.',  'highpots-spam-protection')];
        }
        return ['value' => $value];
    }


    public function validate_max_time($value, $field): array {
        $value = absint($value);
        if ($value < 60) {
            return ['error' => esc_html__('Maximum time must be at least 60 seconds.',  'highpots-spam-protection')];
        }
        if ($value > 7200) {
            return ['error' => esc_html__('Maximum time cannot exceed 7200 seconds (2 hours).',  'highpots-spam-protection')];
        }
        return ['value' => $value];
    }

    /**
     * Validates token grace period value.
     *
     * @param mixed $value The value to validate.
     * @return int The validated value.
     */
    public function validate_token_grace_period($value, $field): array {
        $value = absint($value);

        // Ensure value is within reasonable bounds
        if ($value < 0) {
            return ['error' => esc_html__('Token grace period cannot be negative. Using default value (30 seconds).',  'highpots-spam-protection')];


        }

        if ($value > 300) {
            return ['error' => esc_html__('Token grace period too high (max 300 seconds). Using maximum value.',  'highpots-spam-protection')];
        }

        // Warn if set to 0
        /* if ($value === 0) {
            return ['error' => esc_html__('Warning: Token grace period set to 0. AJAX forms may fail on network retries.',  'highpots-spam-protection')];
        } */

        return ['value' => $value];
    }

    public function validate_rate_limit_window($value, $field): array {
        $value = absint($value);
        if ($value < 10) {
            return ['error' => esc_html__('Rate limit window must be at least 10 seconds.',  'highpots-spam-protection')];
        }
        if ($value > 3600) {
            return ['error' => esc_html__('Rate limit window cannot exceed 300 seconds.',  'highpots-spam-protection')];
        }
        return ['value' => $value];
    }

    public function validate_rate_limit_max($value, $field): array {
        $value = absint($value);
        if ($value < 1) {
            return ['error' => esc_html__('Maximum submissions must be at least 1.',  'highpots-spam-protection')];
        }
        if ($value > 100) {
            return ['error' => esc_html__('Maximum submissions cannot exceed 100.',  'highpots-spam-protection')];
        }
        return ['value' => $value];
    }

    public function validate_log_retention($value, $field): array {
        $value = absint($value);
        if ($value < 1) {
            return ['error' => esc_html__('Log retention must be at least 1 day.',  'highpots-spam-protection')];
        }
        if ($value > 365) {
            return ['error' => esc_html__('Log retention cannot exceed 365 days.',  'highpots-spam-protection')];
        }
        return ['value' => $value];
    }

    public function validate_blocked_systems($value, $field): array {
        if (!is_array($value)) {
            return ['value' => []];
        }
        
        $allowed_systems = array_keys($this->writing_systems);
        $sanitized = [];
        
        foreach ($value as $system) {
            $system = sanitize_key($system);
            if (in_array($system, $allowed_systems)) {
                $sanitized[] = $system;
            }
        }
        
        return ['value' => $sanitized];
    }

}