<?php
namespace Highpots\SpamProtection\Admin;


if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Represents a single setting field in the plugin settings.
 *
 * This class encapsulates all the properties needed to render and
 * validate a setting field in the WordPress admin settings page.
 *
 * @package Highpots\SpamProtection\Admin
 */
class HPSP_Setting_Field {
    /** @var string Unique identifier for this field */
    public string $id;

    /** @var string Display title/label for the field */
    public string $title;

    /** @var string Field type (text, number, checkbox, select, etc.) */
    public string $type;

    /** @var mixed Default value for the field */
    public mixed $default;

    /** @var string Help text description displayed below the field */
    public string $description;

    /** @var string Tooltip text displayed on hover (optional) */
    public string $help_text;

    /** @var array Options array for select/checkbox_group fields */
    public array $options;

    /** @var array HTML attributes for the field element */
    public array $attributes;

    /** @var callable|null Optional validation callback function */
    public $validator;

    /** @var bool Whether to render checkbox as toggle switch */
    public bool $render_as_toggle;

    /** @var callable|null Optional condition callback to determine if field should be rendered */
    private $condition;

    /**
     * Constructor for setting field.
     *
     * @param string $id Unique identifier for this field
     * @param string $title Display title/label for the field
     * @param string $type Field type (text, number, checkbox, select, checkbox_group)
     * @param mixed $default Default value for the field
     * @param string $description Help text displayed below the field
     * @param array $options Options for select/checkbox_group fields (key => label pairs)
     * @param array $attributes HTML attributes for the field element (e.g., min, max, step)
     * @param callable|null $validator Optional validation callback function that receives value and field
     * @param string $help_text Tooltip text displayed on hover (optional)
     * @param bool $render_as_toggle Whether to render checkbox as toggle switch (only applies to checkbox type)
     * @param callable|null $condition Optional condition callback to determine if field should be rendered
     */
    public function __construct(
        string $id,
        string $title,
        string $type = 'text',
        mixed $default = '',
        string $description = '',
        array $options = [],
        array $attributes = [],
        callable $validator = null,
        string $help_text = '',
        bool $render_as_toggle = false,
        ?callable $condition = null
    ) {
        $this->id = $id;
        $this->title = $title;
        $this->type = $type;
        $this->default = $default;
        $this->description = $description;
        $this->options = $options;
        $this->attributes = $attributes;
        $this->validator = $validator;
        $this->help_text = $help_text;
        $this->render_as_toggle = $render_as_toggle;
        $this->condition = $condition;
    }

    /**
     * Determine if this field should be rendered based on its condition.
     *
     * @return bool True if the field should be rendered, false otherwise
     */
    public function should_render(): bool {
        if ($this->condition === null) {
            return true;  // No condition = always render
        }
        return call_user_func($this->condition);
    }
}
    
