<?php
namespace Highpots\SpamProtection\Admin;

use Highpots\SpamProtection\Helpers\HPSP_Salt_Generator;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Displays admin notices for missing salts.
 *
 * @package Highpots\SpamProtection\Admin

 */
class HPSP_Salt_Admin_Notice {

	/**
	 * Singleton instance.
	 *
	 * @var HPSP_Salt_Admin_Notice|null
	 */
	private static $instance = null;

	/**
	 * Initialize hooks.
	 * Private constructor to enforce singleton pattern.
	 */
	private function __construct() {
		add_action( 'admin_notices', [ $this, 'show_missing_salts_notice' ], 999 );
		add_action( 'admin_post_hpsp_generate_salts', [ $this, 'handle_generate_salts' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_scripts' ] );
		add_action( 'wp_ajax_hpsp_dismiss_salt_notice', [ $this, 'dismiss_salt_notice' ] );
		add_action( 'wp_ajax_hpsp_generate_salts_ajax', [ $this, 'handle_generate_salts_ajax' ] );

	}
	
	/**
	 * Get singleton instance.
	 *
	 * @return HPSP_Salt_Admin_Notice
	 */
	public static function get_instance(): self {
		if ( self::$instance === null ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Enqueue admin scripts for clipboard functionality.
	*/
	public function enqueue_scripts(): void {

		$missing_salts = HPSP_Salt_Generator::get_missing_salts();



		// Only load if salts are missing
		if ( empty( $missing_salts ) ) {
			return;
		}

		// Ensure dependencies are loaded
		wp_enqueue_script( 'jquery' );

		// Inline script (no external file needed)
		wp_add_inline_script( 'jquery', $this->get_clipboard_script() );

		// Add CSS for button styling - use admin-bar which is always loaded in admin
		wp_add_inline_style( 'dashicons', $this->get_clipboard_styles() );
	}

	/**
	 * Gets the clipboard JavaScript code.
	 *
	 * @return string JavaScript code.
	 */
	private function get_clipboard_script(): string {
		return "
		document.addEventListener('DOMContentLoaded', function() {
			const copyButton = document.getElementById('hpsp-copy-salts');
			if (!copyButton) return;

			copyButton.addEventListener('click', function(e) {
				e.preventDefault();

				const textarea = document.getElementById('hpsp-salts-code');
				if (!textarea) return;

				// Select and copy text
				textarea.select();
				textarea.setSelectionRange(0, 99999); // For mobile devices

				try {
					// Modern approach (Clipboard API)
					if (navigator.clipboard && window.isSecureContext) {
						navigator.clipboard.writeText(textarea.value).then(function() {
							showCopyFeedback(copyButton, true);
						}).catch(function(err) {
							// Fallback to execCommand
							fallbackCopy(textarea, copyButton);
						});
					} else {
						// Fallback for older browsers
						fallbackCopy(textarea, copyButton);
					}
				} catch (err) {
					showCopyFeedback(copyButton, false);
				}
			});

			function fallbackCopy(textarea, button) {
				try {
					const successful = document.execCommand('copy');
					showCopyFeedback(button, successful);
				} catch (err) {
					showCopyFeedback(button, false);
				}
			}

			function showCopyFeedback(button, success) {
				const originalHTML = button.innerHTML;
				const originalClass = button.className;

				if (success) {
					button.innerHTML = '<span class=\"dashicons dashicons-yes\"></span> " . esc_js( __( 'Copied!', 'highpots-spam-protection' ) ) . "';
					button.className = button.className.replace('button-secondary', 'button-primary');
					button.disabled = true;
				} else {
					button.innerHTML = '<span class=\"dashicons dashicons-no\"></span> " . esc_js( __( 'Copy Failed', 'highpots-spam-protection' ) ) . "';
				}

				// Reset after 2 seconds
				setTimeout(function() {
					button.innerHTML = originalHTML;
					button.className = originalClass;
					button.disabled = false;
				}, 2000);
			}

			// Handle dismiss notice
			const dismissButton = document.querySelector('.hpsp-salt-notice .notice-dismiss');
			if (dismissButton) {
				dismissButton.addEventListener('click', function(e) {
					fetch(ajaxurl, {
						method: 'POST',
						headers: {
							'Content-Type': 'application/x-www-form-urlencoded',
						},
						body: new URLSearchParams({
							action: 'hpsp_dismiss_salt_notice',
							hpsp_dismiss_salt_notice_nonce: '" . wp_create_nonce( 'hpsp_dismiss_salt_notice' ) . "'
						})
					});
				});
			}

			// Handle AJAX salt generation
			const generateButton = document.getElementById('hpsp-generate-salts-button');
			if (generateButton) {
				generateButton.addEventListener('click', function(e) {
					e.preventDefault();
					console.log('Generate salts button clicked');

					const button = this;
					const originalHTML = button.innerHTML;
					const messageContainer = document.getElementById('hpsp-generate-message');

					// Disable button and show loading
					button.disabled = true;
					button.innerHTML = '<span class=\"dashicons dashicons-update dashicons-spin\"></span> " . esc_js( __( 'Generating...', 'highpots-spam-protection' ) ) . "';

					// Clear previous messages
					if (messageContainer) {
						messageContainer.innerHTML = '';
						messageContainer.style.display = 'none';
					}

					console.log('Sending AJAX request to:', ajaxurl);
					fetch(ajaxurl, {
						method: 'POST',
						headers: {
							'Content-Type': 'application/x-www-form-urlencoded',
						},
						body: new URLSearchParams({
							action: 'hpsp_generate_salts_ajax',
							hpsp_generate_salts_nonce: '" . wp_create_nonce( 'hpsp_generate_salts' ) . "'
						})
					})
					.then(function(response) {
						return response.json();
					})
					.then(function(data) {
						console.log('AJAX Response:', data);
						if (messageContainer) {
							messageContainer.style.display = 'block';
							if (data.success) {
								messageContainer.className = 'hpsp-notice notice-success';
								messageContainer.innerHTML = '<p>' + data.data.message + '</p>';

								// Auto-reload after 3 seconds on success
								setTimeout(function() {
									window.location.reload();
								}, 3000);
							} else {
								messageContainer.className = 'hpsp-notice notice-error';
								const errorMsg = data.data && data.data.message ? data.data.message : 'Unknown error occurred';
								messageContainer.innerHTML = '<p>' + errorMsg + '</p>';

								// Re-enable button on error
								button.disabled = false;
								button.innerHTML = originalHTML;
							}
						}
					})
					.catch(function(error) {
						if (messageContainer) {
							messageContainer.style.display = 'block';
							messageContainer.className = 'hpsp-notice notice-error';
							messageContainer.innerHTML = '<p><strong>" . esc_js( __( 'Error:', 'highpots-spam-protection' ) ) . "</strong> ' + error.message + '</p>';
						}

						// Re-enable button on error
						button.disabled = false;
						button.innerHTML = originalHTML;
					});
				});
			}
		});
		";
	}

	/**
	 * Gets the CSS styles for clipboard button.
	 *
	 * @return string CSS code.
	 */
	private function get_clipboard_styles(): string {
		return "
		.hpsp-salt-notice {
			position: relative;
			border-left: 4px solid #f0b849;
			background: #fff9e6;
			padding: 15px 20px !important;
		}
		
		.hpsp-salt-notice .notice-title {
			display: flex;
			align-items: center;
			gap: 10px;
			margin: 0 0 15px 0;
			font-size: 16px;
			font-weight: 600;
			color: #1d2327;
		}
		
		.hpsp-salt-notice .notice-title .dashicons {
			font-size: 24px;
			width: 24px;
			height: 24px;
			color: #f0b849;
		}
		
		.hpsp-salt-notice p {
			margin: 10px 0;
			line-height: 1.6;
		}
		
		.hpsp-salt-notice ul {
			margin: 10px 0 10px 20px;
		}
		
		.hpsp-salt-notice ul li {
			margin: 5px 0;
		}
		
		.hpsp-salt-notice code {
			background: #f0f0f1;
			padding: 2px 6px;
			border-radius: 3px;
			font-family: 'Courier New', Courier, monospace;
			font-size: 13px;
			color: #d63638;
		}
		
		.hpsp-notice-section {
			background: #fff;
			border: 1px solid #dcdcde;
			border-radius: 4px;
			padding: 15px;
			margin: 15px 0;
		}
		
		.hpsp-notice-section-title {
			display: flex;
			align-items: center;
			gap: 8px;
			margin: 0 0 12px 0;
			font-size: 14px;
			font-weight: 600;
			color: #1d2327;
		}
		
		.hpsp-notice-section-title .dashicons {
			font-size: 18px;
			width: 18px;
			height: 18px;
			color: #2271b1;
		}
		
		.hpsp-notice-section .description {
			margin: 8px 0 0 0;
			color: #646970;
			font-size: 13px;
		}
		
		.hpsp-code-container {
			position: relative;
			margin: 10px 0;
			min-height: 100px;
		}
		
		.hpsp-copy-button {
			margin-top: 8px;
			display: inline-flex;
			align-items: center;
			gap: 5px;
		}
		
		.hpsp-copy-button .dashicons {
			line-height: inherit;
			font-size: 16px;
		}
		
		#hpsp-salts-code {
			font-family: 'Courier New', Courier, monospace;
			font-size: 12px;
			padding: 12px;
			background: #f8f9fa;
			border: 1px solid #ddd;
			border-radius: 4px;
			width: 100%;
			height: 140px;
			resize: vertical;
			line-height: 1.5;
			color: #1d2327;
		}
		
		#hpsp-salts-code:focus {
			outline: none;
			border-color: #2271b1;
			box-shadow: 0 0 0 1px #2271b1;
		}
		
		.hpsp-notice-divider {
			height: 1px;
			background: #dcdcde;
			margin: 15px 0;
		}
		
		.hpsp-auto-generate-form {
			margin: 10px 0;
		}
		
		.hpsp-auto-generate-form button {
			display: inline-flex;
			align-items: center;
			gap: 6px;
		}
		
		.hpsp-auto-generate-form button .dashicons {
			line-height: inherit;
		}
		
		
		
		.hpsp-manual-steps {
			margin: 10px 0;
			padding-left: 20px;
		}
		
		.hpsp-manual-steps li {
			margin: 8px 0;
			line-height: 1.6;
		}
		
		@media screen and (max-width: 782px) {
			.hpsp-salt-notice {
				padding: 12px 15px !important;
			}
			
			.hpsp-notice-section {
				padding: 12px;
			}
			
			#hpsp-salts-code {
				font-size: 11px;
				height: 120px;
			}
			
			.hpsp-help-text {
				width: 220px;
			}
		}
		";
	}

	/**
	 * Shows admin notice if salts are missing.
	 */
	public function show_missing_salts_notice(): void {
		// Check if user dismissed the notice
		if ( get_user_meta( get_current_user_id(), 'hpsp_salt_notice_dismissed', true ) ) {
			return;
		}

		$missing_salts = HPSP_Salt_Generator::get_missing_salts();

		if ( empty( $missing_salts ) ) {
			return;
		}

		$is_writable = HPSP_Salt_Generator::is_wp_config_writable();
		$config_path = HPSP_Salt_Generator::get_wp_config_path();
		$config_code = HPSP_Salt_Generator::generate_config_code();

		?>
		<div class="notice notice-warning hpsp-notice is-dismissible hpsp-salt-notice">
			<div class="notice-title">
				<span class="dashicons dashicons-shield-alt"></span>
				<?php esc_html_e( 'Security Enhancement Available', 'highpots-spam-protection' ); ?>
			</div>

			<p>
				<?php esc_html_e( 'You can further increase the data security of this plugin (highly recommended). This requires access to the wp-config.php file on your server.', 'highpots-spam-protection' ); ?>
				<span class="hpsp-help-tooltip">
					<span class="dashicons dashicons-info hpsp-help-icon"></span>
					<span class="hpsp-help-text"><?php esc_html_e( 'Security constants ensure that tokens cannot be forged even if the database is compromised. This is similar to WordPress salt keys.', 'highpots-spam-protection' ); ?></span>
				</span>
			</p>

			<p>
				<?php esc_html_e( 'The following security constants need to be defined:', 'highpots-spam-protection' ); ?>
			</p>
			<ul>
				<?php foreach ( $missing_salts as $salt ): ?>
					<li><code><?php echo esc_html( $salt ); ?></code></li>
				<?php endforeach; ?>
			</ul>

			<?php if ( $is_writable ): ?>
				<div class="hpsp-notice-section">
					<div class="hpsp-notice-section-title">
						<span class="dashicons dashicons-admin-tools"></span>
						<?php esc_html_e( 'Option 1: Automatic Setup (Recommended)', 'highpots-spam-protection' ); ?>
					</div>

					<div class="hpsp-auto-generate-form">
						<button type="button" id="hpsp-generate-salts-button" class="button button-primary">
							<span class="dashicons dashicons-admin-network"></span>
							<?php esc_html_e( 'Generate and Add Salts Automatically', 'highpots-spam-protection' ); ?>
						</button>
					</div>

					<div id="hpsp-generate-message" style="display: none; margin-top: 10px;"></div>

					<p class="description">
						<?php
						echo wp_kses(
							sprintf(
								/* translators: %s: wp-config.php filename with code tag */
								__( 'This will automatically add the salts to %s. A backup will be created before any changes are made.', 'highpots-spam-protection' ),
								'<code>' . esc_html( basename( $config_path ) ) . '</code>'
							),
							array( 'code' => array() )
						);
						?>
					</p>
				</div>

						<div class="hpsp-notice-divider"></div>
			<?php endif; ?>

			<div class="hpsp-notice-section">
				<div class="hpsp-notice-section-title">
					<span class="dashicons dashicons-edit"></span>
					<?php echo $is_writable ? esc_html__( 'Option 2: Manual Setup', 'highpots-spam-protection' ) : esc_html__( 'Manual Setup Instructions', 'highpots-spam-protection' ); ?>
				</div>

				<p><?php esc_html_e( 'Follow these steps to manually add the security constants:', 'highpots-spam-protection' ); ?></p>
				
				<ol class="hpsp-manual-steps">
					<li>
						<?php
						echo wp_kses(
							sprintf(
								/* translators: %s: wp-config.php file path with code tag */
								__( 'Connect to your server via FTP/SFTP or file manager and locate %s', 'highpots-spam-protection' ),
								'<code>' . esc_html( $config_path ) . '</code>'
							),
							array( 'code' => array() )
						);
						?>
					</li>
					<li><?php esc_html_e( 'Create a backup of the file before making any changes', 'highpots-spam-protection' ); ?></li>
					<li><?php esc_html_e( 'Open the file in a text editor', 'highpots-spam-protection' ); ?></li>
					<li>
						<?php
						echo wp_kses(
							sprintf(
								/* translators: %s: specific line of code with code tag */
								__( 'Find the line: %s', 'highpots-spam-protection' ),
								'<code>' . esc_html( "/* That's all, stop editing! Happy publishing. */" ) . '</code>'
							),
							array( 'code' => array() )
						);
						?>
					</li>
					<li><?php esc_html_e( 'Paste the code below BEFORE that line', 'highpots-spam-protection' ); ?></li>
					<li><?php esc_html_e( 'Save the file and upload it back to the server', 'highpots-spam-protection' ); ?></li>
					<li><?php esc_html_e( 'Refresh this page to verify the constants are properly configured', 'highpots-spam-protection' ); ?></li>
				</ol>
				
				<div class="hpsp-code-container">
					<textarea 
						id="hpsp-salts-code" 
						readonly 
						onclick="this.select()"
						aria-label="<?php esc_attr_e( 'Generated salt configuration code', 'highpots-spam-protection' ); ?>"
					><?php echo esc_textarea( $config_code ); ?></textarea>
					
					<button
						type="button"
						id="hpsp-copy-salts"
						class="button button-secondary hpsp-copy-button"
						title="<?php esc_attr_e( 'Copy to clipboard', 'highpots-spam-protection' ); ?>"
					>
						<span class="dashicons dashicons-clipboard"></span>
						<?php esc_html_e( 'Copy to Clipboard', 'highpots-spam-protection' ); ?>
					</button>
				</div>
				
				<p class="description">
					<!-- <span class="dashicons dashicons-warning" style="color: #d63638;"></span> -->
					<?php
					echo wp_kses(
						sprintf(
							/* translators: %s: Important label with strong tag */
							__( '%s Make sure to paste the code in the correct location. Incorrect placement can break your site.', 'highpots-spam-protection' ),
							'<strong>' . esc_html__( 'Important:', 'highpots-spam-protection' ) . '</strong>'
						),
						array( 'strong' => array() )
					);
					?>
				</p>
			</div>
			
			<?php if ( ! $is_writable ): ?>
				<div class="hpsp-notice-section" style="background: #f0f6fc; border-color: #c3e4f7;">
					<div class="hpsp-notice-section-title">
						<!-- <span class="dashicons dashicons-info"></span> -->
						<?php esc_html_e( 'Why is automatic setup not available?', 'highpots-spam-protection' ); ?>
					</div>
					<p>
						<?php
						echo wp_kses(
							sprintf(
								/* translators: %s: wp-config.php filename with code tag */
								__( 'The file %s is not writable by the web server. This is often a security measure set by your hosting provider.', 'highpots-spam-protection' ),
								'<code>wp-config.php</code>'
							),
							array( 'code' => array() )
						);
						?>
					</p>
					<p><?php esc_html_e( 'You can either:', 'highpots-spam-protection' ); ?></p>
					<ul>
						<li><?php esc_html_e( 'Use the manual method above (recommended)', 'highpots-spam-protection' ); ?></li>
						<li><?php esc_html_e( 'Temporarily change file permissions to 644, use automatic setup, then restore to 440/400', 'highpots-spam-protection' ); ?></li>
						<li><?php esc_html_e( 'Contact your hosting provider for assistance', 'highpots-spam-protection' ); ?></li>
					</ul>
				</div>
			<?php endif; ?>
			
			<p style="margin-top: 15px;">
				<strong><?php esc_html_e( 'Need help?', 'highpots-spam-protection' ); ?></strong>
				<?php
				echo wp_kses(
					sprintf(
						/* translators: 1: documentation link, 2: support email link */
						__( 'Visit our %1$s or contact support at %2$s', 'highpots-spam-protection' ),
						'<a href="https://highpots.com/wordpress-plugins/anti-spam-protection" target="_blank" rel="noopener noreferrer">' . esc_html__( 'documentation', 'highpots-spam-protection' ) . '</a>',
						'<a href="mailto:support@highpots.com">support@highpots.com</a>'
					),
					array(
						'a' => array(
							'href' => array(),
							'target' => array(),
							'rel' => array()
						)
					)
				);
				?>
			</p>
		</div>
		<?php
	}

	/**
	 * Handle AJAX request to dismiss the salt notice.
	 */
	public function dismiss_salt_notice(): void {
		check_ajax_referer( 'hpsp_dismiss_salt_notice', 'hpsp_dismiss_salt_notice_nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( esc_html__( 'Insufficient permissions', 'highpots-spam-protection' ) );
		}
		
		update_user_meta( get_current_user_id(), 'hpsp_salt_notice_dismissed', true );
		wp_send_json_success();
	}

	/**
	 * Handles the automatic salt generation request.
	 */
	public function handle_generate_salts(): void {
		// Verify nonce
		if ( ! isset( $_POST['hpsp_generate_salts_nonce'] ) ||
		     ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['hpsp_generate_salts_nonce'] ) ), 'hpsp_generate_salts' ) ) {
			wp_die(
				esc_html__( 'Security check failed.', 'highpots-spam-protection' ),
				esc_html__( 'Security Error', 'highpots-spam-protection' ),
				[ 'response' => 403 ]
			);
		}

		// Check permissions
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die(
				esc_html__( 'You do not have permission to perform this action.', 'highpots-spam-protection' ),
				esc_html__( 'Permission Error', 'highpots-spam-protection' ),
				[ 'response' => 403 ]
			);
		}

		// Attempt to write salts
		$result = HPSP_Salt_Generator::write_salts_to_config();

		if ( is_wp_error( $result ) ) {
			// Failed - store in options as fallback
			HPSP_Salt_Generator::store_salts_in_options();

			add_settings_error(
				'hpsp_salts',
				'write_failed',
				wp_kses(
					sprintf(
						'<strong>%s</strong><br>%s<br><em>%s</em>',
						esc_html__( '⚠️ Could not write to wp-config.php', 'highpots-spam-protection' ),
						sprintf(
							/* translators: %s: error message */
							esc_html__( 'Error: %s', 'highpots-spam-protection' ),
							esc_html( $result->get_error_message() )
						),
						esc_html__( 'Salts have been stored in the database as a fallback (less secure). Please use the manual method for better security.', 'highpots-spam-protection' )
					),
					array(
						'strong' => array(),
						'br' => array(),
						'em' => array()
					)
				),
				'error'
			);
		} else {
			// Success - clear the dismissed notice flag so users can see confirmation
			delete_user_meta( get_current_user_id(), 'hpsp_salt_notice_dismissed' );

			add_settings_error(
				'hpsp_salts',
				'success',
				wp_kses(
					sprintf(
						'<strong>%s</strong><br>%s<br><em>%s</em>',
						esc_html__( '✅ Security constants successfully added!', 'highpots-spam-protection' ),
						esc_html__( 'The security constants have been written to wp-config.php.', 'highpots-spam-protection' ),
						esc_html__( 'A backup of your original file was created. Please refresh this page to dismiss this notice.', 'highpots-spam-protection' )
					),
					array(
						'strong' => array(),
						'br' => array(),
						'em' => array()
					)
				),
				'success'
			);
		}

		set_transient( 'settings_errors', get_settings_errors(), 30 );

		// Redirect back
		$redirect_url = wp_get_referer() ? wp_get_referer() : admin_url();
		$redirect_url = add_query_arg( 'settings-updated', 'true', $redirect_url );
		
		wp_safe_redirect( $redirect_url );
		exit;
	}

	/**
	 * Reset the dismissed notice (useful for testing or if user wants to see it again).
	 * This can be called from elsewhere if needed.
	 */
	public static function reset_dismissed_notice(): void {
		delete_metadata( 'user', 0, 'hpsp_salt_notice_dismissed', '', true );
	}

	/**
	 * Handles the AJAX salt generation request.
	 */
	public function handle_generate_salts_ajax(): void {

		
		// Verify nonce
		if ( ! isset( $_POST['nonce'] ) ||
		     ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'hpsp_generate_salts' ) ) {
			wp_send_json_error( [
				'message' => esc_html__( 'Security check failed.', 'highpots-spam-protection' )
			] );
		}

		// Check permissions
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( [
				'message' => esc_html__( 'You do not have permission to perform this action.', 'highpots-spam-protection' )
			] );
		}

		// Attempt to write salts
		$result = HPSP_Salt_Generator::write_salts_to_config();

		if ( is_wp_error( $result ) ) {
			// Failed - store in options as fallback
			HPSP_Salt_Generator::store_salts_in_options();

			wp_send_json_error( [
				'message' => wp_kses(
					sprintf(
						'<strong>%s</strong><br>%s<br><em>%s</em>',
						esc_html__( '⚠️ Could not write to wp-config.php', 'highpots-spam-protection' ),
						sprintf(
							/* translators: %s: error message */
							esc_html__( 'Error: %s', 'highpots-spam-protection' ),
							esc_html( $result->get_error_message() )
						),
						esc_html__( 'Salts have been stored in the database as a fallback (less secure). Please use the manual method for better security.', 'highpots-spam-protection' )
					),
					array(
						'strong' => array(),
						'br' => array(),
						'em' => array()
					)
				)
			] );
		} else {
			// Success
			wp_send_json_success( [
				'message' => wp_kses(
					sprintf(
						'<strong>%s</strong><br>%s<br><em>%s</em>',
						esc_html__( '✅ Security constants successfully added!', 'highpots-spam-protection' ),
						esc_html__( 'The security constants have been written to wp-config.php.', 'highpots-spam-protection' ),
						esc_html__( 'A backup of your original file was created. Please refresh this page to dismiss this notice.', 'highpots-spam-protection' )
					),
					array(
						'strong' => array(),
						'br' => array(),
						'em' => array()
					)
				)
			] );
		}
	}
}