<?php

namespace Highpots\SpamProtection\Admin;

use Highpots\SpamProtection\Enums\HPSP_Log_Type;
use Highpots\SpamProtection\Helpers\HPSP_Constants;
use Highpots\SpamProtection\Helpers\HPSP_Utils;
use WP_List_Table;

if (!class_exists('WP_List_Table')) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class HPSP_Log_List_Table extends WP_List_Table {

    private $table_name;
    private $data;
    public $total_items;
    public $items_per_page = 20;
    private $columns;
    public $hidden;
    private $sortable;
    
    private const DEFAULT_ITEMS_PER_PAGE = 20;

    public function __construct() {
        global $wpdb;

        parent::__construct(array(
            'singular' => 'spam_log',
            'plural'   => 'spam_logs',
            'ajax'     => true
        ));

        $this->table_name = HPSP_Constants::get_logs_table_name();
        $this->columns    = $this->get_columns();
        $this->hidden     = array('id', 'location');
        $this->sortable   = $this->get_sortable_columns();
    }

	
	/**
	 * Retrieve spam log data from the database
	 *
	 * @param int    $per_page       Number of items per page
	 * @param int    $page_number    Current page number
	 * @param array  $date_params    Date filter parameters: ['time_range' => string, 'from_date' => string, 'to_date' => string]
	 * @param string $reason         Optional reason filter
	 *
	 * @return array
	 */
    public function get_data($per_page = 20, $page_number = 1, $date_params = [], $reason = '') {
        global $wpdb;

        $prepare_values = [];

        // Get date filter configuration
        $date_filter = $this->build_date_filter($date_params);
        $filter_type = $date_filter['type'];

        // Get sorting parameters (validated via whitelist)
        $orderby = $this->get_orderby();
        $order   = $this->get_order();

        // Build ORDER BY clause
        $order_clause = !empty($orderby) ? "ORDER BY {$orderby} {$order}" : 'ORDER BY timestamp DESC';

        // Calculate pagination
        $offset = ($page_number - 1) * $per_page;

        // Build complete SQL query based on filter type and reason
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe, defined by plugin constant
        $base_select = "SELECT id, timestamp, form_id, reason, ip_address, location, note FROM {$this->table_name}";

        if (!empty($reason)) {
            // With reason filter
            switch ($filter_type) {
                case 'today':
                    $sql = "$base_select WHERE DATE(timestamp) = CURDATE() AND reason = %s $order_clause LIMIT %d OFFSET %d";
                    $prepare_values = [$reason, $per_page, $offset];
                    break;
                case '30days':
                    $sql = "$base_select WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 30 DAY) AND reason = %s $order_clause LIMIT %d OFFSET %d";
                    $prepare_values = [$reason, $per_page, $offset];
                    break;
                case 'custom':
                    $sql = "$base_select WHERE timestamp >= %s AND timestamp <= %s AND reason = %s $order_clause LIMIT %d OFFSET %d";
                    $prepare_values = [$date_filter['from_datetime'], $date_filter['to_datetime'], $reason, $per_page, $offset];
                    break;
                case '7days':
                default:
                    $sql = "$base_select WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 7 DAY) AND reason = %s $order_clause LIMIT %d OFFSET %d";
                    $prepare_values = [$reason, $per_page, $offset];
                    break;
            }
        } else {
            // Without reason filter
            switch ($filter_type) {
                case 'today':
                    $sql = "$base_select WHERE DATE(timestamp) = CURDATE() $order_clause LIMIT %d OFFSET %d";
                    $prepare_values = [$per_page, $offset];
                    break;
                case '30days':
                    $sql = "$base_select WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 30 DAY) $order_clause LIMIT %d OFFSET %d";
                    $prepare_values = [$per_page, $offset];
                    break;
                case 'custom':
                    $sql = "$base_select WHERE timestamp >= %s AND timestamp <= %s $order_clause LIMIT %d OFFSET %d";
                    $prepare_values = [$date_filter['from_datetime'], $date_filter['to_datetime'], $per_page, $offset];
                    break;
                case '7days':
                default:
                    $sql = "$base_select WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 7 DAY) $order_clause LIMIT %d OFFSET %d";
                    $prepare_values = [$per_page, $offset];
                    break;
            }
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Direct query needed, $order_clause uses validated whitelist values from get_orderby() and get_order(), $sql contains safe interpolated variables
        return $wpdb->get_results(
            $wpdb->prepare($sql, ...$prepare_values), // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- $sql contains validated safe values
            'ARRAY_A'
        );
    }
	/**
	 * Parse and validate date filter parameters
	 *
	 * @param array $date_params Date filter parameters: ['time_range' => string, 'from_date' => string, 'to_date' => string]
	 *
	 * @return array ['type' => string, 'from_datetime' => string, 'to_datetime' => string] Filter type and formatted date values
	 * @throws \Exception If date validation fails
	 */
	private function build_date_filter($date_params) {
		if (empty($date_params) || empty($date_params['time_range'])) {
			// Default: last 7 days
			return [
				'type' => '7days',
				'from_datetime' => '',
				'to_datetime' => ''
			];
		}

		$time_range = $date_params['time_range'];

		// Handle custom date range
		if ($time_range === 'custom') {
			$from_date = $date_params['from_date'] ?? '';
			$to_date = $date_params['to_date'] ?? '';

			if (empty($from_date) || empty($to_date)) {
				throw new \Exception(esc_html__('From and To dates are required for a custom range.', 'highpots-spam-protection'));
			}

			if (!$this->is_valid_date($from_date) || !$this->is_valid_date($to_date)) {
				throw new \Exception(esc_html__('Invalid date format. Please use YYYY-MM-DD.', 'highpots-spam-protection'));
			}

			$from = new \DateTime($from_date);
			$to = new \DateTime($to_date);

			if ($from > $to) {
				throw new \Exception(esc_html__('From date cannot be later than To date.', 'highpots-spam-protection'));
			}

			return [
				'type' => 'custom',
				'from_datetime' => $from->format('Y-m-d 00:00:00'),
				'to_datetime' => $to->format('Y-m-d 23:59:59')
			];
		}

		// Handle predefined time ranges
		$valid_ranges = ['today', '7days', '30days'];
		$filter_type = in_array($time_range, $valid_ranges, true) ? $time_range : '7days';

		return [
			'type' => $filter_type,
			'from_datetime' => '',
			'to_datetime' => ''
		];
	}

	/**
	 * Validate date format
	 *
	 * @param string $date Date string
	 *
	 * @return bool True if valid YYYY-MM-DD format
	 */
	private function is_valid_date($date) {
		$d = \DateTime::createFromFormat('Y-m-d', $date);
		return $d && $d->format('Y-m-d') === $date;
	}

	/**
	 * Get the current orderby column
	 *
	 * @return string
	 */
	private function get_orderby() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Table sorting via GET parameter, no state change
		if (!isset($_GET['orderby'])) {
			return '';
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Table sorting via GET parameter, no state change
		$orderby = sanitize_text_field(wp_unslash($_GET['orderby']));

		// Whitelist allowed columns
		$sortable = $this->get_sortable_columns();
		$allowed_columns = array_keys($sortable);

		if (in_array($orderby, $allowed_columns, true)) {
			return $orderby;
		}

		return '';
	}

	/**
	 * Get the current sort order
	 *
	 * @return string
	 */
	private function get_order() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Table sorting via GET parameter, no state change
		if (!isset($_GET['order'])) {
			return 'DESC';
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Table sorting via GET parameter, no state change
		$order = strtoupper(sanitize_text_field(wp_unslash($_GET['order'])));

		return ($order === 'ASC') ? 'ASC' : 'DESC';
	}
    /**
     * Returns the count of records in the database
     *
     * @param array  $date_params Date filter parameters: ['time_range' => string, 'from_date' => string, 'to_date' => string]
     * @param string $reason      Optional reason filter
     *
     * @return int
     */
   public function record_count($date_params = [], $reason = '') {
        global $wpdb;

        // Get date filter configuration
        $date_filter = $this->build_date_filter($date_params);
        $filter_type = $date_filter['type'];

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe, defined by plugin constant
        $base_select = "SELECT COUNT(*) FROM {$this->table_name}";

        $where_clauses = [];
        $prepare_values = [];

        // Build date filter
        switch ($filter_type) {
            case 'today':
                $where_clauses[] = 'DATE(timestamp) = CURDATE()';
                break;
            case '30days':
                $where_clauses[] = 'timestamp >= DATE_SUB(NOW(), INTERVAL 30 DAY)';
                break;
            case 'custom':
                $where_clauses[] = 'timestamp >= %s AND timestamp <= %s';
                $prepare_values[] = $date_filter['from_datetime'];
                $prepare_values[] = $date_filter['to_datetime'];
                break;
            case '7days':
            default:
                $where_clauses[] = 'timestamp >= DATE_SUB(NOW(), INTERVAL 7 DAY)';
                break;
        }

        // Add reason filter if provided
        if (!empty($reason)) {
            $where_clauses[] = 'reason = %s';
            $prepare_values[] = $reason;
        }

        // Build complete SQL
        $sql = $base_select;
        if (!empty($where_clauses)) {
            $sql .= ' WHERE ' . implode(' AND ', $where_clauses);
        }

        // Execute query - Custom spam log table requires direct queries; caching not suitable for frequently changing log data
        if (!empty($prepare_values)) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- SQL safely constructed with placeholders, all values properly escaped via wpdb->prepare()
            return (int) $wpdb->get_var($wpdb->prepare($sql, ...$prepare_values));
        } else {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- No user input, SQL uses MySQL functions only
            return (int) $wpdb->get_var($sql);
        }
    }

    /**
     * Text displayed when no data is available
     */
    public function no_items() {
        esc_html_e('No spam logs found', 'highpots-spam-protection');
    }

    /**
     * Column: ID
     */
    public function column_id($item) {
        return '<em>' . esc_html($item['id']) . '</em>';
    }

    /**
     * Column: Timestamp
     */
    public function column_timestamp($item) {
        return '<em>' . esc_html($item['timestamp']) . '</em>';
    }

    /**
     * Column: Form ID
     */
    public function column_form_id($item) {
        return '<em>' . esc_html($item['form_id']) . '</em>';
    }

    /**
     * Column: Reason
     */
    public function column_reason($item) {
        $reason_label = HPSP_Log_Type::tryLabel($item['reason']);
        $class = ($item['reason'] === 'valid') ? 'hpsp-valid' : 'hpsp-invalid';
        
        return sprintf(
            '<em class="%s">%s</em>',
            esc_attr($class),
            esc_html($reason_label)
        );
    }

    /**
     * Column: IP Hash
     */
    public function column_ip_address($item) {
        $ip_address = $item['ip_address'];

        // Check if IP masking is enabled
        if (HPSP_Constants::get_mask_ip_in_logs_option()) {
            $ip_address = HPSP_Utils::mask_ip_address($ip_address);
        }

        return '<em>' . esc_html($ip_address) . '</em>';
    }

    /**
     * Column: Location
     */
    public function column_location($item) {
        return '<em>' . esc_html($item['location']) . '</em>';
    }

    /**
     * Column: Note
     */
    public function column_note($item) {
        return '<em>' . esc_html($item['note']) . '</em>';
    }

    /**
     * Get all available columns
     *
     * @return array
     */
    public function get_columns() {
        return array(
            'id'        => esc_html__('ID',  'highpots-spam-protection'),
            'timestamp' => esc_html__('Timestamp',  'highpots-spam-protection'),
            'form_id'   => esc_html__('Form ID',  'highpots-spam-protection'),
            'reason'    => esc_html__('Reason',  'highpots-spam-protection'),
            'ip_address'   => esc_html__('IP address',  'highpots-spam-protection'),
            'location'  => esc_html__('Location',  'highpots-spam-protection'),
            'note'      => esc_html__('Note',  'highpots-spam-protection'),
        );
    }

    /**
     * Get columns excluding specified ones
     *
     * @param array $exclude_columns Column keys to exclude
     *
     * @return array
     */
    public function get_columns_filtered($exclude_columns = array()) {
        $all_columns = $this->get_columns();
        
        if (empty($exclude_columns)) {
            return $all_columns;
        }
        
        return array_diff_key($all_columns, array_flip($exclude_columns));
    }

    /**
     * Handles data query, filter, sorting and pagination
     *
     * @param array  $date_params Date filter parameters: ['time_range' => string, 'from_date' => string, 'to_date' => string]
     * @param string $reason      Optional reason filter
     */
    public function prepare_items($date_params = [], $reason = '') {
        $current_page = $this->get_pagenum();
        $this->items_per_page = $this->get_items_per_page('spam_logs_per_page', self::DEFAULT_ITEMS_PER_PAGE);
        $this->total_items = $this->record_count($date_params, $reason);

        $this->set_pagination_args(array(
            'total_items' => $this->total_items,
            'per_page'    => $this->items_per_page
        ));

        $this->items = $this->get_data($this->items_per_page, $current_page, $date_params, $reason);
        $this->_column_headers = array($this->columns, $this->hidden, $this->sortable);
    }

    /**
     * Define which columns are sortable
     *
     * @return array
     */
    public function get_sortable_columns() {
        return array(
            'id'        => array('id', true),
            'timestamp' => array('timestamp', true)
        );
    }

    /**
     * Get summary data grouped by date (only enabled columns)
     *
     * @param array  $date_params Date filter parameters: ['time_range' => string, 'from_date' => string, 'to_date' => string]
     * @param string $reason      Optional reason filter
     *
     * @return array
     */
   public function get_summary_data($date_params = [], $reason = '') {
        global $wpdb;

        // Base columns (always included)
        $select_columns = [
            "DATE(timestamp) as date",
            "SUM(reason='valid') as valid",
            "SUM(reason='invalid_token') as invalid_token",
            "SUM(reason='honeypot') as honeypot",
            "SUM(reason='fast_submission') as fast_submission",
            "SUM(reason='slow_submission') as slow_submission",
            "SUM(reason='form_id_missing') as form_id_missing"
        ];

        // Add rate limiting column if enabled
        if (HPSP_Constants::get_enable_rate_limiting_option()) {
            $select_columns[] = "SUM(reason='too_many_submissions') as too_many_submissions";
        }

        // Add blocked writing systems column if any are configured
        $blocked_systems = HPSP_Constants::get_blocked_writing_systems_option();
        if (!empty($blocked_systems)) {
            $select_columns[] = "SUM(reason='blocked_writing_systems') as blocked_writing_systems";
        }

        // Add user agent validation column if enabled
        if (HPSP_Constants::get_enable_user_agent_validation_option()) {
            $select_columns[] = "SUM(reason='suspicious_user_agent') as suspicious_user_agent";
        }

        // Add referrer validation column if enabled
        if (HPSP_Constants::get_enable_referrer_validation_option()) {
            $select_columns[] = "SUM(reason='suspicious_referer') as invalid_referer";
        }

        // Always include total
        $select_columns[] = "COUNT(*) as total";

        $prepare_values = [];

        // Get date filter configuration
        $date_filter = $this->build_date_filter($date_params);
        $filter_type = $date_filter['type'];

        // Build SELECT clause (columns are hardcoded, safe to implode)
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name and column list are safe constants
        $select_clause = "SELECT " . implode(", ", $select_columns) . " FROM {$this->table_name}";

        // Build complete SQL query based on filter type
        if (!empty($reason)) {
            // With reason filter
            switch ($filter_type) {
                case 'today':
                    $sql = "$select_clause WHERE DATE(timestamp) = CURDATE() AND reason = %s GROUP BY DATE(timestamp) ORDER BY date DESC";
                    $prepare_values = [$reason];
                    break;
                case '30days':
                    $sql = "$select_clause WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 30 DAY) AND reason = %s GROUP BY DATE(timestamp) ORDER BY date DESC";
                    $prepare_values = [$reason];
                    break;
                case 'custom':
                    $sql = "$select_clause WHERE timestamp >= %s AND timestamp <= %s AND reason = %s GROUP BY DATE(timestamp) ORDER BY date DESC";
                    $prepare_values = [$date_filter['from_datetime'], $date_filter['to_datetime'], $reason];
                    break;
                case '7days':
                default:
                    $sql = "$select_clause WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 7 DAY) AND reason = %s GROUP BY DATE(timestamp) ORDER BY date DESC";
                    $prepare_values = [$reason];
                    break;
            }
        } else {
            // Without reason filter
            switch ($filter_type) {
                case 'today':
                    $sql = "$select_clause WHERE DATE(timestamp) = CURDATE() GROUP BY DATE(timestamp) ORDER BY date DESC";
                    break;
                case '30days':
                    $sql = "$select_clause WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 30 DAY) GROUP BY DATE(timestamp) ORDER BY date DESC";
                    break;
                case 'custom':
                    $sql = "$select_clause WHERE timestamp >= %s AND timestamp <= %s GROUP BY DATE(timestamp) ORDER BY date DESC";
                    $prepare_values = [$date_filter['from_datetime'], $date_filter['to_datetime']];
                    break;
                case '7days':
                default:
                    $sql = "$select_clause WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 7 DAY) GROUP BY DATE(timestamp) ORDER BY date DESC";
                    break;
            }
        }

        // Execute query with or without prepared statement
        if (!empty($prepare_values)) {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- SQL safely constructed with placeholders, all values properly escaped via wpdb->prepare()
            return $wpdb->get_results($wpdb->prepare($sql, ...$prepare_values), 'ARRAY_A');
        } else {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- No user input, SQL uses MySQL functions only
            return $wpdb->get_results($sql, 'ARRAY_A');
        }
    }
    /**
     * Get summary table column headers (filtered based on enabled settings)
     *
     * @return array
     */
    public function get_group_columns() {
        $columns = array(
            'date'                      => esc_html__('Date',  'highpots-spam-protection'),
            'valid'                     => esc_html__('Valid',  'highpots-spam-protection'),
            'invalid_token'             => esc_html__('Invalid Token',  'highpots-spam-protection'),
            'honeypot'                  => esc_html__('Honeypot',  'highpots-spam-protection'),
            'fast_submission'           => esc_html__('Fast Submission',  'highpots-spam-protection'),
            'slow_submission'           => esc_html__('Form Expired',  'highpots-spam-protection'),
            'form_id_missing'           => esc_html__('Form ID Missing',  'highpots-spam-protection'),
        );

        // Add rate limiting column if enabled
        if (HPSP_Constants::get_enable_rate_limiting_option()) {
            $columns['too_many_submissions'] = esc_html__('Rate Limited', 'highpots-spam-protection');
        }

        // Add blocked writing systems column if any are configured
        $blocked_systems = HPSP_Constants::get_blocked_writing_systems_option();
        if (!empty($blocked_systems)) {
            $columns['blocked_writing_systems'] = esc_html__('Blocked Writing', 'highpots-spam-protection');
        }

        // Add user agent validation column if enabled
        if (HPSP_Constants::get_enable_user_agent_validation_option()) {
            $columns['suspicious_user_agent'] = esc_html__('Suspicious user agent', 'highpots-spam-protection');
        }

        // Add referrer validation column if enabled
        if (HPSP_Constants::get_enable_referrer_validation_option()) {
            $columns['invalid_referer'] = esc_html__('Invalid referrer', 'highpots-spam-protection');
        }

        $columns['total'] = esc_html__('Total', 'highpots-spam-protection');

        return $columns;
    }

    /**
     * Display summary table
     *
     * @param array  $date_params Date filter parameters: ['time_range' => string, 'from_date' => string, 'to_date' => string]
     * @param string $reason      Optional reason filter
     */
    public function display_summary_table($date_params = [], $reason = '') {
        $data = $this->get_summary_data($date_params, $reason);

        if (empty($data)) {
            echo '<p>' . esc_html__('No data found for the selected period.',  'highpots-spam-protection') . '</p>';
            return;
        }

        // Get enabled columns
        $columns = $this->get_group_columns();
        $column_keys = array_keys($columns);

        // Calculate totals (only for enabled columns)
        $totals = array();
        foreach ($column_keys as $key) {
            $totals[$key] = 0;
        }

        foreach ($data as $row) {
            foreach ($column_keys as $key) {
                if (isset($row[$key])) {
                    $totals[$key] += intval($row[$key]);
                }
            }
        }

        ?>
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <?php foreach ($columns as $header_title): ?>
                        <th><?php echo esc_html($header_title); ?></th>
                    <?php endforeach; ?>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($data as $row): ?>
                    <tr>
                        <?php foreach ($column_keys as $key): ?>
                            <?php if ($key === 'date'): ?>
                                <td><?php echo esc_html($row[$key]); ?></td>
                            <?php elseif ($key === 'total'): ?>
                                <td><strong><?php echo intval($row[$key]); ?></strong></td>
                            <?php elseif ($key === 'valid'): ?>
                                <td class="hpsp-valid"><?php echo intval($row[$key]); ?></td>
                            <?php else: ?>
                                <td class="hpsp-invalid"><?php echo isset($row[$key]) ? intval($row[$key]) : 0; ?></td>
                            <?php endif; ?>
                        <?php endforeach; ?>
                    </tr>
                <?php endforeach; ?>
            </tbody>
            <tfoot>
                <tr>
                    <?php foreach ($column_keys as $key): ?>
                        <?php if ($key === 'date'): ?>
                            <th><?php esc_html_e('Total', 'highpots-spam-protection'); ?></th>
                        <?php elseif ($key === 'total'): ?>
                            <th><strong><?php echo intval($totals[$key]); ?></strong></th>
                        <?php elseif ($key === 'valid'): ?>
                            <th class="hpsp-valid"><?php echo intval($totals[$key]); ?></th>
                        <?php else: ?>
                            <th class="hpsp-invalid"><?php echo intval($totals[$key]); ?></th>
                        <?php endif; ?>
                    <?php endforeach; ?>
                </tr>
            </tfoot>
        </table>
        <?php
    }
}