<?php

namespace Highpots\SpamProtection\Admin;

use DateTime;
use Exception;


/**
 * Class Spam_Log_Controller
 * Handles the display and filtering of spam logs.
 */
class HPSP_Log_Controller
{


    private static ?self $instance = null;

    private function __construct()
    {
        $this->init_ajax_handlers();
    }

    /**
     * Get singleton instance
     */
    public static function get_instance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Initialize AJAX handlers
     */
    private function init_ajax_handlers(): void
    {
        add_action('wp_ajax_hpsp_get_spam_log_data', [$this, 'hpsp_get_spam_log_data_ajax']);
        add_action('wp_ajax_hpsp_export_spam_log_csv', [$this, 'hpsp_export_spam_log_csv_ajax']);
    }

    /**
     * Renders the logs page.
     */
    public function render_logs_page()
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'highpots-spam-protection'));
        }

        // Verify nonce if form was submitted
        $nonce_verified = isset($_GET['hpsp_logs_nonce'])
            && wp_verify_nonce(
                sanitize_text_field(wp_unslash($_GET['hpsp_logs_nonce'])),
                'hpsp_filter_logs'
            );

        // Only use GET parameters if nonce is verified or not present (initial page load)
        $time_range = '7days';
        $from_date  = '';
        $to_date    = '';
        $detailed   = false;
        $reason     = '';

        if ($nonce_verified || ! isset($_GET['hpsp_logs_nonce'])) {
            $time_range = isset($_GET['time_range']) ? sanitize_text_field(wp_unslash($_GET['time_range'])) : '7days';
            $from_date  = isset($_GET['from_date']) ? sanitize_text_field(wp_unslash($_GET['from_date'])) : '';
            $to_date    = isset($_GET['to_date']) ? sanitize_text_field(wp_unslash($_GET['to_date'])) : '';
            $detailed   = isset($_GET['detailed']);
            $reason     = isset($_GET['reason']) ? sanitize_text_field(wp_unslash($_GET['reason'])) : '';
        }

?>
        <div>
            <div>&nbsp;</div>
            <form id="spam-log-filter-form" class="hpsp-container" method="get">
                <?php wp_nonce_field('hpsp_filter_logs', 'hpsp_logs_nonce'); ?>
                <?php
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- GET parameter for page identification
                ?>
                <input type="hidden" name="page" value="<?php echo isset($_GET['page']) ? esc_attr(sanitize_text_field(wp_unslash($_GET['page']))) : ''; ?>">

                <div class="hpsp-filters-row">
                    <div class="hpsp-filter-group">
                        <label for="time_range"><?php esc_html_e('Time Range:', 'highpots-spam-protection'); ?></label>
                        <select name="time_range" id="time_range">
                            <option value="today" <?php selected($time_range, 'today'); ?>><?php esc_html_e('Today', 'highpots-spam-protection'); ?></option>
                            <option value="7days" <?php selected($time_range, '7days'); ?>><?php esc_html_e('Last 7 Days', 'highpots-spam-protection'); ?></option>
                            <option value="30days" <?php selected($time_range, '30days'); ?>><?php esc_html_e('Last 30 Days', 'highpots-spam-protection'); ?></option>
                            <option value="custom" <?php selected($time_range, 'custom'); ?>><?php esc_html_e('Custom Range', 'highpots-spam-protection'); ?></option>
                        </select>
                    </div>

                    <div id="custom_date_range" class="hpsp-filter-group" style="display: none;">
                        <label for="from_date" class="required"><?php esc_html_e('From:', 'highpots-spam-protection'); ?></label>
                        <input type="date" name="from_date" id="from_date" value="<?php echo esc_attr($from_date); ?>">

                        <label for="to_date" class="required"><?php esc_html_e('To:', 'highpots-spam-protection'); ?></label>
                        <input type="date" name="to_date" id="to_date" value="<?php echo esc_attr($to_date); ?>">
                    </div>

                    <div class="hpsp-filter-group">
                        <label for="detailed"><?php esc_html_e('Detailed View:', 'highpots-spam-protection'); ?></label>
                        <input type="checkbox" name="detailed" id="detailed" <?php checked($detailed); ?>>
                    </div>

                    <div class="hpsp-filter-group" id="log-type-filter" style="display: none;">
                        <label for="reason"><?php esc_html_e('Log Type:', 'highpots-spam-protection'); ?></label>
                        <select name="reason" id="reason">
                            <option value="" <?php selected($reason, ''); ?>><?php esc_html_e('All Types', 'highpots-spam-protection'); ?></option>
                            <option value="valid" <?php selected($reason, 'valid'); ?>><?php esc_html_e('Valid', 'highpots-spam-protection'); ?></option>
                            <option value="honeypot" <?php selected($reason, 'honeypot'); ?>><?php esc_html_e('Honeypot', 'highpots-spam-protection'); ?></option>
                            <option value="fast_submission" <?php selected($reason, 'fast_submission'); ?>><?php esc_html_e('Too Fast', 'highpots-spam-protection'); ?></option>
                            <option value="slow_submission" <?php selected($reason, 'slow_submission'); ?>><?php esc_html_e('Too Slow', 'highpots-spam-protection'); ?></option>
                            <option value="form_id_missing" <?php selected($reason, 'form_id_missing'); ?>><?php esc_html_e('Form ID Missing', 'highpots-spam-protection'); ?></option>
                            <option value="too_many_submissions" <?php selected($reason, 'too_many_submissions'); ?>><?php esc_html_e('Rate Limited', 'highpots-spam-protection'); ?></option>
                            <option value="invalid_token" <?php selected($reason, 'invalid_token'); ?>><?php esc_html_e('Invalid Token', 'highpots-spam-protection'); ?></option>
                            <option value="blocked_writing_systems" <?php selected($reason, 'blocked_writing_systems'); ?>><?php esc_html_e('Blocked Content', 'highpots-spam-protection'); ?></option>
                            <option value="suspicious_user_agent" <?php selected($reason, 'suspicious_user_agent'); ?>><?php esc_html_e('Suspicious User Agent', 'highpots-spam-protection'); ?></option>
                            <option value="suspicious_referer" <?php selected($reason, 'suspicious_referer'); ?>><?php esc_html_e('Invalid Referrer', 'highpots-spam-protection'); ?></option>
                        </select>
                    </div>

                    <div class="hpsp-filter-group">
                        <input type="submit" value="<?php esc_attr_e('Apply Filters', 'highpots-spam-protection'); ?>" class="button button-primary" id="hpsp-apply-filters">
                    </div>


                    <div class="hpsp-filter-divider"></div>
                    <div class="hpsp-filter-group">
                        <button type="button"
                            id="hpsp-export2csv-btn"
                            class="button button-secondary hpsp-export-btn"
                            title="<?php esc_attr_e('Export to CSV', 'highpots-spam-protection'); ?>">
                            <span>CSV</span>
                            <span class="dashicons dashicons-download"></span>
                        </button>
                    </div>
                </div>
            </form>

            <div id="hpsp-loading" style="display: none;">
                <div class="hpsp-spinner-container">
                    <div class="hpsp-spinner"></div>
                    <p><?php esc_html_e('Loading...', 'highpots-spam-protection'); ?></p>
                </div>
            </div>

            <div id="spam-log-table-container">
                <?php
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Output escaped by HPSP_Log_List_Table class methods via esc_html(), esc_attr(), and intval()
                echo $this->display_spam_log_table($time_range, $from_date, $to_date, $detailed, $reason);
                ?>
            </div>
        </div>
<?php
    }


    /**
     * Displays the spam log table.
     *
     * @param string $time_range
     * @param string $from_date
     * @param string $to_date
     * @param bool $detailed
     * @param string $reason
     */
    public function display_spam_log_table($time_range, $from_date, $to_date, bool $detailed  = false, string $reason = '')
    {
        // Prepare date parameters array
        $date_params = [
            'time_range' => $time_range,
            'from_date' => $from_date,
            'to_date' => $to_date
        ];

        $myListTable = new HPSP_Log_List_Table();
        ob_start();

        try {
            if ($detailed) {
                // Detailed view
                $myListTable->prepare_items($date_params, $reason);
                $myListTable->display();
            } else {
                // Summary view
                echo '<h3>' . esc_html__('Summary by Date',  'highpots-spam-protection') . '</h3>';
                $myListTable->display_summary_table($date_params, $reason);
            }
        } catch (Exception $e) {
            echo '<p class="error">' . esc_html($e->getMessage()) . '</p>';
        }

        return ob_get_clean();
    }



    /**
     * Verify AJAX request security.
     * 
     * Checks user authentication, nonce validity, and user capabilities.
     * Terminates with JSON error response if any check fails.
     *
     * @return void Exits on failure with wp_send_json_error()
     */
    private function verify_ajax_request($type)
    {
        if ($type === 'filter_logs') {
            $nonce_action = 'hpsp_filter_logs';
            $nonce_field = 'hpsp_filter_logs_nonce';
        } elseif ($type === 'export_csv') {
            $nonce_action = 'hpsp_export_csv';
            $nonce_field = 'hpsp_export_csv_nonce';
        } else {
            wp_send_json_error(array(
                'message' => esc_html__('Invalid request type.', 'highpots-spam-protection')
            ), 400);
        }
        // Verify nonce (includes logged-in check)
        $nonce_check = check_ajax_referer($nonce_action, $nonce_field, false);

        if (false === $nonce_check) {
            wp_send_json_error(array(
                'message' => esc_html__('Security verification failed. Please refresh the page.', 'highpots-spam-protection')
            ), 403);
        }

        // Check user capabilities
        if (! current_user_can('manage_options')) {
            wp_send_json_error(array(
                'message' => esc_html__('You do not have permission to perform this action.', 'highpots-spam-protection')
            ), 403);
        }
    }


    public function hpsp_get_spam_log_data_ajax()
    {

        // Check access grants
        $this->verify_ajax_request('filter_logs');

        // Retrieve filter parameters from $_POST
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax_request()
        $time_range = isset($_POST['time_range']) ? sanitize_text_field(wp_unslash($_POST['time_range'])) : '7days';
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax_request()
        $from_date  = isset($_POST['from_date']) ? sanitize_text_field(wp_unslash($_POST['from_date'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax_request()
        $to_date    = isset($_POST['to_date']) ? sanitize_text_field(wp_unslash($_POST['to_date'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax_request()
        $detailed   = isset($_POST['detailed']);
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax_request()
        $reason     = isset($_POST['reason']) ? sanitize_text_field(wp_unslash($_POST['reason'])) : '';

        // Get sorting parameters from POST (sent via AJAX)
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax_request()
        $_GET['orderby'] = isset($_POST['orderby']) ? sanitize_text_field(wp_unslash($_POST['orderby'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax_request()
        $_GET['order']   = isset($_POST['order']) ? sanitize_text_field(wp_unslash($_POST['order'])) : '';

        try {
            $table_html = $this->display_spam_log_table($time_range, $from_date, $to_date, $detailed, $reason);

            wp_send_json_success([
                'html' => $table_html,
            ]);
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }



    public function hpsp_export_spam_log_csv_ajax()
    {


        // Check access grants
        $this->verify_ajax_request('export_csv');

        // Retrieve filter parameters from $_POST
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax_request()
        $time_range = isset($_POST['time_range']) ? sanitize_text_field(wp_unslash($_POST['time_range'])) : '7days';
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax_request()
        $from_date  = isset($_POST['from_date']) ? sanitize_text_field(wp_unslash($_POST['from_date'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax_request()
        $to_date    = isset($_POST['to_date']) ? sanitize_text_field(wp_unslash($_POST['to_date'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax_request()
        $reason     = isset($_POST['reason']) ? sanitize_text_field(wp_unslash($_POST['reason'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax_request()
        $check_only = isset($_POST['check_only']) && sanitize_text_field(wp_unslash($_POST['check_only']));

        // Accept both '1' and 'on' as truthy values
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in verify_ajax_request()
        $detailed = isset($_POST['detailed']) &&
            in_array(sanitize_text_field(wp_unslash($_POST['detailed'])), array('1', 'on', 'true'), true);
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        // Prepare date parameters array
        $date_params = [
            'time_range' => $time_range,
            'from_date' => $from_date,
            'to_date' => $to_date
        ];

        $hpsp_log_list_table = new HPSP_Log_List_Table();

        try {
            if ($detailed) {
                // Export detailed individual logs
                $data = $hpsp_log_list_table->get_data(PHP_INT_MAX, 1, $date_params, $reason);
                // Get all columns and filter out hidden ones
                $all_columns = $hpsp_log_list_table->get_columns();
                $hidden_columns = $hpsp_log_list_table->hidden;
                $column_names = array_diff_key($all_columns, array_flip($hidden_columns));
                $filename_suffix = 'detailed';
            } else {
                // Export summary data
                $data = array_values($hpsp_log_list_table->get_summary_data($date_params, $reason));
                $column_names = $hpsp_log_list_table->get_group_columns();
                $filename_suffix = 'summary';
            }
        } catch (Exception $e) {
            wp_send_json_error([
                'message' => esc_html($e->getMessage())
            ]);
        }

        if (empty($data)) {
            wp_send_json_error([
                'message' => esc_html__('No data to export. Please adjust your filters and try again.',  'highpots-spam-protection')
            ]);
        }

        // If this is just a check (pre-validation), return success
        if ($check_only) {
            wp_send_json_success([
                'message' => esc_html__('Data is available for export.',  'highpots-spam-protection'),
                'count' => count($data)
            ]);
        }

        // Set headers for CSV download
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="spam_log_' . gmdate('YmdHis') . '_' . $filename_suffix . '.csv"');
        header('Pragma: no-cache');
        header('Expires: 0');

        // Generate and output CSV data directly
        $this->generate_spam_log_csv($data, $column_names);

        // Exit cleanly without additional output
        exit;
    }

    /**
     * Generates and outputs the CSV data from the spam log results.
     *
     * @param array $data The spam log data.
     * @param array $column_names The column names.
     * @return void
     */
    private function generate_spam_log_csv($data, $column_names): void
    {
        if (empty($data)) {
            return;
        }

        // Open output stream
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen -- Writing to php://output stream for CSV download
        $output = fopen('php://output', 'w');

        // Add BOM for UTF-8 (helps Excel display special characters correctly)
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fprintf -- Writing to output stream
        fprintf($output, chr(0xEF) . chr(0xBB) . chr(0xBF));

        // CSV Header - use the column labels
        $csv_header = array_values($column_names);
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fputcsv -- Writing CSV to output stream
        fputcsv($output, $csv_header);

        // CSV Rows
        foreach ($data as $row) {
            $export_row = array();
            foreach ($column_names as $key => $value) {
                // Handle cases where key might not exist in row
                $export_row[] = isset($row[$key]) ? $row[$key] : '';
            }
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fputcsv -- Writing CSV to output stream
            fputcsv($output, $export_row);
        }

        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing output stream
        fclose($output);
    }
}
