<?php

namespace Highpots\SpamProtection\Admin;

use Highpots\SpamProtection\Helpers\HPSP_Constants;

if (! defined('ABSPATH')) {
    exit;
}


class HPSP_Admin_Manager
{
    private HPSP_Settings $settings;
    private HPSP_Log_Controller $logs_page;
    private static ?self $instance = null;

    public function __construct()
    {
        $this->settings = HPSP_Settings::get_instance();
        $this->logs_page = HPSP_Log_Controller::get_instance();

        add_action('admin_menu', [$this, 'add_pages']);
        add_action('admin_init', [$this, 'init']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_notice_scripts']);
    }


    public function init()
    {

        // Add installation notice
        add_action('wp_ajax_hpsp_dismiss_installation_notice', [$this, 'dismiss_installation_notice']);
    }

    /**
     * Enqueue scripts and styles for admin notice
     */
    public function enqueue_admin_notice_scripts(): void
    {
        // Only enqueue on plugin pages
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'hpsp') === false) {
            return;
        }


        // Inline CSS
        $style = "
        /* Hide all external notices on this page, keep only plugin notices */
        .notice:not(.hpsp-notice), .notice-error:not(.hpsp-notice), div.error:not(.hpsp-notice) {
            display: none !important;
        }";

        wp_register_style('hpsp-notice-style', false, array(), HPSP_PLUGIN_VERSION);
        wp_enqueue_style('hpsp-notice-style');
        wp_add_inline_style('hpsp-notice-style', $style);



        // Check if notice should be shown
        if (get_option(HPSP_Constants::INSTALLATION_NOTICE_DISMISSED, false)) {
            return;
        }

        // Enqueue jQuery (dependency)
        wp_enqueue_script('jquery');


        // Localize AJAX data
        wp_localize_script('jquery', 'hpspNotice', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce('hpsp_dismiss_notice')
        ));

        // Inline JavaScript (small and conditional)
        $script = "
        jQuery(document).ready(function($) {
            $(window).on('load', function() {
                $('.hpsp-installation-notice').css('display', 'block').animate({opacity: 1}, 500);
            });

            $('.hpsp-dismiss-notice').on('click', function(e) {
                e.preventDefault();
                var \$notice = $(this).closest('.hpsp-installation-notice');
                
                $.post(hpspNotice.ajaxurl, {
                    action: 'hpsp_dismiss_installation_notice',
                    hpsp_dismiss_notice_nonce: hpspNotice.nonce
                }).done(function(response) {
                    if (response.success) \$notice.fadeOut(300);
                });
            });
        });
        ";
        wp_add_inline_script('jquery', $script);
    }

    /**
     * Get singleton instance
     */
    public static function get_instance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }




    /**
     * Display installation welcome notice
     */
    public function display_installation_notice(): void
    {

        // Check if notice has been dismissed
        if (get_option(HPSP_Constants::INSTALLATION_NOTICE_DISMISSED, false)) {
            return;
        }

        // Only show on plugin pages
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'hpsp') === false) {
            return;
        }



?>
        <div class="notice notice-success hpsp-notice hpsp-installation-notice" style="position: relative; padding-right: 38px; display: none; opacity: 0;">
            <span class="dashicons dashicons-yes" style="color: #46b450; font-size: 24px; float: left; margin: 12px 12px 0 0;"></span>
            <div style="margin-left: 36px;">
                <h3 style="margin: 0.5em 0;"><?php esc_html_e('Congratulations — The plugin to protect your web forms against spam has been successfully installed.', 'highpots-spam-protection'); ?></h3>

                <p><?php
                    echo wp_kses(
                        sprintf(
                            /* translators: %1$s: opening <a> tag, %2$s: closing </a> tag */
                            __('Thank you for choosing our plugin. This plugin was developed because other anti-spam plugins did not work with our WordPress configuration (Crocoblock, Elementor Pro, WooCommerce + WooCommerce in-house development %1$s"StoreConnect"%2$s).', 'highpots-spam-protection'),
                            '<a href="' . esc_url('https://highpots.com/produkte/woocommerce-storeconnect/') . '" target="_blank" rel="noopener">',
                            '</a>'
                        ),
                        array(
                            'a' => array(
                                'href'   => array(),
                                'target' => array(),
                                'rel'    => array()
                            )
                        )
                    );
                    ?></p>

                <p><?php
                    echo wp_kses(
                        sprintf(
                            /* translators: %s: email address */
                            __('This version of the plugin is open source, free of charge, and licensed under GPLv2. We have developed this plugin from scratch and want to learn from it - that\'s why we even provide support for this free version at %s', 'highpots-spam-protection'),
                            '<a href="mailto:support@highpots.com">support@highpots.com</a>'
                        ),
                        array(
                            'a' => array(
                                'href' => array()
                            )
                        )
                    );
                    ?></p>

                <p><?php
                    echo wp_kses(
                        sprintf(
                            /* translators: %1$s: opening <a> tag, %2$s: closing </a> tag */
                            __('Of course, our support offering may change in the future. Therefore, please visit our %1$splugin landing page%2$s regularly to check whether the support promise is still valid.', 'highpots-spam-protection'),
                            '<a href="' . esc_url('https://highpots.com/produkte/wordpress-plugins/anti-spam-protection/') . '" target="_blank" rel="noopener">',
                            '</a>'
                        ),
                        array(
                            'a' => array(
                                'href'   => array(),
                                'target' => array(),
                                'rel'    => array()
                            )
                        )
                    );
                    ?></p>
                <br>
                <p><?php esc_html_e('Warm regards,', 'highpots-spam-protection'); ?><br>
                    <?php esc_html_e('Your HighPots Team', 'highpots-spam-protection'); ?></p>

            </div>

            <button type="button" class="notice-dismiss hpsp-dismiss-notice" style="position: absolute; top: 0; right: 1px;">
                <span class="screen-reader-text"><?php esc_html_e('Hide this message forever', 'highpots-spam-protection'); ?></span>
            </button>
        </div>
    <?php
    }

    /**
     * Handle AJAX request to dismiss installation notice
     */
    public function dismiss_installation_notice(): void
    {
        check_ajax_referer('hpsp_dismiss_notice', 'hpsp_dismiss_notice_nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error();
        }

        update_option(HPSP_Constants::INSTALLATION_NOTICE_DISMISSED, true);
        wp_send_json_success();
    }




    public function add_pages()
    {
        // Parent page
        add_menu_page(
            esc_html__('HighPots Spam Protection',  'highpots-spam-protection'), // Page title
            esc_html__('Spam Protection',  'highpots-spam-protection'),             // Menu title
            'manage_options',             // Capability required to access the page
            'hpsp-logs', // Menu slug
            [$this, 'render_logs_page'], // Callback function to render the page
            'dashicons-shield', // Icon url
            //10   // Position
        );

        add_submenu_page(
            'hpsp-logs',  // Parent slug
            esc_html__('HighPots Spam Protection Logs',  'highpots-spam-protection'), // Page title
            esc_html__('Logs',  'highpots-spam-protection'), // Menu title
            'manage_options', // Capability required to access the page
            'hpsp-logs', // Menu slug
            [$this, 'render_logs_page'] // Callback function to render the page

        );

        add_submenu_page(
            'hpsp-logs', // Parent slug
            esc_html__('HighPots Spam Protection Settings',  'highpots-spam-protection'), // Page title
            esc_html__('Settings',  'highpots-spam-protection'), // Menu title
            'manage_options', // Capability required to access the page
            'hpsp-settings', // Menu slug
            [$this, 'render_settings_page'] // Callback function to render the page
        );


        add_action('admin_notices', [$this, 'display_installation_notice'], 998);
    }


    /**
     * Render page
     */
    private function render_page(callable $render_content, string $tab_id): void
    {

        echo '<div class="wrap" id="hpsp-wrapper">';
        $this->render_title();

        $this->render_tabs($tab_id);
        $render_content(); // Call the callback to render specific content
        echo '</div>';
    }



    /**
     * Render logs page - create instance only when needed
     */
    public function render_logs_page(): void
    {
        $this->render_page(function () {
            $this->logs_page->render_logs_page();
        }, 'logs');
    }



    /**
     * Render settings page - create instance only when needed
     */
    public function render_settings_page(): void
    {
        $this->render_page(function () {
            $this->settings->render_settings_page();
        }, 'settings');
    }

    /**
     * Render Title
     */
    private function render_title()
    {
    ?>
        <h1 class="hpsp-page-heading"><?php echo esc_html__('HighPots Spam Protection for Web Forms', 'highpots-spam-protection'); ?></h1>
    <?php
    }


    /**
     * Render tab navigation
     *
     * @param string $current_tab The currently active tab ('logs' or 'settings')
     */
    private function render_tabs(string $current_tab): void
    {
        $tabs = [
            'logs' => [
                'title' => esc_html__('Logs',  'highpots-spam-protection'),
                'url' => admin_url('admin.php?page=hpsp-logs')
            ],
            'settings' => [
                'title' => esc_html__('Settings',  'highpots-spam-protection'),
                'url' => admin_url('admin.php?page=hpsp-settings')
            ]
        ];
    ?>
        <div class="hpsp-tab-navigation">
            <?php foreach ($tabs as $tab_key => $tab): ?>
                <a href="<?php echo esc_url($tab['url']); ?>"
                    class="hpsp-tab <?php echo $current_tab === $tab_key ? 'hpsp-tab-active' : ''; ?>">
                    <?php echo esc_html($tab['title']); ?>
                </a>
            <?php endforeach; ?>
        </div>
<?php
    }
}
