<?php
namespace Highpots\SpamProtection;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Autoloader class.
 *
 * PSR-4 compliant autoloader.
 *
 * After registering this autoloader, the following
 * line would cause the function to attempt to load
 * the `\Pressidium\Limit_Login_Attempts\Standalone\Button`
 * class from the `./includes/Standalone/Button.php` file:
 * 
 * new \Pressidium\Limit_Login_Attempts\Standalone\Button;
 */
class HPSP_Autoloader {

    /**
     * @var string Project-specific namespace prefix.
     */
    const PREFIX = 'Highpots\\SpamProtection\\';

    /**
     * @var string Base directory for the namespace prefix.
     */
    //const BASE_DIR = __DIR__ . '/includes/';
    const BASE_DIR = HPSP_PLUGIN_DIR . 'includes/';

    /**
     * Register loader.
     *
     * @link https://www.php.net/manual/en/function.spl-autoload-register.php
     */
    public static function register() {
        spl_autoload_register( array( __CLASS__, 'load_class' ) );
    }

    /**
     * Check whether the given class name uses the namespace prefix.
     *
     * @param string $class The class name to check.
     * @return bool
     */
    private static function starts_with_namespace_prefix( $class ) {
        $len = strlen( self::PREFIX );
        return strncmp( self::PREFIX, $class, $len ) === 0;
    }


    
    private static function is_in_enum_namespace( $class ) {
        return str_contains( $class, '\\Enums\\' );
    }

 /**
     * Return the mapped file for the namespace prefix and the given class name.
     *
     * Replace the namespace prefix with the base directory,
     * replace namespace separators with directory separators,
     * and append with `.php`.
     *
     * @param string $class The fully-qualified class name.
     * @return string
     */
    private static function get_mapped_file( $class ) {
        //wp_die($class);
        $relative_class = substr( $class, strlen( self::PREFIX ) ); // Get the part after the namespace
        $parts = explode('\\', $relative_class); //Explode so we get the namespace and class name parts
    	$className = array_pop($parts); // Get the class name
    	$namespacePath = strtolower(implode('/', $parts)); //Get namespace path, convert it to lowercase


        $classOrEnum = self::is_in_enum_namespace($class) ? 'enum' : 'class';
        return self::BASE_DIR . $namespacePath . "/{$classOrEnum}-" . strtolower(str_replace('_', '-', $className)) . '.php';
    }
    /**
     * Require the file at the given path, if it exists.
     *
     * @param string $file
     */
    private static function require_file( $file ) {
       
        if ( file_exists( $file ) ) {
            require $file;
        }
    }

    /**
     * Load the class file for the given class name.
     *
     * @param string $class The fully-qualified class name.
     */
    private static function load_class( $class ) {
        if ( ! self::starts_with_namespace_prefix( $class ) ) {
            /*
             * Class does not use the namespace prefix,
             * move to the next registered autoloader.
             */
            return;
        }

        $mapped_file = self::get_mapped_file( $class );

        //wp_die('Loading class: ' . $class . ' from file: ' . $mapped_file . PHP_EOL);
        self::require_file( $mapped_file );
    }

}
