(function() {
    'use strict';

    // Check if config is available
    if (typeof hpsp_config === 'undefined') {
        return;
    }

    const config = hpsp_config;
    const processedForms = new WeakSet();

    /**
     * Generate a unique form ID based on form attributes
     */
    function generateFormId(form) {
        // Try existing form ID
        if (form.id) {
            return form.id;
        }

        // Try form name
        if (form.name) {
            return form.name;
        }

        // Try action URL
        if (form.action && form.action !== '#' && form.action !== '') {
            return 'form_' + hashString(form.action);
        }

        // Fallback: generate from form position
        const forms = document.querySelectorAll('form');
        const index = Array.from(forms).indexOf(form);
        return 'form_auto_' + index + '_' + Date.now();
    }

    /**
     * Simple hash function for strings
     */
    function hashString(str) {
        let hash = 0;
        for (let i = 0; i < str.length; i++) {
            const char = str.charCodeAt(i);
            hash = ((hash << 5) - hash) + char;
            hash = hash & hash;
        }
        return Math.abs(hash).toString(16);
    }

    /**
     * Create hidden input element
     */
    function createHiddenInput(name, value) {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = name;
        input.value = value;
        return input;
    }

    /**
     * Create honeypot input element
     */
    function createHoneypotInput(name) {
        const input = document.createElement('input');
        input.type = 'text';
        input.name = name;
        input.value = '';
        input.style.cssText = 'position:absolute;left:-9999px;width:1px;height:1px;opacity:0;';
        input.setAttribute('aria-hidden', 'true');
        input.setAttribute('tabindex', '-1');
        input.setAttribute('autocomplete', 'off');
        return input;
    }

    /**
     * Inject spam protection fields into a form
     */
    async function protectForm(form) {
        // Skip if already processed
        if (processedForms.has(form)) {
            return;
        }

        // Skip forms that already have our fields
        if (form.querySelector('input[name="hpsp_form_id"]')) {
            processedForms.add(form);
            return;
        }

        // Skip forms with specific exclusion attribute
        if (form.hasAttribute('data-hpsp-skip')) {
            return;
        }

        const formId = generateFormId(form);

        try {
            // Fetch token from REST API
            const response = await fetch(config.rest_url + 'token', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': config.nonce
                },
                body: JSON.stringify({ form_id: formId })
            });

            if (!response.ok) {
                console.warn('HPSP: Failed to fetch token for form:', formId);
                return;
            }

            const data = await response.json();

            if (!data.success) {
                console.warn('HPSP: Token generation failed:', data);
                return;
            }

            // Create container for our fields
            const container = document.createElement('div');
            container.style.display = 'none';
            container.setAttribute('data-hpsp-fields', 'true');

            // Add hidden fields
            container.appendChild(createHiddenInput('hpsp_form_id', data.form_id));
            container.appendChild(createHiddenInput('hpsp_hmac_token', data.token));
            container.appendChild(createHiddenInput('hpsp_hmac_timestamp', data.timestamp));
            container.appendChild(createHiddenInput('hpsp_hmac_nonce', data.nonce));
            container.appendChild(createHiddenInput('hpsp_rendered_at', data.rendered_at));

            // Add honeypot field
            container.appendChild(createHoneypotInput(data.honeypot_name));

            // Inject into form
            form.appendChild(container);

            // Mark as processed
            processedForms.add(form);

        } catch (error) {
            console.warn('HPSP: Error protecting form:', error);
        }
    }

    /**
     * Process all forms on the page
     */
    function processAllForms() {
        const forms = document.querySelectorAll('form');
        forms.forEach(form => protectForm(form));
    }

    /**
     * Set up MutationObserver for dynamically added forms
     */
    function observeDynamicForms() {
        const observer = new MutationObserver(function(mutations) {
            mutations.forEach(function(mutation) {
                mutation.addedNodes.forEach(function(node) {
                    if (node.nodeType !== Node.ELEMENT_NODE) {
                        return;
                    }

                    // Check if the added node is a form
                    if (node.tagName === 'FORM') {
                        protectForm(node);
                    }

                    // Check for forms inside the added node
                    const forms = node.querySelectorAll ? node.querySelectorAll('form') : [];
                    forms.forEach(form => protectForm(form));
                });
            });
        });

        observer.observe(document.body, {
            childList: true,
            subtree: true
        });
    }

    /**
     * Initialize when DOM is ready
     */
    function init() {
        // Process existing forms
        processAllForms();

        // Watch for dynamically added forms
        observeDynamicForms();
    }

    // Start when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

})();
