<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

/**
 * Highlights search keywords in the title, content, or excerpt.
 */
function htsk_highlight($text, $post_id = 0) {
    
    // Read enabled option from stored options array (fallback to enabled)
    $opts = get_option('htsk_options', array());
    $enabled = isset($opts['enabled']) ? intval($opts['enabled']) : 1;

    if (!is_search() || ! $enabled) {
        return $text; 
    }

    $query = get_search_query();
    if (!$query) {
        return $text;
    }

    $current = current_filter();

    // Check if this is an excerpt and if we need to highlight read more
    $should_highlight_readmore = false;
    if ($current === 'the_excerpt' && $post_id) {
        $should_highlight_readmore = htsk_should_highlight_readmore($query, $post_id);
    }

    // Process the text for keyword highlighting
    $text = htsk_process_text_highlighting($text, $query);
    
    // Handle read more highlighting if needed
    if ($should_highlight_readmore) {
        $text = htsk_highlight_readmore_elements($text);
    }

    return $text;
}

/**
 * Process text highlighting with improved phrase and word matching
 */
function htsk_process_text_highlighting($text, $query) {
    $words = preg_split('/\s+/', trim($query));
    $words = array_filter($words, 'strlen'); // Remove empty elements
    
    if (empty($words)) {
        return $text;
    }

    $split_text = wp_html_split($text);
    
    foreach ($split_text as $key => $part) {
        // Only process text parts, not HTML tags
        if (false === strpos($part, '<')) {
            $split_text[$key] = htsk_highlight_in_text($part, $words, $query);
        }
    }
    
    return implode('', $split_text);
}

/**
 * Highlight keywords in a text part with phrase priority
 */
function htsk_highlight_in_text($text, $words, $full_query) {
    // Try phrase first if multiple words
    if (count($words) > 1) {
        $phrase_pattern = '/' . preg_quote($full_query, '/') . '/iu';
        $text = preg_replace(
            $phrase_pattern,
            '<span class="htsk-highlight">$0</span>',
            $text
        );
    }
    
    // Highlight individual words
    foreach ($words as $word) {
        if (strlen($word) > 0) {
            $word_pattern = '/(' . preg_quote($word, '/') . ')/iu';
            $text = preg_replace(
                $word_pattern,
                '<span class="htsk-highlight">$1</span>',
                $text
            );
        }
    }
    
    return $text;
}

/**
 * Check if the full post content contains search keywords (for read more highlighting)
 */
function htsk_should_highlight_readmore($query, $post_id) {
    if (!$post_id) {
        return false;
    }
    
    $post = get_post($post_id);
    if (!$post) {
        return false;
    }
    
    $full_content = $post->post_content;
    $excerpt = $post->post_excerpt ? $post->post_excerpt : wp_trim_excerpt('', $post);
    
    // If excerpt == content, no read more
    if (trim(wp_strip_all_tags($full_content)) === trim(wp_strip_all_tags($excerpt))) {
        return false;
    }
    
    $remaining_content = str_replace($excerpt, '', $full_content);
    $words = preg_split('/\s+/', trim($query));
    
    foreach ($words as $word) {
        if (strlen($word) > 0 && stripos($remaining_content, $word) !== false) {
            return true;
        }
    }
    
    return false;
}

/**
 * Highlight read more text and ellipsis when keywords exist in remaining content
 */
function htsk_highlight_readmore_elements($text) {
    $patterns = array(
        '/(\[&hellip;\])/i',
        '/(\.{3,})/i',
        '/(&hellip;)/i',
        '/(…)/i',
        '/(read more)/i',
        '/(continue reading)/i',
        '/(\[\.{3,}\])/i',
    );
    
    foreach ($patterns as $pattern) {
        $text = preg_replace(
            $pattern,
            '<span class="htsk-highlight">$1</span>',
            $text
        );
    }
    
    return $text;
}

/**
 * Wrapper with post context
 */
function htsk_highlight_with_post_context($text, $post_id = 0) {
    global $post;
    if (!$post_id && isset($post) && is_object($post)) {
        $post_id = $post->ID;
    }
    return htsk_highlight($text, $post_id);
}

// Apply highlight filters
add_filter('the_title', 'htsk_highlight_with_post_context', 10, 2);
add_filter('get_the_title', 'htsk_highlight_with_post_context', 10, 2);
add_filter('the_excerpt', 'htsk_highlight_with_post_context', 10, 1);
add_filter('the_content', 'htsk_highlight_with_post_context', 10, 1);

/**
 * Enqueue and add inline CSS for highlighting
 */
function htsk_enqueue_highlight_styles() {
    if (is_search()) {

        $opts = wp_parse_args(get_option('htsk_options', array()), array(
            'enabled'    => 1,
            'text_color' => '#000000',
            'bg_color'   => '#ffff00',
        ));
        
        $text = esc_attr($opts['text_color']);
        $bg   = esc_attr($opts['bg_color']);
        
        $css = ".htsk-highlight { 
            color: {$text} !important; 
            background-color: {$bg} !important; 
            padding: 2px 4px; 
            border-radius: 3px; 
        }";
        
        // Register first
        wp_register_style('htsk-highlight-styles', false, array(), '1.0.0');
        wp_enqueue_style('htsk-highlight-styles', false, array(), '1.0.0');
        
        // Then add inline CSS
        wp_add_inline_style('htsk-highlight-styles', esc_html($css));
    }
}
add_action('wp_enqueue_scripts', 'htsk_enqueue_highlight_styles');