<?php
/*
Plugin Name: Hide Reminders
Description: Removes all admin notices and reminders from the WordPress dashboard for a distraction-free experience.
Version: 1.0
Author: FardanSultan
Author URI: https://profiles.wordpress.org/fardansultan
License: GPL2
License URI: http://www.gnu.org/licenses/gpl-2.0.html
*/

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Add custom admin menu
function clean_wp_admin_menu() {
    add_menu_page(
        'Hide reminders',
        'Hide reminders',
        'manage_options',
        'clean-wp-admin',
        'clean_wp_admin_settings_page',
        'dashicons-visibility',
        80
    );
}
add_action('admin_menu', 'clean_wp_admin_menu');

// Create settings page
function clean_wp_admin_settings_page() {
    ?>
    <div class="wrap-1">
        <div class="wrap-2">
            <h2 class="hide-remin">Hide Reminders</h2>
            <form method="post" action="options.php">
                <?php
                settings_fields('clean_wp_admin_settings');
                do_settings_sections('clean-wp-admin');
                submit_button();
                ?>
            </form>
        </div>
        <img class="hide-remin-logo" src="<?php echo esc_url(plugin_dir_url(__FILE__) . 'assets/logo.png'); ?>" alt="Plugin Logo" height="25">
    </div>
    <?php
}

// Register settings with sanitization
function clean_wp_admin_register_settings() {
    register_setting('clean_wp_admin_settings', 'clean_wp_admin_disable_notices', 'intval');
    add_settings_section('clean_wp_admin_section', 'Settings', null, 'clean-wp-admin');
    add_settings_field(
        'clean_wp_admin_disable_notices',
        'Disable Admin Notices',
        'clean_wp_admin_disable_notices_callback',
        'clean-wp-admin',
        'clean_wp_admin_section'
    );
}
add_action('admin_init', 'clean_wp_admin_register_settings');

// Callback for checkbox setting
function clean_wp_admin_disable_notices_callback() {
    $option = get_option('clean_wp_admin_disable_notices');
    echo '<input type="checkbox" name="clean_wp_admin_disable_notices" value="1" ' . checked(1, $option, false) . ' /> Hide all admin notices';
}

// Remove admin notices
function clean_wp_admin_remove_notices() {
    if (get_option('clean_wp_admin_disable_notices')) {
        remove_all_actions('admin_notices');
        remove_all_actions('all_admin_notices');
    }
}
add_action('admin_init', 'clean_wp_admin_remove_notices');

// Add custom CSS to hide notices properly
function clean_wp_admin_hide_notices_css() {
    if (get_option('clean_wp_admin_disable_notices')) {
        wp_register_style('hide-reminders-style', false);
        wp_enqueue_style('hide-reminders-style');
        wp_add_inline_style('hide-reminders-style', '.notice, .updated, .error { display: none !important; }');
    }
}
add_action('admin_enqueue_scripts', 'clean_wp_admin_hide_notices_css');

// Add custom styles for your plugin settings page
function clean_wp_admin_custom_styles($hook_suffix) {
    // Only load on your plugin page
    if ($hook_suffix === 'toplevel_page_clean-wp-admin') {
        wp_register_style('hide-reminders-custom-style', false);
        wp_enqueue_style('hide-reminders-custom-style');
        wp_add_inline_style('hide-reminders-custom-style', '
            .wrap-1 {
                display: flex;
                justify-content: space-between;
                padding-right: 20px;
                align-items: flex-start;
            }
            .hide-remin {
                font-size:23px;
            }
            .hide-remin-logo {
                padding-top:24px;
            }
        ');
    }
}
add_action('admin_enqueue_scripts', 'clean_wp_admin_custom_styles');
