<?php
/**
 * Plugin Name: Hide Price Until Login for WooCommerce
 * Description: Hide WooCommerce product prices and Add to Cart button until the user logs in.
 * Version: 1.0
 * Author: Ankit Patel
 * Author URI: https://ankitpatel.co.in/
 * Text Domain: hide-price-until-login-for-woocommerce
 * License: GNU General Public License v3.0
 * Requires Plugins: woocommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('HPUL_Hide_Price')) {

    class HPUL_Hide_Price {

        public function __construct() {

            add_filter('woocommerce_get_price_html', array($this, 'hide_price'), 10, 2);
            add_filter('woocommerce_is_purchasable', array($this, 'disable_purchase_for_guests'), 10, 2);
            add_filter('woocommerce_loop_add_to_cart_link', array($this, 'remove_loop_button'), 10, 2);

            add_action('admin_menu', array($this, 'add_settings_page'));
            add_action('admin_init', array($this, 'register_settings'));
        }

        /* ---------------------------------------------
         * FRONTEND LOGIC
         * ------------------------------------------- */

        public function hide_price($price, $product) {

            if (!is_user_logged_in()) {

                $message = get_option(
                    'hpul_login_message',
                    __('Please login to see prices.', 'hide-price-until-login-for-woocommerce')
                );

                return '<span class="hpul-login-message">' . esc_html($message) . '</span>';
            }

            return $price;
        }

        public function disable_purchase_for_guests($purchasable, $product) {

            if (!is_user_logged_in()) {
                return false;
            }

            return $purchasable;
        }

        public function remove_loop_button($button, $product) {

            if (!is_user_logged_in()) {
                return '';
            }

            return $button;
        }

        /* ---------------------------------------------
         * SETTINGS
         * ------------------------------------------- */

        public function register_settings() {

            register_setting(
                'hpul_settings_group',
                'hpul_login_message',
                array(
                    'type'              => 'string',
                    'sanitize_callback' => 'sanitize_text_field',
                    'default'           => __('Please login to see prices.', 'hide-price-until-login-for-woocommerce'),
                )
            );
        }

        public function add_settings_page() {

            add_submenu_page(
                'woocommerce',
                __('Hide Price Settings', 'hide-price-until-login-for-woocommerce'),
                __('Hide Price Until Login', 'hide-price-until-login-for-woocommerce'),
                'manage_woocommerce',
                'hide-price-until-login-for-woocommerce',
                array($this, 'settings_page')
            );
        }

        public function settings_page() {
            ?>
            <div class="wrap">
                <h1><?php esc_html_e('Hide Price Until Login Settings', 'hide-price-until-login-for-woocommerce'); ?></h1>

                <form method="post" action="options.php">
                    <?php
                    settings_fields('hpul_settings_group');
                    ?>
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <?php esc_html_e('Login Message', 'hide-price-until-login-for-woocommerce'); ?>
                            </th>
                            <td>
                                <input type="text"
                                       name="hpul_login_message"
                                       value="<?php echo esc_attr(get_option('hpul_login_message', __('Please login to see prices.', 'hide-price-until-login-for-woocommerce'))); ?>"
                                       class="regular-text" />
                                <p class="description">
                                    <?php esc_html_e('Message shown instead of price for guests.', 'hide-price-until-login-for-woocommerce'); ?>
                                </p>
                            </td>
                        </tr>
                    </table>
                    <?php submit_button(); ?>
                </form>
            </div>
            <?php
        }
    }

    new HPUL_Hide_Price();
}