<?php
/**
 * Plugin Name: Hide Login Shield
 * Description: Allows changing the login URL for improved security.
 * Version: 1.0
 * Author: Oliver D.
 * License: GPLv2 or later
 * Text Domain: hide-login-shield
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Hide_Login_Shield {
    
    private $option_name = 'hlsw_login_slug';

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'hlsw_add_admin_menu' ) );
        add_action( 'admin_init', array( $this, 'hlsw_settings_init' ) );
        add_action( 'init', array( $this, 'hlsw_handle_login' ) );
        add_action( 'init', array( $this, 'hlsw_redirect_old_login' ) );
        add_filter( 'login_url', array( $this, 'hlsw_custom_login_url' ), 10, 3 );
        add_filter( 'site_url', array( $this, 'hlsw_site_url' ), 10, 4 );
        register_deactivation_hook( __FILE__, array( $this, 'hlsw_deactivate' ) );
    }

    public function hlsw_add_admin_menu() {
        add_options_page(
            'Hide Login Shield',
            'Hide Login Shield',
            'manage_options',
            'hide-login-shield',
            array( $this, 'hlsw_options_page' )
        );
    }

    public function hlsw_settings_init() {
        register_setting( 'hlsw_settings', $this->option_name, array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => '',
        ) );

        add_settings_section(
            'hlsw_settings_section',
            __( 'Hide Login Shield Settings', 'hide-login-shield' ),
            null,
            'hlsw_settings'
        );

        add_settings_field(
            'hlsw_login_slug_field',
            __( 'New Login Prefix', 'hide-login-shield' ),
            array( $this, 'hlsw_login_slug_field_render' ),
            'hlsw_settings',
            'hlsw_settings_section'
        );
    }

    public function hlsw_login_slug_field_render() {
        $option = get_option( $this->option_name );
        ?>
        <input type="text" name="<?php echo esc_attr( $this->option_name ); ?>" value="<?php echo esc_attr( $option ); ?>" />
        <p class="description">Enter a unique prefix for the login page, for example, "mylogin".</p>
        <?php
    }

    public function hlsw_options_page() {
        ?>
        <div class="wrap">
            <h1>Hide Login Shield</h1>
            <form action="options.php" method="post">
                <?php
                settings_fields( 'hlsw_settings' );
                do_settings_sections( 'hlsw_settings' );
                submit_button();
                ?>
            </form>
            <?php
            $slug = get_option( $this->option_name );
            if ( $slug ) {
                $login_url = site_url( '/' . $slug );
                ?>
                <div class="updated notice">
                    <p><strong>New Login URL:</strong> <a href="<?php echo esc_url( $login_url ); ?>"><?php echo esc_html( $login_url ); ?></a></p>
                    <p><strong>Important:</strong> Remember this URL. If you forget it, you can restore access by deleting the plugin via FTP or your hosting control panel's file manager.</p>
                </div>
                <?php
            }
            ?>
        </div>
        <?php
    }

    public function hlsw_handle_login() {
        $slug = get_option( $this->option_name );
        if ( $slug ) {
            $current_url = filter_input( INPUT_SERVER, 'REQUEST_URI', FILTER_SANITIZE_URL );
            $current_url = sanitize_text_field( trim( wp_parse_url( $current_url, PHP_URL_PATH ), '/' ) );

            if ( $current_url === $slug ) {
                require_once( ABSPATH . 'wp-login.php' );
                exit;
            }
        }
    }

    public function hlsw_redirect_old_login() {
        $slug = get_option( $this->option_name );
        if ( $slug ) {
            $current_url = filter_input( INPUT_SERVER, 'REQUEST_URI', FILTER_SANITIZE_URL );
            $current_url = sanitize_text_field( trim( wp_parse_url( $current_url, PHP_URL_PATH ), '/' ) );

            if ( in_array( $current_url, array( 'wp-login.php', 'wp-admin' ), true ) ) {
                wp_redirect( home_url() );
                exit;
            }
        }
    }

    public function hlsw_custom_login_url( $login_url, $redirect, $force_reauth ) {
        $slug = get_option( $this->option_name );
        if ( $slug ) {
            $login_url = site_url( '/' . $slug );
        }
        return $login_url;
    }

    public function hlsw_site_url( $url, $path, $orig_scheme, $blog_id ) {
        $slug = get_option( $this->option_name );
        if ( $slug && strpos( $path, 'wp-login.php' ) !== false ) {
            return site_url( '/' . $slug );
        }
        return $url;
    }

    public function hlsw_deactivate() {
        delete_option( $this->option_name );
    }
}

new Hide_Login_Shield();