<?php
/**
 * Plugin Name: Hibiscus Instant Chat
 * Plugin URI: https://hibiscustechnolab.com/hibiscus-instant-chat
 * Version: 1.0.2
 * Author: Hibiscus Technolab
 * Text Domain: hibiscus-instant-chat
 * Domain Path: /languages
 * Description: Instant Chat 📞 — Turn your website visitors into instant conversations! Connect seamlessly with a chat widget on your website.
 * Author URI: https://hibiscustechnolab.com
 * License: GPLv2 or later
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

// Constants
define( 'HIBISCUS_CHAT_MESSENGER_VERSION', '1.0.2' );
define( 'HIBISCUS_CHAT_MESSENGER_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'HIBISCUS_CHAT_MESSENGER_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

/*--------------------------------------------------
|  Register Settings
--------------------------------------------------*/
add_action( 'admin_init', 'hibiscus_chat_messenger_register_settings' );
function hibiscus_chat_messenger_register_settings() {
    register_setting(
        'hibiscus_chat_messenger_options',
        'hibiscus_chat_messenger_options',
        'hibiscus_chat_messenger_validate_options'
    );

    add_settings_section(
        'hibiscus_chat_messenger_main',
        __( 'Main Settings', 'hibiscus-instant-chat' ),
        '__return_false',
        'hibiscus_chat_messenger'
    );

    add_settings_field( 'phone', __( 'Phone (international format, no + or spaces)', 'hibiscus-instant-chat' ), 'hibiscus_chat_messenger_field_phone', 'hibiscus_chat_messenger', 'hibiscus_chat_messenger_main' );
    add_settings_field( 'default_message', __( 'Default message', 'hibiscus-instant-chat' ), 'hibiscus_chat_messenger_field_default_message', 'hibiscus_chat_messenger', 'hibiscus_chat_messenger_main' );
    add_settings_field( 'greeting', __( 'Default greeting (shown in popup)', 'hibiscus-instant-chat' ), 'hibiscus_chat_messenger_field_greeting', 'hibiscus_chat_messenger', 'hibiscus_chat_messenger_main' );
    add_settings_field( 'position', __( 'Button position', 'hibiscus-instant-chat' ), 'hibiscus_chat_messenger_field_position', 'hibiscus_chat_messenger', 'hibiscus_chat_messenger_main' );
    add_settings_field( 'label', __( 'Button label (optional)', 'hibiscus-instant-chat' ), 'hibiscus_chat_messenger_field_label', 'hibiscus_chat_messenger', 'hibiscus_chat_messenger_main' );
    add_settings_field( 'autopopup', __( 'Auto popup (seconds, 0 to disable)', 'hibiscus-instant-chat' ), 'hibiscus_chat_messenger_field_autopopup', 'hibiscus_chat_messenger', 'hibiscus_chat_messenger_main' );
    add_settings_field(
        'buy_me_a_coffee',
        __( 'Support Us ☕', 'hibiscus-instant-chat' ),
        'hibiscus_chat_messenger_field_buy_me_a_coffee',
        'hibiscus_chat_messenger',
        'hibiscus_chat_messenger_main'
    );
}
function hibiscus_chat_messenger_field_buy_me_a_coffee() {
    echo '<a href="https://buymeacoffee.com/hibiscustechnolab" target="_blank" style="text-decoration:none; font-weight:bold;">☕ Buy Me a Coffee</a>';
}
/*--------------------------------------------------
|  Settings Field Callbacks
--------------------------------------------------*/
function hibiscus_chat_messenger_field_phone() {
    $opts = hibiscus_chat_messenger_get_options();
    printf('<input type="text" name="hibiscus_chat_messenger_options[phone]" value="%s" class="regular-text" />', esc_attr( $opts['phone'] ?? '' ));
    echo '<p class="description">' . esc_html__( 'E.g. 919999999999 (country code + number, no +)', 'hibiscus-instant-chat' ) . '</p>';
}

function hibiscus_chat_messenger_field_default_message() {
    $opts = hibiscus_chat_messenger_get_options();
    printf('<textarea name="hibiscus_chat_messenger_options[default_message]" rows="3" cols="40">%s</textarea>', esc_textarea( $opts['default_message'] ?? '' ));
}

function hibiscus_chat_messenger_field_greeting() {
    $opts = hibiscus_chat_messenger_get_options();
    printf('<input type="text" name="hibiscus_chat_messenger_options[greeting]" value="%s" class="regular-text" />', esc_attr( $opts['greeting'] ?? '' ));
    echo '<p class="description">' . esc_html__( 'Greeting shown in popup above the text box.', 'hibiscus-instant-chat' ) . '</p>';
}

function hibiscus_chat_messenger_field_position() {
    $opts = hibiscus_chat_messenger_get_options();
    $pos = $opts['position'] ?? 'right';
    printf('<select name="hibiscus_chat_messenger_options[position]"><option value="right" %s>%s</option><option value="left" %s>%s</option></select>',
        selected( $pos, 'right', false ), esc_html__( 'Right', 'hibiscus-instant-chat' ),
        selected( $pos, 'left', false ), esc_html__( 'Left', 'hibiscus-instant-chat' )
    );
}

function hibiscus_chat_messenger_field_label() {
    $opts = hibiscus_chat_messenger_get_options();
    printf('<input type="text" name="hibiscus_chat_messenger_options[label]" value="%s" class="regular-text" />', esc_attr( $opts['label'] ?? '' ));
}

function hibiscus_chat_messenger_field_autopopup() {
    $opts = hibiscus_chat_messenger_get_options();
    printf('<input type="number" min="0" name="hibiscus_chat_messenger_options[autopopup]" value="%s" class="small-text" />', esc_attr( $opts['autopopup'] ?? '8' ));
    echo '<p class="description">' . esc_html__( 'Auto popup after n seconds. Set 0 to disable.', 'hibiscus-instant-chat' ) . '</p>';
}

/*--------------------------------------------------
|  Get Options
--------------------------------------------------*/
function hibiscus_chat_messenger_get_options() {
    $defaults = array(
        'phone' => '',
        'default_message' => __( 'Hello, I would like to know more about your services.', 'hibiscus-instant-chat' ),
        'greeting' => __( 'Hi! How can we help you today?', 'hibiscus-instant-chat' ),
        'position' => 'right',
        'label' => '',
        'autopopup' => 8,
    );
    $opts = get_option( 'hibiscus_chat_messenger_options', array() );
    return wp_parse_args( $opts, $defaults );
}

/*--------------------------------------------------
|  Validate Options
--------------------------------------------------*/
function hibiscus_chat_messenger_validate_options( $input ) {
    $output = array();
    $output['phone'] = isset( $input['phone'] ) ? substr( preg_replace( '/\D+/', '', $input['phone'] ), 0, 20 ) : '';
    $output['default_message'] = isset( $input['default_message'] ) ? sanitize_textarea_field( $input['default_message'] ) : '';
    $output['greeting'] = isset( $input['greeting'] ) ? sanitize_text_field( $input['greeting'] ) : '';
    $output['position'] = in_array( $input['position'] ?? 'right', array( 'left', 'right' ), true ) ? $input['position'] : 'right';
    $output['label'] = isset( $input['label'] ) ? sanitize_text_field( $input['label'] ) : '';
    $output['autopopup'] = isset( $input['autopopup'] ) ? intval( $input['autopopup'] ) : 0;
    return $output;
}

/*--------------------------------------------------
|  Admin Menu
--------------------------------------------------*/
add_action( 'admin_menu', 'hibiscus_chat_messenger_admin_menu' );
function hibiscus_chat_messenger_admin_menu() {
    add_options_page(
        __( 'Hibiscus Instant Chat', 'hibiscus-instant-chat' ),
        __( 'Hibiscus Instant Chat', 'hibiscus-instant-chat' ),
        'manage_options',
        'hibiscus_chat_messenger',
        'hibiscus_chat_messenger_options_page'
    );
}

function hibiscus_chat_messenger_options_page() {
    if ( ! current_user_can( 'manage_options' ) ) return;
    ?>
    <div class="wrap">
        <h1><?php esc_html_e( 'Hibiscus Instant Chat', 'hibiscus-instant-chat' ); ?></h1>
        <form method="post" action="options.php">
            <?php
            settings_fields( 'hibiscus_chat_messenger_options' );
            do_settings_sections( 'hibiscus_chat_messenger' );
            submit_button();
            ?>
        </form>
    </div>
    <?php
}

/*--------------------------------------------------
|  Enqueue Scripts
--------------------------------------------------*/
add_action( 'wp_enqueue_scripts', 'hibiscus_chat_messenger_enqueue_scripts' );
function hibiscus_chat_messenger_enqueue_scripts() {
    wp_enqueue_style( 'hibiscus_chat_messenger_style', HIBISCUS_CHAT_MESSENGER_PLUGIN_URL . 'inc/css/style.css', array(), HIBISCUS_CHAT_MESSENGER_VERSION );
    wp_enqueue_script( 'hibiscus_chat_messenger_js', HIBISCUS_CHAT_MESSENGER_PLUGIN_URL . 'inc/js/main.js', array(), HIBISCUS_CHAT_MESSENGER_VERSION, true );

    $opts = hibiscus_chat_messenger_get_options();
    wp_localize_script( 'hibiscus_chat_messenger_js', 'ChatsAppWP_Settings', array(
        'phone' => $opts['phone'],
        'message' => $opts['default_message'],
        'greeting' => $opts['greeting'],
        'position' => $opts['position'],
        'label' => $opts['label'],
        'autopopup' => intval($opts['autopopup']),
        'is_mobile' => wp_is_mobile(),
    ) );
}

/*--------------------------------------------------
|  Shortcode
--------------------------------------------------*/
add_shortcode( 'hibiscus_chat_messenger_button', 'hibiscus_chat_messenger_render_button' );
function hibiscus_chat_messenger_render_button( $atts ) {
    $opts = hibiscus_chat_messenger_get_options();
    if ( empty( $opts['phone'] ) ) return '';
    ob_start();
    ?>
    <div id="hibiscus_chat_messenger_container" class="hibiscus_chat_messenger_container hibiscus_chat_messenger-<?php echo esc_attr( $opts['position'] ); ?>" aria-live="polite">
        <button id="hibiscus_chat_messenger_toggle" class="hibiscus_chat_messenger_toggle" aria-expanded="false" aria-controls="hibiscus_chat_messenger_popup">
            <span class="hibiscus_chat_messenger_icon" aria-hidden="true">&#x1F4DE;</span>
            <span class="hibiscus_chat_messenger_label"><?php echo esc_html( $opts['label'] ?: __( 'Chat', 'hibiscus-instant-chat' ) ); ?></span>
        </button>
        <div id="hibiscus_chat_messenger_popup" class="hibiscus_chat_messenger_popup" role="dialog" aria-hidden="true">
            <button id="hibiscus_chat_messenger_close" class="hibiscus_chat_messenger_close" aria-label="<?php echo esc_attr__( 'Close chat', 'hibiscus-instant-chat' ); ?>">&times;</button>
            <div class="hibiscus_chat_messenger_content">
                <div class="hibiscus_chat_messenger_greeting"><?php echo esc_html( $opts['greeting'] ); ?></div>
                <div class="hibiscus_chat_messenger_sub"><?php echo esc_html__( 'Write a message and we will reply as soon as possible.', 'hibiscus-instant-chat' ); ?></div>
                <div class="hibiscus_chat_messenger_inputs">
                    <label for="hibiscus_chat_messenger_user_msg" class="screen-reader-text"><?php echo esc_html__( 'Your message','hibiscus-instant-chat' ); ?></label>
                    <textarea id="hibiscus_chat_messenger_user_msg" class="hibiscus_chat_messenger_user_msg" rows="3" placeholder="<?php echo esc_attr__( 'Type your message here...', 'hibiscus-instant-chat' ); ?>"></textarea>
                    <div class="hibiscus_chat_messenger_actions">
                        <button id="hibiscus_chat_messenger_send" class="hibiscus_chat_messenger_send"><?php echo esc_html__( 'Send to WhatsApp', 'hibiscus-instant-chat' ); ?></button>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <?php
    return ob_get_clean();
}

/*--------------------------------------------------
|  Uninstall Cleanup
--------------------------------------------------*/
register_uninstall_hook( __FILE__, 'hibiscus_chat_messenger_uninstall' );
function hibiscus_chat_messenger_uninstall() {
    if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) exit;
    delete_option( 'hibiscus_chat_messenger_options' );
}
