import React from "react";
import { useSelect } from "@wordpress/data";

const sanitizeHref = (raw = "") => {
  const value = String(raw || "").trim();
  if (!value) return "#";

  // allow relative URLs like /about, #section, ?q=...
  if (value.startsWith("/") || value.startsWith("#") || value.startsWith("?")) {
    return value;
  }

  // allow absolute URLs only with safe protocols
  try {
    const url = new URL(value);
    const allowed = ["http:", "https:", "mailto:", "tel:"];
    return allowed.includes(url.protocol) ? url.href : "#";
  } catch {
    return "#";
  }
};

const HeroSection = ({ attributes }) => {
  const { data = {} } = attributes || {};
  const href = sanitizeHref(data?.btn?.link);
  const isBlank = !!data?.btn?.newTab;

  // WP recommended: store only attachment ID
  const imageId = Number(data?.imageId || 0);

  // Resolve attachment URL from Media Library for display (safe + stable)
  const { imageUrl } = useSelect(
    (select) => {
      if (!imageId) return { imageUrl: "" };

      const core = select("core");

      // Ensure request finished (prevents undefined flicker)
      const done = core.hasFinishedResolution("getMedia", [imageId]);
      if (!done) return { imageUrl: "" };

      const media = core.getMedia(imageId);

      const url =
        media?.source_url ||
        media?.media_details?.sizes?.full?.source_url ||
        media?.media_details?.sizes?.large?.source_url ||
        "";

      return { imageUrl: url };
    },
    [imageId]
  );

  return (
    <section className="bph-hero bph-hero--bg-image" style={{ "--overlay": 0.55 }}>
      <div
        className="bph-hero__bg"
        style={{ backgroundImage: imageUrl ? `url("${imageUrl}")` : "" }}
        aria-hidden="true"
      />

      <div className="bph-hero__container">
        <div className="bph-hero__content">
          <h1 className="bph-hero__title">{data?.title}</h1>

          <p className="bph-hero__subtitle">{data?.des}</p>

          <div className="bph-hero__actions">
            <a
              className="bph-btn bph-btn--primary"
              href={href}
              target={isBlank ? "_blank" : undefined}
              rel={isBlank ? "noopener noreferrer" : undefined}
            >
              {data?.btn?.text}
            </a>
          </div>
        </div>
      </div>
    </section>
  );
};

export default HeroSection;
