import { __ } from "@wordpress/i18n";
import { PanelBody, TextControl, Button, Spinner } from "@wordpress/components";
import { MediaUpload, MediaUploadCheck, URLInput } from "@wordpress/block-editor";
import { useSelect } from "@wordpress/data";

const General = ({ attributes, setAttributes }) => {
  const { data = {} } = attributes || {};
  const setData = (patch) => setAttributes({ data: { ...data, ...patch } });

  // Store ONLY attachment ID (Media Library based)
  const imageId = Number(data?.imageId || 0);

  // Resolve URL from Media Library for preview only
  const { imageUrl, isResolving } = useSelect(
    (select) => {
      if (!imageId) return { imageUrl: "", isResolving: false };

      const core = select("core");
      const isResolving = !core.hasFinishedResolution("getMedia", [imageId]);
      const media = core.getMedia(imageId);

      const url =
        media?.source_url ||
        media?.media_details?.sizes?.full?.source_url ||
        media?.media_details?.sizes?.large?.source_url ||
        "";

      return { imageUrl: url, isResolving };
    },
    [imageId]
  );

  return (
    <PanelBody
      className="bPlPanelBody"
      title={__("Hero Section Data", "hero-section")}
      initialOpen={false}
    >
      <TextControl
        label={__("Title", "hero-section")}
        value={data?.title || ""}
        onChange={(v) => setData({ title: v })}
      />

      <TextControl
        label={__("Description", "hero-section")}
        value={data?.des || ""}
        onChange={(v) => setData({ des: v })}
      />

      <TextControl
        label={__("Button Text", "hero-section")}
        value={data?.btn?.text || ""}
        onChange={(v) => setData({ btn: { ...(data?.btn || {}), text: v } })}
      />

      {/* WP recommended URL control */}
      <div style={{ marginTop: 12 }}>
        <div style={{ fontSize: 12, opacity: 0.85, marginBottom: 6 }}>
          {__("Button URL", "hero-section")}
        </div>

        <URLInput
          value={data?.btn?.link || ""}
          onChange={(url) =>
            setData({ btn: { ...(data?.btn || {}), link: (url || "").trim() } })
          }
        />

        <p style={{ marginTop: 6, fontSize: 12, opacity: 0.85 }}>
          {__("Prefer internal links where possible.", "hero-section")}
        </p>
      </div>

      {/* Image Upload / Select (Media Library only) */}
      <div style={{ marginTop: 16 }}>
        <div style={{ fontSize: 12, opacity: 0.85, marginBottom: 6 }}>
          {__("Background Image", "hero-section")}
        </div>

        <MediaUploadCheck
          fallback={
            <p style={{ fontSize: 12, opacity: 0.85, margin: 0 }}>
              {__("You do not have permission to upload media.", "hero-section")}
            </p>
          }
        >
          <MediaUpload
            allowedTypes={["image"]}
            value={imageId || undefined}
            onSelect={(media) => {
              // MediaUpload can return object or array depending on settings
              const item = Array.isArray(media) ? media?.[0] : media;
              const id = Number(item?.id || 0);

              // Store only ID to avoid remote URL usage
              setData({ imageId: id });
            }}
            render={({ open }) => (
              <div
                style={{
                  display: "flex",
                  gap: 8,
                  flexWrap: "wrap",
                  alignItems: "center",
                }}
              >
                <Button variant="primary" onClick={open}>
                  {imageId
                    ? __("Change Image", "hero-section")
                    : __("Upload / Select Image", "hero-section")}
                </Button>

                {imageId ? (
                  <Button
                    variant="secondary"
                    isDestructive
                    onClick={() => setData({ imageId: 0 })}
                  >
                    {__("Remove", "hero-section")}
                  </Button>
                ) : null}

                {/* Preview */}
                {imageId ? (
                  <div style={{ display: "flex", alignItems: "center", minHeight: 56 }}>
                    {isResolving ? (
                      <Spinner />
                    ) : imageUrl ? (
                      <img
                        src={imageUrl}
                        alt={__("Selected background image", "hero-section")}
                        loading="lazy"
                        style={{
                          width: 56,
                          height: 56,
                          objectFit: "cover",
                          borderRadius: 6,
                          border: "1px solid rgba(0,0,0,0.12)",
                        }}
                      />
                    ) : null}
                  </div>
                ) : null}
              </div>
            )}
          />
        </MediaUploadCheck>

        <p style={{ marginTop: 8, fontSize: 12, opacity: 0.85 }}>
          {__(
            "Select an image from the Media Library. External image URLs are not used.",
            "hero-section"
          )}
        </p>
      </div>
    </PanelBody>
  );
};

export default General;
