import OnboardingLayout from '../layout/OnboardingLayout';
import { NavLink } from 'react-router-dom';

const serverConfigs = [
    {
        name: 'Apache (.htaccess)',
        snippet: `RewriteEngine On
RewriteCond %{HTTP:Authorization} ^(.*)
RewriteRule ^(.*) - [E=HTTP_AUTHORIZATION:%1]`,
    },
    {
        name: 'NGINX',
        snippet: `location / {
    try_files $uri $uri/ /index.php?$args;
    if ($http_authorization != "") {
        set $auth_header $http_authorization;
    }
    fastcgi_param HTTP_AUTHORIZATION $auth_header;
}`,
    },
    {
        name: 'LiteSpeed',
        snippet: `RewriteEngine On
RewriteRule .* - [E=HTTP_AUTHORIZATION:%{HTTP:Authorization}]`,
    },
    {
        name: 'WP Engine',
        snippet: `SetEnvIf Authorization "(.*)" HTTP_AUTHORIZATION=$1`,
    },
];

const onboardingSteps = [
    {
        title: '1. Add constants to wp-config.php',
        description: 'Define the secret key and optional CORS toggle as early as possible.',
        code: `define( 'headlesskey_SECRET_KEY', 'your-64-char-secret' );
define( 'headlesskey_CORS', true );`,
    },
    {
        title: '2. Flush permalinks',
        description: 'Visit Settings → Permalinks or click the "Flush Permalinks" admin-bar shortcut to register /headlesskey/v1 routes.',
    },
    {
        title: '3. Test the token endpoint',
        description: 'Send POST /headlesskey/v1/token with username + password. A valid JSON response confirms the core features are live.',
    },
];

const Onboarding = () => {
    return (
        <OnboardingLayout>
            <div className="mb-6 text-center">
                <NavLink
                    to="/dashboard"
                    className="text-gray-400 hover:text-gray-800 inline-flex items-center px-1 pt-1 text-base shadow-none"
                >
                    Skip to Dashboard
                </NavLink>
            </div>
            <div className="bg-white rounded p-8 space-y-10">
                <div className="text-center">
                    <p className="text-base font-semibold text-gray-600 uppercase tracking-wide">Quick Start</p>
                    <h1 className="text-4xl font-bold text-gray-900 mt-2">Configure HeadlessKey – JWT Auth in Minutes</h1>
                    <p className="mt-3 text-gray-600 text-lg">Secure JWT auth, premium security guardrails, and complete API docs—ready on day one.</p>
                </div>

                <div className="grid gap-6 md:grid-cols-3">
                    {onboardingSteps.map((step) => (
                        <div key={step.title} className="border border-gray-200 rounded-lg p-5">
                            <h3 className="text-lg font-semibold text-gray-900">{step.title}</h3>
                            <p className="text-sm text-gray-600 mt-2">{step.description}</p>
                            {step.code && (
                                <pre className="mt-3 bg-gray-50 rounded-md p-3 text-xs text-left text-gray-800 whitespace-pre-wrap">
                                    {step.code}
                                </pre>
                            )}
                        </div>
                    ))}
                                    </div>

                <div>
                    <h2 className="text-2xl font-bold text-gray-900">Authorization Header Compatibility</h2>
                    <p className="text-gray-600 mt-2">Add the matching snippet to ensure the <code>Authorization</code> header reaches WordPress.</p>
                    <div className="mt-4 grid gap-4 md:grid-cols-2">
                        {serverConfigs.map((server) => (
                            <div key={server.name} className="border border-gray-200 rounded-lg p-4">
                                <p className="text-sm font-semibold text-gray-900">{server.name}</p>
                                <pre className="mt-2 bg-gray-50 rounded-md p-3 text-xs text-gray-800 whitespace-pre-wrap">
                                    {server.snippet}
                                </pre>
                            </div>
                        ))}
                    </div>
                </div>

                <div className="border border-gray-200 rounded-lg p-6">
                    <h2 className="text-2xl font-bold text-gray-900">What’s Next?</h2>
                    <ul className="mt-4 space-y-2 text-gray-600">
                        <li>• Configure tokens, algorithms, and CORS under Settings → General.</li>
                        <li>• Enable brute-force protection, IP rules, and device limits inside Security Options.</li>
                        <li>• Monitor logins from Token Logs, Activity Logs, and the Live Monitor dashboard.</li>
                        <li>• Share the API documentation tab with developers integrating the endpoints.</li>
                    </ul>
                </div>
            </div>
        </OnboardingLayout>
    );
};

export default Onboarding;