import { useEffect, useState } from '@wordpress/element';
import Layout from '../layout/Layout';
import { getActivityLogs } from '../../api/local';
import { showPromiseToast } from '../../utils';

const ActivityLogs = () => {
    const [logs, setLogs] = useState([]);
    const [loading, setLoading] = useState(false);

    const loadLogs = () => {
        setLoading(true);
        const res = getActivityLogs({ per_page: 100 }).then((data) => {
            setLogs(data?.logs || []);
            setLoading(false);
        });
        showPromiseToast(res, 'Loading activity logs');
    };

    useEffect(() => {
        loadLogs();
    }, []);

    return (
        <Layout>
            <div className="bg-white rounded p-6">
                <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
                    <div>
                        <h2 className="text-2xl font-bold text-gray-900">Activity Logs</h2>
                        <p className="text-gray-600 text-sm mt-1">Trace password resets, failed logins, and security events.</p>
                    </div>
                    <button
                        type="button"
                        className="rounded-md bg-gray-800 px-3 py-2 text-sm font-semibold text-white shadow-sm hover:bg-gray-900"
                        onClick={loadLogs}
                    >
                        Refresh
                    </button>
                </div>

                <div className="mt-6 overflow-x-auto">
                    <table className="min-w-full divide-y divide-gray-200">
                        <thead className="bg-gray-50">
                        <tr>
                            <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Event</th>
                            <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">User</th>
                            <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">IP</th>
                            <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Details</th>
                            <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Date</th>
                        </tr>
                        </thead>
                        <tbody className="divide-y divide-gray-200 bg-white">
                        {loading && (
                            <tr>
                                <td colSpan="5" className="px-3 py-6 text-center text-sm text-gray-500">
                                    Loading activity logs...
                                </td>
                            </tr>
                        )}
                        {!loading && logs.map((log) => (
                            <tr key={log.id}>
                                <td className="px-3 py-3 text-sm font-medium text-gray-900">{log.event}</td>
                                <td className="px-3 py-3 text-sm text-gray-500">#{log.user_id || '—'}</td>
                                <td className="px-3 py-3 text-sm text-gray-500">{log.ip_address || '—'}</td>
                                <td className="px-3 py-3 text-sm text-gray-500">
                                    {log.metadata && <pre className="text-xs text-gray-500 whitespace-pre-wrap">{JSON.stringify(log.metadata, null, 2)}</pre>}
                                </td>
                                <td className="px-3 py-3 text-sm text-gray-500">{log.created_at}</td>
                            </tr>
                        ))}
                        {!loading && !logs.length && (
                            <tr>
                                <td colSpan="5" className="px-3 py-6 text-center text-sm text-gray-500">
                                    No activity logged yet.
                                </td>
                            </tr>
                        )}
                        </tbody>
                    </table>
                </div>
            </div>
        </Layout>
    );
};

export default ActivityLogs;


