<?php
/**
 * Primary class file for the headlesskey.
 *
 * @package headlesskey
 */

namespace headlesskey;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use headlesskey\Core\Installer;
use headlesskey\Core\Options;
use headlesskey\API\OptionsAPI;
use headlesskey\API\AuthAPI;
use headlesskey\API\AdminToolsAPI;
use headlesskey\Blocks\RegisterBlocks;
use headlesskey\Admin\RegisterAdmin;

/**
 * Class Plugin
 */
class Plugin {
	/**
	 * Options manager.
	 *
	 * @var Options
	 */
	public $options_manager;

	/**
	 * Options API manager.
	 *
	 * @var OptionsAPI
	 */
	public $options_api_manager;

	/**
	 * Blocks manager.
	 *
	 * @var RegisterBlocks
	 */
	public $blocks_manager;

	/**
	 * Admin Manager.
	 *
	 * @var RegisterAdmin;
	 */
	public $admin_manager;

	/**
	 * Auth API manager.
	 *
	 * @var AuthAPI
	 */
	public $auth_api_manager;

	/**
	 * Admin tools API.
	 *
	 * @var AdminToolsAPI
	 */
	public $admin_tools_api;

	/**
	 * Constructor.
	 */
	public function __construct() {
		// Get options manager instance.
		$this->options_manager = Options::get_instance();

		// Register APIs.
		$this->options_api_manager = new OptionsAPI();
		$this->auth_api_manager    = new AuthAPI();

		// Register Blocks.
		$this->blocks_manager = new RegisterBlocks();

		// Register Admin.
		$this->admin_manager = new RegisterAdmin();
		$this->admin_tools_api = new AdminToolsAPI();

		$this->register_hooks();
	}

	/**
	 * Registers core hooks.
	 */
	public function register_hooks() {
		/**
		 * Add "Dashboard" link to plugins page.
		 */
		add_filter(
			'plugin_action_links_' . headlesskey_ROOT_FILE_RELATIVE_PATH,
			array( $this, 'action_links' )
		);
	}

	/**
	 * Registers plugin action links.
	 *
	 * @param array $actions A list of actions for the plugin.
	 * @return array
	 */
	public function action_links( $actions ) {
		$settings_link = '<a href="' . esc_url( admin_url( 'admin.php?page=' . headlesskey_SLUG ) ) . '">' . __( 'Dashboard', 'headlesskey-jwt-auth' ) . '</a>';
		array_unshift( $actions, $settings_link );

		return $actions;
	}

	/**
	 * Plugin deactivation hook.
	 *
	 * @access public
	 * @static
	 */
	public static function plugin_activation() {
		// Clear the permalinks in case any new post types has been registered.
		flush_rewrite_rules();
		Installer::activate();
	}

	/**
	 * Plugin deactivation hook.
	 *
	 * @access public
	 * @static
	 */
	public static function plugin_deactivation() {
	}
}
