<?php
/**
 * JWT token service.
 *
 * @package headlesskey
 */

namespace headlesskey\JWT;

use Firebase\JWT\JWT;
use Firebase\JWT\Key;
use headlesskey\Core\Settings;

if (!defined('ABSPATH')) {
	exit;
}

/**
 * Class TokenService
 */
class TokenService
{
	/**
	 * Issue a token.
	 *
	 * @param array $payload Token payload.
	 *
	 * @return string
	 * @throws \Exception When key/algorithm invalid.
	 */
	public function issue(array $payload)
	{
		$algorithm = $this->get_algorithm();
		$key = $this->get_private_key($algorithm);

		$payload = apply_filters('headlesskey_token_before_sign', $payload);

		return JWT::encode($payload, $key, $algorithm);
	}

	/**
	 * Validate a token.
	 *
	 * @param string $token Token string.
	 *
	 * @return object
	 */
	public function decode($token)
	{
		$algorithm = $this->get_algorithm();
		$key = $this->get_public_key($algorithm);

		return JWT::decode($token, new Key($key, $algorithm));
	}

	/**
	 * Get configured algorithm.
	 *
	 * @return string
	 */
	protected function get_algorithm()
	{
		$algorithm = Settings::get('active_algorithm', 'HS256');

		return apply_filters('headlesskey_algorithm', $algorithm);
	}

	/**
	 * Resolve the shared/secret key.
	 *
	 * @param string $algorithm Algorithm slug.
	 *
	 * @return string
	 * @throws \Exception When key missing.
	 */
	protected function get_private_key($algorithm)
	{
		switch (strtoupper($algorithm)) {
			case 'HS256':
				$key = $this->resolve_secret_key();
				break;
			case 'RS256':
				$key = Settings::get('rs256_private_key', '');
				break;
			case 'ES256':
				$key = Settings::get('es256_private_key', '');
				break;
			default:
				$key = $this->resolve_secret_key();
		}

		if (empty($key)) {
				throw new \Exception(esc_html__('JWT signing key is missing.', 'headlesskey-jwt-auth'));
		}

		return $key;
	}

	/**
	 * Resolve verification key.
	 *
	 * @param string $algorithm Algorithm slug.
	 *
	 * @return string
	 * @throws \Exception When key missing.
	 */
	protected function get_public_key($algorithm)
	{
		switch (strtoupper($algorithm)) {
			case 'HS256':
				$key = $this->resolve_secret_key();
				break;
			case 'RS256':
				$key = Settings::get('rs256_public_key', '');
				break;
			case 'ES256':
				$key = Settings::get('es256_public_key', '');
				break;
			default:
				$key = $this->resolve_secret_key();
		}

		if (empty($key)) {
				throw new \Exception(esc_html__('JWT verification key is missing.', 'headlesskey-jwt-auth'));
		}

		return $key;
	}

	/**
	 * Figure out secret key order: wp-config constant -> settings -> fallback.
	 *
	 * @return string
	 */
	protected function resolve_secret_key()
	{
		if (defined('headlesskey_SECRET_KEY') && headlesskey_SECRET_KEY) {
			return headlesskey_SECRET_KEY;
		}

		$key = Settings::get('secret_key', '');

		if (empty($key)) {
			$key = wp_generate_password(64, true, true);
			Settings::set('secret_key', $key);
		}

		return $key;
	}
}


