<?php
/**
 * Centralized plugin settings helper.
 *
 * @package headlesskey
 */

namespace headlesskey\Core;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Settings
 */
class Settings {
	/**
	 * Get all settings merged with defaults.
	 *
	 * @return array
	 */
	public static function all() {
		return wp_parse_args( Options::get_instance()->get(), self::defaults() );
	}

	/**
	 * Get a single setting with fallback to defaults.
	 *
	 * @param string $key Setting key.
	 * @param mixed  $default Default value.
	 *
	 * @return mixed
	 */
	public static function get( $key, $default = null ) {
		$settings      = self::all();
		$default_value = array_key_exists( $key, self::defaults() ) ? self::defaults()[ $key ] : $default;

		return array_key_exists( $key, $settings ) ? $settings[ $key ] : $default_value;
	}

	/**
	 * Update a single setting.
	 *
	 * @param string $key Setting key.
	 * @param mixed  $value Setting value.
	 *
	 * @return void
	 */
	public static function set( $key, $value ) {
		Options::get_instance()->set( $key, $value );
	}

	/**
	 * Default plugin settings.
	 *
	 * @return array
	 */
	public static function defaults() {
		return array(
			'secret_key'                 => '',
			'cors_enabled'               => (bool) headlesskey_CORS,
			'cors_allowed_origins'       => array( '*' ),
			'token_expiration'           => 3600,
			'allowed_algorithms'         => array( 'HS256', 'RS256', 'ES256' ),
			'active_algorithm'           => 'HS256',
			'rs256_private_key'          => '',
			'rs256_public_key'           => '',
			'es256_private_key'          => '',
			'es256_public_key'           => '',
			'allow_token_refresh'        => true,
			'allow_token_revoke'         => true,
			'allow_registration'         => true,
			'auto_login_after_register'  => true,
			'allow_registration_via_api' => true,
			'allowed_roles'              => array( 'administrator' ),
			'rbac_rules'                 => array(
				'token'           => array( 'mode' => 'public' ),
				'token_refresh'   => array( 'mode' => 'authenticated' ),
				'token_revoke'    => array( 'mode' => 'authenticated' ),
				'register'        => array( 'mode' => 'public' ),
				'login'           => array( 'mode' => 'public' ),
				'forgot_password' => array( 'mode' => 'public' ),
				'reset_password'  => array( 'mode' => 'public' ),
				'change_password' => array( 'mode' => 'authenticated' ),
				'sso_exchange'    => array( 'mode' => 'public' ),
			),
			'allow_token_logs'           => true,
			'brute_force_max_attempts'   => 5,
			'brute_force_lockout'        => 15, // Minutes.
			'ip_whitelist'               => array(),
			'ip_blacklist'               => array(),
			'device_limit_enabled'       => false,
			'device_limit_max'           => 3,
			'live_monitor_enabled'       => true,
			'otp_expiration'             => 900,
			'login_response_fields'      => array( 'ID', 'user_login', 'user_email', 'display_name', 'roles' ),
			'token_meta_fields'          => array(),
			'token_acf_fields'           => array(),
			'token_woo_fields'           => array(),
			'allow_multi_token_sessions' => true,
			'max_user_tokens'            => 5,
			'security_webhooks'          => array(),
			'webhook_secret'             => '',
			'webhook_retry_attempts'     => 3,
			'sso_enable'                 => false,
			'sso_connections'            => array(),
			'analytics_heatmap_days'     => 14,
		);
	}
}

