<?php
/**
 * Handles plugin installation tasks such as database tables and storage.
 *
 * @package headlesskey
 */

namespace headlesskey\Core;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Installer
 */
class Installer {
	/**
	 * Database version.
	 */
	const DB_VERSION = '1.0.0';

	/**
	 * Run on plugin activation.
	 *
	 * @return void
	 */
	public static function activate() {
		self::maybe_create_tables();
		self::maybe_setup_logs_directory();
		update_option( 'headlesskey_db_version', self::DB_VERSION );
	}

	/**
	 * Create plugin specific tables.
	 *
	 * @return void
	 */
	protected static function maybe_create_tables() {
		global $wpdb;

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		$charset_collate     = $wpdb->get_charset_collate();
		$token_logs_table    = self::get_table_name( 'token_logs' );
		$activity_logs_table = self::get_table_name( 'activity_logs' );

		$token_logs_sql = "CREATE TABLE {$token_logs_table} (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			user_id bigint(20) unsigned NOT NULL DEFAULT 0,
			token_jti varchar(64) NOT NULL,
			token_hash char(64) NOT NULL,
			status varchar(20) NOT NULL DEFAULT 'active',
			issued_at datetime NOT NULL,
			expires_at datetime NOT NULL,
			ip_address varchar(45) DEFAULT NULL,
			device varchar(255) DEFAULT NULL,
			user_agent text DEFAULT NULL,
			error_message text DEFAULT NULL,
			meta longtext DEFAULT NULL,
			created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
			updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			PRIMARY KEY  (id),
			KEY token_hash (token_hash),
			KEY user_id (user_id),
			KEY token_jti (token_jti)
		) {$charset_collate};";

		$activity_logs_sql = "CREATE TABLE {$activity_logs_table} (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			user_id bigint(20) unsigned NOT NULL DEFAULT 0,
			event varchar(60) NOT NULL,
			context varchar(60) DEFAULT NULL,
			ip_address varchar(45) DEFAULT NULL,
			metadata longtext DEFAULT NULL,
			created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY  (id),
			KEY user_id (user_id),
			KEY event (event)
		) {$charset_collate};";

		dbDelta( $token_logs_sql );
		dbDelta( $activity_logs_sql );
	}

	/**
	 * Ensure logs directory and revoke file exist.
	 *
	 * @return void
	 */
	protected static function maybe_setup_logs_directory() {
		$upload_dir = wp_upload_dir();
		$base_dir   = $upload_dir['basedir'] . '/' . headlesskey_SLUG;

		if ( ! file_exists( $base_dir ) ) {
			wp_mkdir_p( $base_dir );
		}

		// Protect the directory.
		$htaccess_file = trailingslashit( $base_dir ) . '.htaccess';
		if ( ! file_exists( $htaccess_file ) ) {
			$rules = 'Order Deny,Allow' . PHP_EOL . 'Deny from all';
			file_put_contents( $htaccess_file, $rules ); // phpcs:ignore
		}

		$index_file = trailingslashit( $base_dir ) . 'index.php';
		if ( ! file_exists( $index_file ) ) {
			file_put_contents( $index_file, "<?php\n// Silence is golden.\n" ); // phpcs:ignore
		}

		$new_revoked_file = trailingslashit( $base_dir ) . 'revoked.json';
		$old_revoked_file = trailingslashit( headlesskey_DIR ) . 'logs/revoked.json';

		// Migrate existing data if available.
		if ( file_exists( $old_revoked_file ) && ! file_exists( $new_revoked_file ) ) {
			copy( $old_revoked_file, $new_revoked_file );
		}

		if ( ! file_exists( $new_revoked_file ) ) {
			file_put_contents( $new_revoked_file, wp_json_encode( array() ) ); // phpcs:ignore
		}
	}

	/**
	 * Helper to get table names.
	 *
	 * @param string $table Table suffix.
	 *
	 * @return string
	 */
	public static function get_table_name( $table ) {
		global $wpdb;

		$map = array(
			'token_logs'    => "{$wpdb->prefix}headlesskey_token_logs",
			'activity_logs' => "{$wpdb->prefix}headlesskey_activity_logs",
		);

		return isset( $map[ $table ] ) ? $map[ $table ] : "{$wpdb->prefix}{$table}";
	}
}


