<?php
/**
 * Data access for token logs.
 *
 * @package headlesskey
 */

namespace headlesskey\Auth;

use headlesskey\Core\Installer;

if (!defined('ABSPATH')) {
	exit;
}

/**
 * Class TokenLogRepository
 */
class TokenLogRepository
{
	/**
	 * Insert new log entry.
	 *
	 * @param array $data Data payload.
	 *
	 * @return int|false
	 */
	public function insert($data)
	{
		global $wpdb;

		$table = Installer::get_table_name('token_logs');

		$defaults = array(
			'user_id' => 0,
			'token_jti' => '',
			'token_hash' => '',
			'status' => 'active',
			'issued_at' => current_time('mysql', true),
			'expires_at' => current_time('mysql', true),
			'ip_address' => '',
			'device' => '',
			'user_agent' => '',
			'error_message' => '',
			'meta' => '',
		);

		$payload = wp_parse_args($data, $defaults);

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		$result = $wpdb->insert(
			$table,
			$payload,
			array(
				'%d',
				'%s',
				'%s',
				'%s',
				'%s',
				'%s',
				'%s',
				'%s',
				'%s',
				'%s',
				'%s',
			)
		);

		return $result ? $wpdb->insert_id : false;
	}

	/**
	 * Update a log by token hash.
	 *
	 * @param string $token_hash Token hash.
	 * @param array  $data Data.
	 *
	 * @return bool
	 */
	public function update_by_hash($token_hash, $data)
	{
		global $wpdb;

		$table = Installer::get_table_name('token_logs');

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		return false !== $wpdb->update(
			$table,
			$data,
			array(
				'token_hash' => $token_hash,
			)
		);
	}

	/**
	 * Fetch logs with optional filters.
	 *
	 * @param array $args Query args.
	 *
	 * @return array
	 */
	public function get_logs($args = array())
	{
		global $wpdb;

		$table = Installer::get_table_name('token_logs');

		$where = array();
		$prepare = array();

		if (isset($args['user_id'])) {
			$where[] = 'user_id = %d';
			$prepare[] = $args['user_id'];
		}

		if (isset($args['status'])) {
			$where[] = 'status = %s';
			$prepare[] = $args['status'];
		}

		$where_sql = empty($where) ? '1=1' : implode(' AND ', $where);
		$limit = isset($args['limit']) ? absint($args['limit']) : 50;
		$offset = isset($args['offset']) ? absint($args['offset']) : 0;

		$sql = "SELECT * FROM {$table} WHERE {$where_sql} ORDER BY id DESC LIMIT %d OFFSET %d";

		$prepare[] = $limit;
		$prepare[] = $offset;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
		return $wpdb->get_results($wpdb->prepare($sql, $prepare), ARRAY_A);
	}

	/**
	 * Count active tokens per user + device.
	 *
	 * @param int    $user_id User ID.
	 * @param string $device_key Device key.
	 *
	 * @return int
	 */
	public function count_active_tokens_for_device($user_id, $device_key)
	{
		global $wpdb;

		$table = Installer::get_table_name('token_logs');

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
		return (int) $wpdb->get_var($wpdb->prepare(
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			"SELECT COUNT(*) FROM {$table} WHERE user_id = %d AND device = %s AND status = %s",
			$user_id,
			$device_key,
			'active'
		));
	}

	/**
	 * Count active tokens per user.
	 *
	 * @param int $user_id User ID.
	 *
	 * @return int
	 */
	public function count_active_tokens_for_user($user_id)
	{
		global $wpdb;

		$table = Installer::get_table_name('token_logs');

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
		return (int) $wpdb->get_var($wpdb->prepare(
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			"SELECT COUNT(*) FROM {$table} WHERE user_id = %d AND status = %s",
			$user_id,
			'active'
		));
	}

	/**
	 * Fetch hashes for oldest active tokens exceeding limit.
	 *
	 * @param int $user_id User ID.
	 * @param int $limit   Max allowed tokens.
	 *
	 * @return array
	 */
	public function get_token_hashes_beyond_limit($user_id, $limit)
	{
		global $wpdb;

		if ($limit < 1) {
			$limit = 1;
		}

		$table = Installer::get_table_name('token_logs');

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
		return $wpdb->get_col($wpdb->prepare(
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			"SELECT token_hash FROM {$table} WHERE user_id = %d AND status = %s ORDER BY issued_at DESC LIMIT 18446744073709551615 OFFSET %d",
			$user_id,
			'active',
			$limit
		));
	}

	/**
	 * Update status for multiple hashes.
	 *
	 * @param array  $hashes Token hashes.
	 * @param string $status Status.
	 *
	 * @return void
	 */
	public function mark_status_for_hashes($hashes, $status = 'revoked')
	{
		if (empty($hashes)) {
			return;
		}

		global $wpdb;
		$table = Installer::get_table_name('token_logs');

		$placeholders = implode(',', array_fill(0, count($hashes), '%s'));
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
		$wpdb->query($wpdb->prepare(
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			"UPDATE {$table} SET status = %s WHERE token_hash IN ( {$placeholders} )",
			array_merge(array($status), $hashes)
		));
	}
}
