<?php
/**
 * Token revocation storage handler.
 *
 * @package headlesskey
 */

namespace headlesskey\Auth;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class RevocationStore
 */
class RevocationStore {
	/**
	 * Revoked tokens file.
	 *
	 * @var string
	 */
	protected $file;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$upload_dir = wp_upload_dir();
		$this->file = $upload_dir['basedir'] . '/' . headlesskey_SLUG . '/revoked.json';
	}

	/**
	 * Revoke a token by its hash.
	 *
	 * @param string $token Token string.
	 * @param array  $context Additional context data.
	 *
	 * @return bool
	 */
	public function revoke( $token, $context = array() ) {
		$data       = $this->all();
		$token_hash = $this->hash_token( $token );

		$data[ $token_hash ] = array_merge(
			array(
				'revoked_at' => current_time( 'mysql', true ),
			),
			$context
		);

		return $this->write( $data );
	}

	/**
	 * Revoke by providing a hashed token (typically from logs).
	 *
	 * @param string $token_hash Token hash (sha256).
	 * @param array  $context Context data.
	 *
	 * @return bool
	 */
	public function revoke_by_hash( $token_hash, $context = array() ) {
		$data                 = $this->all();
		$data[ $token_hash ]  = array_merge(
			array(
				'revoked_at' => current_time( 'mysql', true ),
			),
			$context
		);

		return $this->write( $data );
	}

	/**
	 * Determine whether the token has been revoked.
	 *
	 * @param string $token Token string.
	 *
	 * @return bool
	 */
	public function is_revoked( $token ) {
		$data       = $this->all();
		$token_hash = $this->hash_token( $token );

		return array_key_exists( $token_hash, $data );
	}

	/**
	 * Remove revoked tokens older than provided seconds.
	 *
	 * @param int $seconds Seconds.
	 *
	 * @return void
	 */
	public function prune( $seconds = WEEK_IN_SECONDS ) {
		$data    = $this->all();
		$changed = false;

		foreach ( $data as $hash => $details ) {
			$revoked_at = isset( $details['revoked_at'] ) ? strtotime( $details['revoked_at'] ) : 0;
			if ( $revoked_at && $revoked_at < ( time() - $seconds ) ) {
				unset( $data[ $hash ] );
				$changed = true;
			}
		}

		if ( $changed ) {
			$this->write( $data );
		}
	}

	/**
	 * Return all revoked tokens.
	 *
	 * @return array
	 */
	public function all() {
		if ( ! file_exists( $this->file ) ) {
			return array();
		}

		$content = file_get_contents( $this->file ); // phpcs:ignore
		$data    = json_decode( $content, true );

		return is_array( $data ) ? $data : array();
	}

	/**
	 * Persist tokens to file.
	 *
	 * @param array $data Data.
	 *
	 * @return bool
	 */
	protected function write( $data ) {
		$dir = dirname( $this->file );
		if ( ! file_exists( $dir ) ) {
			wp_mkdir_p( $dir );
		}

		return false !== file_put_contents( $this->file, wp_json_encode( $data, JSON_PRETTY_PRINT ) ); // phpcs:ignore
	}

	/**
	 * Creates hash for stored tokens.
	 *
	 * @param string $token Raw token.
	 *
	 * @return string
	 */
	protected function hash_token( $token ) {
		return hash( 'sha256', $token );
	}
}

