<?php
/**
 * Handles device-based token limiting.
 *
 * @package headlesskey
 */

namespace headlesskey\Auth;

use headlesskey\Core\Settings;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class DeviceManager
 */
class DeviceManager {
	/**
	 * Token logs repository.
	 *
	 * @var TokenLogRepository
	 */
	protected $logs;

	/**
	 * Constructor.
	 *
	 * @param TokenLogRepository $logs Logs repository.
	 */
	public function __construct( TokenLogRepository $logs ) {
		$this->logs = $logs;
	}

	/**
	 * Build device key fingerprint.
	 *
	 * @param string $user_agent User agent string.
	 * @param string $ip_address IP address.
	 *
	 * @return string
	 */
	public function fingerprint( $user_agent, $ip_address ) {
		return substr( hash( 'sha256', $user_agent . '|' . $ip_address ), 0, 64 );
	}

	/**
	 * Determine whether issuing a new token is allowed for this device.
	 *
	 * @param int    $user_id User ID.
	 * @param string $device_key Device identifier.
	 *
	 * @return bool
	 */
	public function can_issue( $user_id, $device_key ) {
		if ( ! Settings::get( 'device_limit_enabled', false ) ) {
			return true;
		}

		$max_devices = (int) Settings::get( 'device_limit_max', 3 );

		if ( $max_devices <= 0 ) {
			return true;
		}

		$active = $this->logs->count_active_tokens_for_device( $user_id, $device_key );

		return $active < $max_devices;
	}
}


