<?php
/**
 * Brute force protection helper.
 *
 * @package headlesskey
 */

namespace headlesskey\Auth;

use headlesskey\Core\Settings;

if (!defined('ABSPATH')) {
	exit;
}

/**
 * Class BruteForceGuard
 */
class BruteForceGuard
{
	/**
	 * Determine whether the current identifier is locked.
	 *
	 * @param string $identifier Identifier (username/email/ip).
	 *
	 * @return bool
	 */
	public function is_locked($identifier)
	{
		$key = $this->get_key($identifier);
		return (bool) get_transient($key . '_locked');
	}

	/**
	 * Register a failed attempt.
	 *
	 * @param string $identifier Identifier (username/email/ip).
	 *
	 * @return void
	 */
	public function add_failure($identifier)
	{
		$key = $this->get_key($identifier);
		$attempts = (int) get_transient($key);
		$attempts++;
		$max = (int) Settings::get('brute_force_max_attempts', 5);

		set_transient($key, $attempts, HOUR_IN_SECONDS);

		if ($attempts >= $max) {
			$lock_key = $key . '_locked';
			$timeout = (int) Settings::get('brute_force_lockout', 15) * MINUTE_IN_SECONDS;
			set_transient($lock_key, true, $timeout);
			do_action('headlesskey_bruteforce_lock', $identifier, $attempts);
		}
	}

	/**
	 * Reset failure attempts.
	 *
	 * @param string $identifier Identifier.
	 *
	 * @return void
	 */
	public function reset($identifier)
	{
		$key = $this->get_key($identifier);
		delete_transient($key);
		delete_transient($key . '_locked');
	}

	/**
	 * Build transient key.
	 *
	 * @param string $identifier Identifier.
	 *
	 * @return string
	 */
	protected function get_key($identifier)
	{
		return 'headlesskey_brute_' . md5(strtolower($identifier) . '_' . sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'] ?? '')));
	}
}


