<?php
/**
 * Handles outbound webhook notifications.
 *
 * @package headlesskey
 */

namespace headlesskey\Alerts;

use headlesskey\Core\Settings;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class WebhookDispatcher
 */
class WebhookDispatcher {
	/**
	 * Dispatch alert to registered webhook endpoints.
	 *
	 * @param string $event Event key.
	 * @param array  $payload Event payload.
	 *
	 * @return void
	 */
	public function dispatch( $event, $payload = array() ) {
		$webhooks = (array) Settings::get( 'security_webhooks', array() );

		if ( empty( $webhooks ) ) {
			return;
		}

		$payload = array(
			'event'     => $event,
			'payload'   => $payload,
			'site_url'  => home_url(),
			'timestamp' => current_time( 'mysql', true ),
		);

		foreach ( $webhooks as $webhook ) {
			$events = isset( $webhook['events'] ) ? (array) $webhook['events'] : array();
			$url    = isset( $webhook['url'] ) ? esc_url_raw( $webhook['url'] ) : '';

			if ( empty( $url ) ) {
				continue;
			}

			if ( ! empty( $events ) && ! in_array( $event, $events, true ) ) {
				continue;
			}

			$this->transmit( $url, $payload, $webhook );
		}
	}

	/**
	 * Transmit a payload with retry logic.
	 *
	 * @param string $url Endpoint URL.
	 * @param array  $payload Payload data.
	 * @param array  $webhook Webhook configuration.
	 *
	 * @return void
	 */
	protected function transmit( $url, $payload, $webhook ) {
		$retries = (int) Settings::get( 'webhook_retry_attempts', 3 );
		$secret  = isset( $webhook['secret'] ) && $webhook['secret'] ? $webhook['secret'] : Settings::get( 'webhook_secret', '' );

		$body      = wp_json_encode( $payload );
		$signature = $secret ? hash_hmac( 'sha256', $body, $secret ) : '';

		for ( $attempt = 0; $attempt < max( 1, $retries ); $attempt ++ ) {
			$response = wp_remote_post(
				$url,
				array(
					'body'        => $body,
					'headers'     => array(
						'Content-Type'            => 'application/json',
						'X-headlesskey-Signature'   => $signature,
						'X-headlesskey-Attempt'     => $attempt + 1,
						'X-headlesskey-Event'       => $payload['event'],
					),
					'timeout'     => 10,
					'data_format' => 'body',
				)
			);

			if ( ! is_wp_error( $response ) && wp_remote_retrieve_response_code( $response ) < 400 ) {
				break;
			}
		}
	}
}

