<?php
/**
 * Authentication REST API.
 *
 * @package headlesskey
 */

namespace headlesskey\API;

use Firebase\JWT\JWT;
use Firebase\JWT\Key;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;
use WP_User;
use headlesskey\Alerts\WebhookDispatcher;
use headlesskey\Auth\BruteForceGuard;
use headlesskey\Auth\DeviceManager;
use headlesskey\Auth\RevocationStore;
use headlesskey\Auth\RoleAccessManager;
use headlesskey\Auth\TokenLogRepository;
use headlesskey\Core\Installer;
use headlesskey\Core\Settings;
use headlesskey\JWT\TokenService;

if (!defined('ABSPATH')) {
	exit;
}

/**
 * Class AuthAPI
 */
class AuthAPI
{
	/**
	 * Token service.
	 *
	 * @var TokenService
	 */
	protected $tokens;

	/**
	 * Revocation store.
	 *
	 * @var RevocationStore
	 */
	protected $revoked;

	/**
	 * Token logs.
	 *
	 * @var TokenLogRepository
	 */
	protected $logs;

	/**
	 * Role manager.
	 *
	 * @var RoleAccessManager
	 */
	protected $roles;

	/**
	 * Brute force guard.
	 *
	 * @var BruteForceGuard
	 */
	protected $guard;

	/**
	 * Device manager.
	 *
	 * @var DeviceManager
	 */
	protected $devices;

	/**
	 * Webhook dispatcher.
	 *
	 * @var WebhookDispatcher
	 */
	protected $alerts;

	/**
	 * REST namespace.
	 *
	 * @var string
	 */
	protected $namespace;

	/**
	 * Constructor.
	 */
	public function __construct()
	{
		$this->tokens = new TokenService();
		$this->revoked = new RevocationStore();
		$this->logs = new TokenLogRepository();
		$this->roles = new RoleAccessManager();
		$this->guard = new BruteForceGuard();
		$this->devices = new DeviceManager($this->logs);
		$this->alerts = new WebhookDispatcher();

		$this->namespace = trailingslashit(headlesskey_REST_NAMESPACE) . headlesskey_REST_VERSION;

		add_action('rest_api_init', array($this, 'register_routes'));
		add_action('rest_api_init', array($this, 'register_cors_handler'), 15);
		add_action('headlesskey_bruteforce_lock', array($this, 'handle_bruteforce_lock'), 10, 2);
	}

	/**
	 * Register API routes.
	 *
	 * @return void
	 */
	public function register_routes()
	{
		$routes = array(
			array(
				'route' => '/token',
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => 'generate_token',
				'endpoint' => 'token',
				'permission' => '__return_true',
			),
			array(
				'route' => '/token/validate',
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => 'validate_token',
				'endpoint' => 'token_validate',
				'permission' => '__return_true',
			),
			array(
				'route' => '/token/refresh',
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => 'refresh_token',
				'endpoint' => 'token_refresh',
				'permission' => '__return_true',
			),
			array(
				'route' => '/token/revoke',
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => 'revoke_token',
				'endpoint' => 'token_revoke',
				'permission' => 'permissions_check_authorized',
			),
			array(
				'route' => '/register',
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => 'register_user',
				'endpoint' => 'register',
				'permission' => '__return_true',
			),
			array(
				'route' => '/login',
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => 'login_user',
				'endpoint' => 'login',
				'permission' => '__return_true',
			),
			array(
				'route' => '/forgot-password',
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => 'forgot_password',
				'endpoint' => 'forgot_password',
				'permission' => '__return_true',
			),
			array(
				'route' => '/reset-password',
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => 'reset_password',
				'endpoint' => 'reset_password',
				'permission' => '__return_true',
			),
			array(
				'route' => '/change-password',
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => 'change_password',
				'endpoint' => 'change_password',
				'permission' => 'permissions_check_authorized',
			),
			array(
				'route' => '/sso/exchange',
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => 'sso_exchange',
				'endpoint' => 'sso_exchange',
				'permission' => '__return_true',
			),
		);

		foreach ($routes as $route) {
			$permission_callback = $route['permission'];
			if ( $permission_callback !== '__return_true' ) {
				$permission_callback = array( $this, $permission_callback );
			}

			register_rest_route(
				$this->namespace,
				$route['route'],
				array(
					'methods' => $route['methods'],
					'callback' => array($this, $route['callback']),
					'permission_callback' => $permission_callback,
					'args' => array(),
				)
			);
		}
	}

	/**
	 * Check if the user is authorized.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return true|WP_Error
	 */
	public function permissions_check_authorized( WP_REST_Request $request ) {
		$auth = $this->get_authenticated_user( $request );

		if ( $auth['user'] ) {
			return true;
		}

		return $this->error( 'unauthorized', __( 'Authorization required.', 'headlesskey-jwt-auth' ), 401 );
	}

	/**
	 * Token generation endpoint.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function generate_token(WP_REST_Request $request)
	{
		$username = sanitize_text_field($request->get_param('username'));
		$password = $request->get_param('password');
		$ip = $this->get_ip_address();

		if (empty($username) || empty($password)) {
			return $this->error('missing_credentials', __('Username and password are required.', 'headlesskey-jwt-auth'), 400);
		}

		if ($this->is_ip_blocked()) {
			return $this->error('ip_blocked', __('Access denied for this IP address.', 'headlesskey-jwt-auth'), 403);
		}

		if ($this->guard->is_locked($username)) {
			return $this->error('locked', __('Too many failed attempts. Please try again later.', 'headlesskey-jwt-auth'), 429);
		}

		$user = wp_authenticate($username, $password);

		if (is_wp_error($user)) {
			$this->guard->add_failure($username);
			$this->log_activity(
				'login_failed',
				array(
					'user_login' => $username,
					'message' => $user->get_error_message(),
				)
			);
			$this->send_security_alert(
				'login_failed',
				array(
					'username' => $username,
					'ip' => $ip,
				)
			);
			return $this->error('invalid_credentials', __('Invalid username or password.', 'headlesskey-jwt-auth'), 401);
		}

		$this->guard->reset($username);

		$access = $this->ensure_endpoint_access('token', $user);
		if (is_wp_error($access)) {
			return $access;
		}

		$user_agent = $request->get_header('user-agent');
		$device_key = $this->devices->fingerprint($user_agent, $ip);

		if (!$this->devices->can_issue($user->ID, $device_key)) {
			return $this->error('device_limit', __('Device limit reached for this user.', 'headlesskey-jwt-auth'), 429);
		}

		$expiration = (int) apply_filters('headlesskey_expire', Settings::get('token_expiration', 3600), $user);
		$issued_at = time();
		$expires_at = $issued_at + $expiration;
		$jti = wp_generate_uuid4();

		$payload = array(
			'iss' => home_url(),
			'iat' => $issued_at,
			'nbf' => $issued_at,
			'exp' => $expires_at,
			'jti' => $jti,
			'data' => $this->prepare_user_payload($user),
		);

		try {
			$token = $this->tokens->issue($payload);
		} catch (\Exception $exception) {
			return $this->error('token_error', $exception->getMessage(), 500);
		}

		$response = array(
			'token' => $token,
			'expiration' => gmdate('c', $expires_at),
			'expires_in' => $expiration,
			'user' => $this->prepare_user_payload($user),
			'refreshable' => (bool) Settings::get('allow_token_refresh', true),
			'jti' => $jti,
		);

		$response = apply_filters('headlesskey_token_before_dispatch', $response, $user);

		$this->logs->insert(
			array(
				'user_id' => $user->ID,
				'token_jti' => $jti,
				'token_hash' => hash('sha256', $token),
				'issued_at' => gmdate('Y-m-d H:i:s', $issued_at),
				'expires_at' => gmdate('Y-m-d H:i:s', $expires_at),
				'ip_address' => $ip,
				'device' => $device_key,
				'user_agent' => $user_agent,
			)
		);

		$this->log_activity(
			'token_issued',
			array(
				'user_id' => $user->ID,
				'ip' => $ip,
			)
		);
		$this->send_security_alert(
			'login_success',
			array(
				'user_id' => $user->ID,
				'ip' => $ip,
				'device' => $device_key,
				'jti' => $jti,
			)
		);

		$this->enforce_multi_token_policy($user->ID);

		return new WP_REST_Response($response, 200);
	}

	/**
	 * Validate token endpoint.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_REST_Response
	 */
	public function validate_token(WP_REST_Request $request)
	{
		$token = $request->get_param('token');

		if (empty($token)) {
			return $this->error('missing_token', __('Token is required.', 'headlesskey-jwt-auth'), 400);
		}

		if ($this->revoked->is_revoked($token)) {
			return new WP_REST_Response(
				array(
					'valid' => false,
					'reason' => __('Token has been revoked.', 'headlesskey-jwt-auth'),
				),
				200
			);
		}

		try {
			$decoded = $this->tokens->decode($token);
		} catch (\Exception $exception) {
			$hash = hash('sha256', $token);
			$status = false !== stripos($exception->getMessage(), 'expired') ? 'expired' : 'error';

			$this->logs->update_by_hash(
				$hash,
				array(
					'status' => $status,
					'error_message' => $exception->getMessage(),
				)
			);

			return new WP_REST_Response(
				array(
					'valid' => false,
					'reason' => $exception->getMessage(),
				),
				200
			);
		}

		return new WP_REST_Response(
			array(
				'valid' => true,
				'data' => $decoded,
			),
			200
		);
	}

	/**
	 * Refresh token endpoint.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function refresh_token(WP_REST_Request $request)
	{
		if (!Settings::get('allow_token_refresh', true)) {
			return $this->error('refresh_disabled', __('Token refresh is disabled.', 'headlesskey-jwt-auth'), 403);
		}

		$token = $request->get_param('token');

		if (empty($token)) {
			return $this->error('missing_token', __('Token is required.', 'headlesskey-jwt-auth'), 400);
		}

		if ($this->revoked->is_revoked($token)) {
			return $this->error('revoked', __('Token has been revoked.', 'headlesskey-jwt-auth'), 409);
		}

		try {
			$decoded = $this->tokens->decode($token);
		} catch (\Exception $exception) {
			return $this->error('invalid_token', $exception->getMessage(), 401);
		}

		$user = get_user_by('ID', $decoded->data->ID ?? 0);

		if (!$user) {
			return $this->error('user_missing', __('User not found for this token.', 'headlesskey-jwt-auth'), 404);
		}

		$access = $this->ensure_endpoint_access('token_refresh', $user);
		if (is_wp_error($access)) {
			return $access;
		}

		$request->set_param('username', $user->user_login);
		$request->set_param('password', null);

		$new_token = $this->issue_new_token_from_refresh($user, $token);

		if (is_wp_error($new_token)) {
			return $new_token;
		}

		$this->send_security_alert(
			'token_refresh',
			array(
				'user_id' => $user->ID,
				'jti' => $new_token['jti'] ?? '',
			)
		);

		return new WP_REST_Response($new_token, 200);
	}

	/**
	 * Revoke token endpoint (logout).
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function revoke_token(WP_REST_Request $request)
	{
		if (!Settings::get('allow_token_revoke', true)) {
			return $this->error('revoke_disabled', __('Token revocation is disabled.', 'headlesskey-jwt-auth'), 403);
		}

		$token = $request->get_param('token');

		if (empty($token)) {
			return $this->error('missing_token', __('Token is required.', 'headlesskey-jwt-auth'), 400);
		}

		$actor = $this->get_authenticated_user($request);

		$access = $this->ensure_endpoint_access('token_revoke', $actor['user']);
		if (is_wp_error($access)) {
			return $access;
		}

		try {
			$decoded = $this->tokens->decode($token);
		} catch (\Exception $exception) {
			return $this->error('invalid_token', $exception->getMessage(), 401);
		}

		$token_user_id = isset($decoded->data->ID) ? (int) $decoded->data->ID : 0;

		if ($actor['user']) {
			if ($actor['user']->ID !== $token_user_id && !user_can($actor['user'], 'manage_options')) {
				return $this->error('access_denied', __('You cannot revoke this token.', 'headlesskey-jwt-auth'), 403);
			}
		}

		$this->revoked->revoke(
			$token,
			array(
				'user_id' => $token_user_id,
			)
		);

		$this->logs->update_by_hash(
			hash('sha256', $token),
			array(
				'status' => 'revoked',
			)
		);

		return new WP_REST_Response(
			array(
				'message' => __('Token revoked successfully.', 'headlesskey-jwt-auth'),
			),
			200
		);
	}

	/**
	 * Register endpoint handler.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function register_user(WP_REST_Request $request)
	{
		if (!Settings::get('allow_registration_via_api', true)) {
			return $this->error('registration_disabled', __('Registration via API is disabled.', 'headlesskey-jwt-auth'), 403);
		}

		$access = $this->ensure_endpoint_access('register', $this->get_authenticated_user($request)['user']);
		if (is_wp_error($access)) {
			return $access;
		}

		$username = sanitize_user($request->get_param('username'));
		$email = sanitize_email($request->get_param('email'));
		$password = $request->get_param('password');
		$name = sanitize_text_field($request->get_param('name'));

		if (empty($username) || empty($email) || empty($password)) {
			return $this->error('missing_fields', __('Username, email and password are required.', 'headlesskey-jwt-auth'), 400);
		}

		if (username_exists($username) || email_exists($email)) {
			return $this->error('user_exists', __('Username or email already registered.', 'headlesskey-jwt-auth'), 409);
		}

		$user_id = wp_insert_user(
			array(
				'user_login' => $username,
				'user_pass' => $password,
				'user_email' => $email,
				'display_name' => $name,
			)
		);

		if (is_wp_error($user_id)) {
			return $this->error('registration_failed', $user_id->get_error_message(), 400);
		}

		$user = get_user_by('ID', $user_id);

		$response = array(
			'user_id' => $user_id,
			'user' => $this->prepare_user_payload($user),
		);

		if (Settings::get('auto_login_after_register', true)) {
			$token_response = $this->issue_new_token_from_refresh($user);
			if (!is_wp_error($token_response)) {
				$response['token_response'] = $token_response;
			}
		}

		return new WP_REST_Response($response, 201);
	}

	/**
	 * Login endpoint returning profile + token.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function login_user(WP_REST_Request $request)
	{
		$access = $this->ensure_endpoint_access('login', $this->get_authenticated_user($request)['user']);
		if (is_wp_error($access)) {
			return $access;
		}

		$token_response = $this->generate_token($request);

		if (is_wp_error($token_response)) {
			return $token_response;
		}

		$data = $token_response->get_data();

		return new WP_REST_Response(
			array(
				'token' => $data['token'],
				'expiration' => $data['expiration'],
				'user' => $data['user'],
			),
			200
		);
	}

	/**
	 * Forgot password endpoint.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function forgot_password(WP_REST_Request $request)
	{
		$access = $this->ensure_endpoint_access('forgot_password', $this->get_authenticated_user($request)['user']);
		if (is_wp_error($access)) {
			return $access;
		}

		$login = sanitize_text_field($request->get_param('login'));
		$delivery = $request->get_param('delivery') ?? 'link';

		if (empty($login)) {
			return $this->error('missing_login', __('Please provide a username or email.', 'headlesskey-jwt-auth'), 400);
		}

		$user = get_user_by('email', $login);
		if (!$user) {
			$user = get_user_by('login', $login);
		}

		if (!$user) {
			return $this->error('user_not_found', __('User not found.', 'headlesskey-jwt-auth'), 404);
		}

		if ('otp' === $delivery) {
			$otp = wp_rand(100000, 999999);
			$timeout = (int) Settings::get('otp_expiration', 900);
			set_transient('headlesskey_otp_' . $user->ID, $otp, $timeout);

			// translators: %s: Site title
			$subject = sprintf(__('[%s] Password reset code', 'headlesskey-jwt-auth'), get_bloginfo('name'));
			// translators: %s: One-time password code
			$message = sprintf(__('Use this one time code to reset your password: %s', 'headlesskey-jwt-auth'), $otp);

			wp_mail(
				$user->user_email,
				$subject,
				$message
			);

			$message = __('OTP sent to your email address.', 'headlesskey-jwt-auth');
		} else {
			$result = retrieve_password($user->user_login);
			if (is_wp_error($result)) {
				return $this->error('reset_failed', $result->get_error_message(), 400);
			}

			$message = __('Password reset email sent.', 'headlesskey-jwt-auth');
		}

		$this->send_security_alert(
			'password_reset_request',
			array(
				'user_id' => $user->ID,
				'method' => $delivery,
			)
		);

		return new WP_REST_Response(
			array(
				'message' => $message,
			),
			200
		);
	}

	/**
	 * Reset password endpoint handling token or OTP.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function reset_password(WP_REST_Request $request)
	{
		$access = $this->ensure_endpoint_access('reset_password', $this->get_authenticated_user($request)['user']);
		if (is_wp_error($access)) {
			return $access;
		}

		$login = sanitize_text_field($request->get_param('login'));
		$password = $request->get_param('password');
		$token = $request->get_param('token');
		$otp = $request->get_param('otp');

		if (empty($login) || empty($password)) {
			return $this->error('missing_fields', __('Login and new password are required.', 'headlesskey-jwt-auth'), 400);
		}

		$user = get_user_by('email', $login);
		if (!$user) {
			$user = get_user_by('login', $login);
		}

		if (!$user) {
			return $this->error('user_not_found', __('User not found.', 'headlesskey-jwt-auth'), 404);
		}

		if (!empty($otp)) {
			$stored = get_transient('headlesskey_otp_' . $user->ID);
			if ((int) $stored !== (int) $otp) {
				return $this->error('invalid_otp', __('Invalid or expired OTP.', 'headlesskey-jwt-auth'), 400);
			}
			delete_transient('headlesskey_otp_' . $user->ID);
		} else {
			if (empty($token)) {
				return $this->error('missing_token', __('Reset token is required.', 'headlesskey-jwt-auth'), 400);
			}

			$key_user = check_password_reset_key($token, $user->user_login);

			if (is_wp_error($key_user)) {
				return $this->error('invalid_token', $key_user->get_error_message(), 400);
			}
		}

		reset_password($user, $password);

		return new WP_REST_Response(
			array(
				'message' => __('Password updated successfully.', 'headlesskey-jwt-auth'),
			),
			200
		);
	}

	/**
	 * Change password endpoint.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function change_password(WP_REST_Request $request)
	{
		$auth = $this->get_authenticated_user($request);

		if (!$auth['user']) {
			return $this->error('unauthorized', __('Authorization token required.', 'headlesskey-jwt-auth'), 401);
		}

		$access = $this->ensure_endpoint_access('change_password', $auth['user']);
		if (is_wp_error($access)) {
			return $access;
		}

		$current = $request->get_param('current_password');
		$new = $request->get_param('new_password');

		if (empty($current) || empty($new)) {
			return $this->error('missing_fields', __('Current and new passwords are required.', 'headlesskey-jwt-auth'), 400);
		}

		if (!wp_check_password($current, $auth['user']->user_pass, $auth['user']->ID)) {
			return $this->error('incorrect_password', __('Current password is incorrect.', 'headlesskey-jwt-auth'), 400);
		}

		wp_set_password($new, $auth['user']->ID);

		$this->revoked->revoke($auth['raw_token'], array('user_id' => $auth['user']->ID));
		$this->logs->update_by_hash(
			hash('sha256', $auth['raw_token']),
			array(
				'status' => 'revoked',
			)
		);

		return new WP_REST_Response(
			array(
				'message' => __('Password changed successfully. Please login again.', 'headlesskey-jwt-auth'),
			),
			200
		);
	}

	/**
	 * Exchange tokens between connected sites for SSO.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_REST_Response|WP_Error
	 */
	public function sso_exchange(WP_REST_Request $request)
	{
		if (!Settings::get('sso_enable', false)) {
			return $this->error('sso_disabled', __('SSO is not enabled.', 'headlesskey-jwt-auth'), 403);
		}

		$site_key = sanitize_text_field($request->get_param('site_key'));
		$remote_token = $request->get_param('token');
		$signature = $request->get_param('signature');

		if (empty($site_key) || empty($remote_token) || empty($signature)) {
			return $this->error('missing_fields', __('Site key, token and signature are required.', 'headlesskey-jwt-auth'), 400);
		}

		$connections = (array) Settings::get('sso_connections', array());
		$connection = null;

		foreach ($connections as $conn) {
			if (isset($conn['key']) && $conn['key'] === $site_key) {
				$connection = $conn;
				break;
			}
		}

		if (!$connection) {
			return $this->error('unknown_site', __('The requesting site is not registered.', 'headlesskey-jwt-auth'), 404);
		}

		$secret = $connection['shared_secret'] ?? '';
		$algorithm = isset($connection['algorithm']) ? strtoupper($connection['algorithm']) : 'HS256';

		if (empty($secret)) {
			return $this->error('missing_secret', __('Shared secret is missing for this site.', 'headlesskey-jwt-auth'), 400);
		}

		$expected = hash_hmac('sha256', $remote_token, $secret);

		if (!hash_equals($expected, $signature)) {
			return $this->error('invalid_signature', __('Signature verification failed.', 'headlesskey-jwt-auth'), 401);
		}

		try {
			$decoded = JWT::decode($remote_token, new Key($secret, $algorithm));
		} catch (\Exception $exception) {
			return $this->error('invalid_token', $exception->getMessage(), 401);
		}

		$user = null;

		if (isset($decoded->data->ID)) {
			$user = get_user_by('ID', (int) $decoded->data->ID);
		}

		if (!$user && isset($decoded->data->user_email)) {
			$user = get_user_by('email', sanitize_email($decoded->data->user_email));
		}

		if (!$user && isset($decoded->data->user_login)) {
			$user = get_user_by('login', sanitize_user($decoded->data->user_login));
		}

		if (!$user) {
			return $this->error('user_missing', __('User could not be matched locally.', 'headlesskey-jwt-auth'), 404);
		}

		$payload = $this->issue_new_token_from_refresh($user);

		if (is_wp_error($payload)) {
			return $payload;
		}

		$this->send_security_alert(
			'sso_exchange',
			array(
				'user_id' => $user->ID,
				'source' => $connection['site_url'] ?? '',
			)
		);

		return new WP_REST_Response(
			array(
				'message' => __('SSO exchange completed.', 'headlesskey-jwt-auth'),
				'payload' => $payload,
				'source' => $connection['site_url'] ?? '',
			),
			200
		);
	}

	/**
	 * Register custom CORS headers.
	 *
	 * @return void
	 */
	public function register_cors_handler()
	{
		add_filter(
			'rest_pre_serve_request',
			function ($served, $result, $request) {
				if (strpos($request->get_route(), '/' . headlesskey_REST_NAMESPACE) !== false) {
					$this->send_cors_headers();
				}
				return $served;
			},
			10,
			3
		);
	}

	/**
	 * Output CORS headers.
	 *
	 * @return void
	 */
	protected function send_cors_headers()
	{
		if (!Settings::get('cors_enabled', true)) {
			return;
		}

		$headers = apply_filters(
			'headlesskey_cors_headers',
			array(
				'Access-Control-Allow-Origin' => implode(',', (array) Settings::get('cors_allowed_origins', array('*'))),
				'Access-Control-Allow-Methods' => 'POST,GET,OPTIONS',
				'Access-Control-Allow-Headers' => 'Content-Type, Authorization',
				'Access-Control-Allow-Credentials' => 'true',
			)
		);

		foreach ($headers as $key => $value) {
			header($key . ': ' . $value);
		}
	}

	/**
	 * Prepare standard error.
	 *
	 * @param string $code Error code.
	 * @param string $message Message.
	 * @param int    $status HTTP status.
	 *
	 * @return WP_Error
	 */
	protected function error($code, $message, $status = 400)
	{
		return new WP_Error($code, $message, array('status' => $status));
	}

	/**
	 * Format user payload.
	 *
	 * @param WP_User $user User instance.
	 *
	 * @return array
	 */
	protected function prepare_user_payload(WP_User $user)
	{
		$fields = Settings::get('login_response_fields', array('ID', 'user_login', 'user_email', 'display_name', 'roles'));

		$payload = array();
		foreach ($fields as $field) {
			if ('roles' === $field) {
				$payload['roles'] = $user->roles;
			} elseif (isset($user->{$field})) {
				$payload[$field] = $user->{$field};
			}
		}

		$meta_payload = $this->prepare_meta_payload($user);

		if (!empty($meta_payload)) {
			$payload['meta'] = $meta_payload;
		}

		return $payload;
	}

	/**
	 * Prepare custom meta payload based on settings.
	 *
	 * @param WP_User $user User instance.
	 *
	 * @return array
	 */
	protected function prepare_meta_payload(WP_User $user)
	{
		$meta = array();

		$field_groups = array(
			'token_meta_fields',
			'token_acf_fields',
			'token_woo_fields',
		);

		foreach ($field_groups as $group) {
			$fields = (array) Settings::get($group, array());

			foreach ($fields as $field) {
				$key = sanitize_key($field);

				if (empty($key)) {
					continue;
				}

				$value = get_user_meta($user->ID, $key, true);

				if ('' === $value || null === $value) {
					continue;
				}

				$meta[$key] = $value;
			}
		}

		return $meta;
	}

	/**
	 * Issue new token bypassing password prompt.
	 *
	 * @param WP_User $user User instance.
	 * @param string  $previous_token Optional previous token to revoke.
	 *
	 * @return array|WP_Error
	 */
	protected function issue_new_token_from_refresh(WP_User $user, $previous_token = null)
	{
		$ip = $this->get_ip_address();
		$user_agent = sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'] ?? ''));
		$device_key = $this->devices->fingerprint($user_agent, $ip);

		$expiration = (int) apply_filters('headlesskey_expire', Settings::get('token_expiration', 3600), $user);
		$issued_at = time();
		$expires_at = $issued_at + $expiration;
		$jti = wp_generate_uuid4();

		$payload = array(
			'iss' => home_url(),
			'iat' => $issued_at,
			'nbf' => $issued_at,
			'exp' => $expires_at,
			'jti' => $jti,
			'data' => $this->prepare_user_payload($user),
		);

		try {
			$token = $this->tokens->issue($payload);
		} catch (\Exception $exception) {
			return $this->error('token_error', $exception->getMessage(), 500);
		}

		if ($previous_token) {
			$this->revoked->revoke($previous_token, array('user_id' => $user->ID));
			$this->logs->update_by_hash(
				hash('sha256', $previous_token),
				array(
					'status' => 'revoked',
				)
			);
		}

		$this->logs->insert(
			array(
				'user_id' => $user->ID,
				'token_jti' => $jti,
				'token_hash' => hash('sha256', $token),
				'issued_at' => gmdate('Y-m-d H:i:s', $issued_at),
				'expires_at' => gmdate('Y-m-d H:i:s', $expires_at),
				'ip_address' => $ip,
				'device' => $device_key,
				'user_agent' => $user_agent,
			)
		);

		$this->enforce_multi_token_policy($user->ID);

		return array(
			'token' => $token,
			'expiration' => gmdate('c', $expires_at),
			'user' => $this->prepare_user_payload($user),
			'jti' => $jti,
		);
	}

	/**
	 * Get authenticated user from Authorization header.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return array{user:WP_User|null,raw_token:string|null}
	 */
	protected function get_authenticated_user(WP_REST_Request $request)
	{
		$auth_header = $request->get_header('authorization');

		if (empty($auth_header)) {
			return array(
				'user' => null,
				'raw_token' => null,
			);
		}

		if (preg_match('/Bearer\s+(.*)$/i', $auth_header, $matches)) {
			$token = trim($matches[1]);
		} else {
			$token = $auth_header;
		}

		try {
			$decoded = $this->tokens->decode($token);
		} catch (\Exception $exception) {
			return array(
				'user' => null,
				'raw_token' => null,
			);
		}

		$user_id = isset($decoded->data->ID) ? (int) $decoded->data->ID : 0;
		$user = $user_id ? get_user_by('ID', $user_id) : null;

		return array(
			'user' => $user instanceof WP_User ? $user : null,
			'raw_token' => $token,
		);
	}

	/**
	 * Ensure endpoint access rules are satisfied.
	 *
	 * @param string      $endpoint Endpoint key.
	 * @param \WP_User|null $user   Acting user.
	 *
	 * @return true|WP_Error
	 */
	protected function ensure_endpoint_access($endpoint, $user = null)
	{
		if (!$this->roles->can_access($endpoint, $user)) {
			return $this->error('access_denied', __('You are not allowed to access this endpoint.', 'headlesskey-jwt-auth'), 403);
		}

		return true;
	}

	/**
	 * Enforce max token policies per user.
	 *
	 * @param int $user_id User ID.
	 *
	 * @return void
	 */
	protected function enforce_multi_token_policy($user_id)
	{
		if ($user_id <= 0) {
			return;
		}

		$allowed = Settings::get('allow_multi_token_sessions', true) ? (int) Settings::get('max_user_tokens', 5) : 1;

		$limit = max(1, $allowed);

		$extra_hashes = $this->logs->get_token_hashes_beyond_limit($user_id, $limit);

		if (empty($extra_hashes)) {
			return;
		}

		foreach ($extra_hashes as $hash) {
			$this->revoked->revoke_by_hash($hash);
		}

		$this->logs->mark_status_for_hashes($extra_hashes, 'revoked');
	}

	/**
	 * Handle brute force lock events.
	 *
	 * @param string $identifier Identifier.
	 * @param int    $attempts Attempts count.
	 *
	 * @return void
	 */
	public function handle_bruteforce_lock($identifier, $attempts)
	{
		$this->send_security_alert(
			'brute_force_detected',
			array(
				'identifier' => $identifier,
				'attempts' => $attempts,
				'ip' => $this->get_ip_address(),
			)
		);
	}

	/**
	 * Dispatch webhook alerts.
	 *
	 * @param string $event Event key.
	 * @param array  $payload Payload.
	 *
	 * @return void
	 */
	protected function send_security_alert($event, $payload = array())
	{
		if (!$this->alerts) {
			return;
		}

		$this->alerts->dispatch($event, $payload);
	}

	/**
	 * Determine blocked IPs.
	 *
	 * @return bool
	 */
	protected function is_ip_blocked()
	{
		$ip = $this->get_ip_address();
		$whitelist = array_filter((array) Settings::get('ip_whitelist', array()));
		$blacklist = array_filter((array) Settings::get('ip_blacklist', array()));

		if (!empty($whitelist) && !in_array($ip, $whitelist, true)) {
			$this->send_security_alert(
				'blocked_ip_attempt',
				array(
					'ip' => $ip,
					'reason' => 'not_whitelisted',
				)
			);
			return true;
		}

		if (in_array($ip, $blacklist, true)) {
			$this->send_security_alert(
				'blocked_ip_attempt',
				array(
					'ip' => $ip,
					'reason' => 'blacklisted',
				)
			);
			return true;
		}

		return false;
	}

	/**
	 * Get current IP address.
	 *
	 * @return string
	 */
	protected function get_ip_address()
	{
		foreach (array('HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR') as $key) {
			if (!empty($_SERVER[$key])) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				$ip_list = explode(',', wp_unslash($_SERVER[$key])); // phpcs:ignore
				$ip = trim($ip_list[0]);
				if (filter_var($ip, FILTER_VALIDATE_IP)) {
					return $ip;
				}
			}
		}

		return '';
	}

	/**
	 * Log activity into DB table.
	 *
	 * @param string $event Event code.
	 * @param array  $metadata Additional data.
	 *
	 * @return void
	 */
	protected function log_activity($event, $metadata = array())
	{
		global $wpdb;

		$table = Settings::get('allow_token_logs', true) ? Installer::get_table_name('activity_logs') : null;

		if (!$table) {
			return;
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		$wpdb->insert(
			$table,
			array(
				'user_id' => isset($metadata['user_id']) ? (int) $metadata['user_id'] : 0,
				'event' => $event,
				'context' => isset($metadata['context']) ? $metadata['context'] : null,
				'ip_address' => $this->get_ip_address(),
				'metadata' => maybe_serialize($metadata),
			)
		);
	}
}

