<?php
/**
 * Admin-only REST endpoints (logs, monitor, settings export/import).
 *
 * @package headlesskey
 */

namespace headlesskey\API;

use WP_REST_Request;
use WP_REST_Response;
use WP_REST_Server;
use headlesskey\Auth\RevocationStore;
use headlesskey\Auth\TokenLogRepository;
use headlesskey\Core\Installer;
use headlesskey\Core\Settings;

if (!defined('ABSPATH')) {
	exit;
}

/**
 * Class AdminToolsAPI
 */
class AdminToolsAPI
{
	/**
	 * Namespace.
	 *
	 * @var string
	 */
	protected $namespace;

	/**
	 * Logs repository.
	 *
	 * @var TokenLogRepository
	 */
	protected $logs;

	/**
	 * Revocation store.
	 *
	 * @var RevocationStore
	 */
	protected $revoked;

	/**
	 * Constructor.
	 */
	public function __construct()
	{
		$this->namespace = trailingslashit(headlesskey_REST_NAMESPACE) . headlesskey_REST_VERSION;
		$this->logs = new TokenLogRepository();
		$this->revoked = new RevocationStore();

		add_action('rest_api_init', array($this, 'register_routes'));
	}

	/**
	 * Register routes.
	 *
	 * @return void
	 */
	public function register_routes()
	{
		register_rest_route(
			$this->namespace,
			'/admin/logs/tokens',
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_token_logs'),
				'permission_callback' => array($this, 'permission_check'),
			)
		);

		register_rest_route(
			$this->namespace,
			'/admin/logs/tokens/clear',
			array(
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => array($this, 'clear_token_logs'),
				'permission_callback' => array($this, 'permission_check'),
			)
		);

		register_rest_route(
			$this->namespace,
			'/admin/logs/activity',
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_activity_logs'),
				'permission_callback' => array($this, 'permission_check'),
			)
		);

		register_rest_route(
			$this->namespace,
			'/admin/settings/export',
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'export_settings'),
				'permission_callback' => array($this, 'permission_check'),
			)
		);

		register_rest_route(
			$this->namespace,
			'/admin/settings/import',
			array(
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => array($this, 'import_settings'),
				'permission_callback' => array($this, 'permission_check'),
			)
		);

		register_rest_route(
			$this->namespace,
			'/admin/tokens/revoke',
			array(
				'methods' => WP_REST_Server::CREATABLE,
				'callback' => array($this, 'admin_revoke_token'),
				'permission_callback' => array($this, 'permission_check'),
			)
		);

		register_rest_route(
			$this->namespace,
			'/admin/tokens/summary',
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_token_summary'),
				'permission_callback' => array($this, 'permission_check'),
			)
		);

		register_rest_route(
			$this->namespace,
			'/admin/analytics/overview',
			array(
				'methods' => WP_REST_Server::READABLE,
				'callback' => array($this, 'get_analytics_overview'),
				'permission_callback' => array($this, 'permission_check'),
			)
		);
	}

	/**
	 * Permission callback.
	 *
	 * @return bool
	 */
	public function permission_check()
	{
		return current_user_can('manage_options');
	}

	/**
	 * Return token logs.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_REST_Response
	 */
	public function get_token_logs(WP_REST_Request $request)
	{
		$logs = $this->logs->get_logs(
			array(
				'user_id' => $request->get_param('user_id'),
				'status' => $request->get_param('status'),
				'limit' => $request->get_param('per_page') ?: 50,
				'offset' => $request->get_param('offset') ?: 0,
			)
		);

		return new WP_REST_Response(
			array(
				'logs' => $logs,
			),
			200
		);
	}

	/**
	 * Clear token logs.
	 *
	 * @return WP_REST_Response
	 */
	public function clear_token_logs()
	{
		global $wpdb;

		$table = Installer::get_table_name('token_logs');
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
		$wpdb->query(
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			"TRUNCATE TABLE {$table}"
		);

		return new WP_REST_Response(
			array(
				'message' => __('Token logs cleared.', 'headlesskey-jwt-auth'),
			),
			200
		);
	}

	/**
	 * Activity logs endpoint.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_REST_Response
	 */
	public function get_activity_logs(WP_REST_Request $request)
	{
		global $wpdb;

		$table = Installer::get_table_name('activity_logs');

		$limit = $request->get_param('per_page') ?: 50;
		$offset = $request->get_param('offset') ?: 0;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
		$logs = $wpdb->get_results(
			$wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				"SELECT * FROM {$table} ORDER BY id DESC LIMIT %d OFFSET %d",
				$limit,
				$offset
			),
			ARRAY_A
		);

		if ($logs) {
			foreach ($logs as &$log) {
				$log['metadata'] = maybe_unserialize($log['metadata']);
			}
		}

		return new WP_REST_Response(
			array(
				'logs' => $logs,
			),
			200
		);
	}

	/**
	 * Export settings endpoint.
	 *
	 * @return WP_REST_Response
	 */
	public function export_settings()
	{
		return new WP_REST_Response(
			array(
				'settings' => Settings::all(),
				'generated_at' => current_time('mysql', true),
			),
			200
		);
	}

	/**
	 * Import settings endpoint.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_REST_Response
	 */
	public function import_settings(WP_REST_Request $request)
	{
		$payload = $request->get_param('settings');

		if (empty($payload) || !is_array($payload)) {
			return new WP_REST_Response(
				array(
					'error' => __('Invalid settings payload.', 'headlesskey-jwt-auth'),
				),
				400
			);
		}

		foreach ($payload as $key => $value) {
			Settings::set($key, $value);
		}

		return new WP_REST_Response(
			array(
				'message' => __('Settings imported successfully.', 'headlesskey-jwt-auth'),
			),
			200
		);
	}

	/**
	 * Admin token revocation handler.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_REST_Response
	 */
	public function admin_revoke_token(WP_REST_Request $request)
	{
		$token_hash = sanitize_text_field($request->get_param('token_hash'));

		if (empty($token_hash)) {
			return new WP_REST_Response(
				array(
					'error' => __('Token hash is required.', 'headlesskey-jwt-auth'),
				),
				400
			);
		}

		$this->revoked->revoke_by_hash($token_hash);
		$this->logs->update_by_hash(
			$token_hash,
			array(
				'status' => 'revoked',
			)
		);

		return new WP_REST_Response(
			array(
				'message' => __('Token revoked.', 'headlesskey-jwt-auth'),
			),
			200
		);
	}

	/**
	 * Token summary counts for live monitor.
	 *
	 * @return WP_REST_Response
	 */
	public function get_token_summary()
	{
		global $wpdb;

		$table = Installer::get_table_name('token_logs');

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
		$results = $wpdb->get_results(
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			"SELECT status, COUNT(*) as total FROM {$table} GROUP BY status",
			ARRAY_A
		);

		$summary = array(
			'active' => 0,
			'revoked' => 0,
			'expired' => 0,
			'error' => 0,
		);

		foreach ($results as $row) {
			$status = $row['status'];
			if (isset($summary[$status])) {
				$summary[$status] = (int) $row['total'];
			}
		}

		return new WP_REST_Response(
			array(
				'summary' => $summary,
			),
			200
		);
	}

	/**
	 * Provide analytics overview for dashboard.
	 *
	 * @param WP_REST_Request $request Request.
	 *
	 * @return WP_REST_Response
	 */
	public function get_analytics_overview(WP_REST_Request $request)
	{
		global $wpdb;

		$token_table = Installer::get_table_name('token_logs');
		$activity_table = Installer::get_table_name('activity_logs');
		$days = max(1, (int) Settings::get('analytics_heatmap_days', 14));

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
		$top_users = $wpdb->get_results(
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			"SELECT user_id, COUNT(*) as total FROM {$token_table} WHERE user_id <> 0 GROUP BY user_id ORDER BY total DESC LIMIT 5",
			ARRAY_A
		);

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
		$active_tokens = (int) $wpdb->get_var(
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			"SELECT COUNT(*) FROM {$token_table} WHERE status = 'active'"
		);
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
		$revoked_tokens = (int) $wpdb->get_var(
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			"SELECT COUNT(*) FROM {$token_table} WHERE status = 'revoked'"
		);
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
		$expired_tokens = (int) $wpdb->get_var(
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			"SELECT COUNT(*) FROM {$token_table} WHERE status = 'expired'"
		);

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
		$failed_heatmap = $wpdb->get_results(
			$wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				"SELECT DATE(created_at) as day, HOUR(created_at) as hour, COUNT(*) as total FROM {$activity_table} WHERE event = %s AND created_at >= DATE_SUB(UTC_TIMESTAMP(), INTERVAL %d DAY) GROUP BY day, hour",
				'login_failed',
				$days
			),
			ARRAY_A
		);

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
		$locations = $wpdb->get_results(
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			"SELECT ip_address, COUNT(*) as total FROM {$token_table} WHERE ip_address <> '' GROUP BY ip_address ORDER BY total DESC LIMIT 10",
			ARRAY_A
		);

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
		$devices = $wpdb->get_results(
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			"SELECT device, COUNT(*) as total FROM {$token_table} WHERE device <> '' GROUP BY device ORDER BY total DESC LIMIT 10",
			ARRAY_A
		);

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
		$last_hour_tokens = (int) $wpdb->get_var(
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			"SELECT COUNT(*) FROM {$token_table} WHERE issued_at >= DATE_SUB(UTC_TIMESTAMP(), INTERVAL 1 HOUR)"
		);

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, PluginCheck.Security.DirectDB.UnescapedDBParameter
		$last_day_tokens = (int) $wpdb->get_var(
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			"SELECT COUNT(*) FROM {$token_table} WHERE issued_at >= DATE_SUB(UTC_TIMESTAMP(), INTERVAL 1 DAY)"
		);

		$average_per_hour = $last_day_tokens / max(1, 24);
		$spike_detected = $average_per_hour > 0 && $last_hour_tokens > ($average_per_hour * 1.5);

		return new WP_REST_Response(
			array(
				'summary' => array(
					'active' => $active_tokens,
					'revoked' => $revoked_tokens,
					'expired' => $expired_tokens,
					'last_hour' => $last_hour_tokens,
				),
				'top_users' => $top_users,
				'locations' => $locations,
				'devices' => $devices,
				'failed_heatmap' => $failed_heatmap,
				'spike' => array(
					'detected' => $spike_detected,
					'average_per_hour' => round($average_per_hour, 2),
					'last_hour' => $last_hour_tokens,
				),
			),
			200
		);
	}
}

