<?php
/**
 * HeadlessKey – JWT Auth
 *
 * @package           HeadlessKey – JWT Auth
 * @license           GPL-2.0-or-later
 *
 * @wordpress-plugin
 * Plugin Name:       HeadlessKey – JWT Auth
 * Description:       Secure your Headless WordPress with JWT Authentication. Includes Registration, SSO, RBAC, Password Reset, and Brute Force Protection.
 * Version:           1.0.0
 * Requires at least: 6.0
 * Requires PHP:      8.0
 * Author:            Hidayat Mahetar
 * Author URI:        https://profiles.wordpress.org/hidayatsafewp/
 * Text Domain:       headlesskey-jwt-auth
 * License:           GPL v2 or later
 * License URI:       http://www.gnu.org/licenses/gpl-2.0.txt
 */

/*
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

if (!defined('ABSPATH')) {
	exit;
}

define('headlesskey_VERSION', '1.0.0');
define('headlesskey_DIR', plugin_dir_path(__FILE__));
define('headlesskey_ROOT_FILE', __FILE__);
define('headlesskey_ROOT_FILE_RELATIVE_PATH', plugin_basename(__FILE__));
define('headlesskey_SLUG', 'headlesskey-jwt-auth');
define('headlesskey_FOLDER', dirname(plugin_basename(__FILE__)));
define('headlesskey_URL', plugins_url('', __FILE__));

if (!defined('headlesskey_REST_NAMESPACE')) {
	define('headlesskey_REST_NAMESPACE', 'headlesskey');
}

if (!defined('headlesskey_REST_VERSION')) {
	define('headlesskey_REST_VERSION', 'v1');
}

if (!defined('headlesskey_SECRET_KEY')) {
	define('headlesskey_SECRET_KEY', '');
}

if (!defined('headlesskey_CORS')) {
	define('headlesskey_CORS', true);
}

// Autoloader.
$headlesskey_autoloader = headlesskey_DIR . 'vendor/autoload_packages.php';
if (is_readable($headlesskey_autoloader)) {
	require_once $headlesskey_autoloader;
} else { // Something very unexpected. Error out gently with an admin_notice and exit loading.
	if (defined('WP_DEBUG') && WP_DEBUG) {
		error_log( // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			__('Error loading autoloader file for headlesskey plugin', 'headlesskey-jwt-auth')
		);
	}

	add_action(
		'admin_notices',
		function () {
			?>
		<div class="notice notice-error is-dismissible">
			<p>
				<?php
					printf(
						wp_kses(
							/* translators: Placeholder is a link to a support document. */
							__('Your installation of headlesskey is incomplete. If you installed headlesskey from GitHub, please refer to <a href="%1$s" target="_blank" rel="noopener noreferrer">this document</a> to set up your development environment. headlesskey must have Composer dependencies installed and built via the build command.', 'headlesskey-jwt-auth'),
							array(
								'a' => array(
									'href' => array(),
									'target' => array(),
									'rel' => array(),
								),
							)
						),
						'https://github.com/hidayatmahetar0/headlesskey-jwt-auth'
					);
					?>
			</p>
		</div>
		<?php
		}
	);

	return;
}

// Redirect to plugin onboarding when the plugin is activated.
add_action('activated_plugin', 'headlesskey_activation');

/**
 * Redirects to plugin onboarding when the plugin is activated
 *
 * @param string $plugin Path to the plugin file relative to the plugins directory.
 */
function headlesskey_activation($plugin)
{
	// Clear the permalinks after the post type has been registered.
	flush_rewrite_rules();
	if (
		headlesskey_ROOT_FILE_RELATIVE_PATH === $plugin &&
		\Automattic\Jetpack\Plugins_Installer::is_current_request_activating_plugin_from_plugins_screen(headlesskey_ROOT_FILE_RELATIVE_PATH)
	) {
		wp_safe_redirect(esc_url(admin_url('admin.php?page=headlesskey-jwt-auth#/getting-started')));
		exit;
	}
}

register_activation_hook(__FILE__, array('\headlesskey\Plugin', 'plugin_activation'));
register_deactivation_hook(__FILE__, array('\headlesskey\Plugin', 'plugin_deactivation'));

// Main plugin class.
if (class_exists(\headlesskey\Plugin::class)) {
	new \headlesskey\Plugin();
}
