<?php
/**
 * Plugin Name: Guest Author Meta Block
 * Description: A plugin that adds a "Guest Author(s)" meta box to the post editing sidebar and a Gutenberg block to display it.
 * Version: 1.3
 * Author: <a href="https://github.com/ghost-ng/" target="_blank">ghost-ng</a>
 * Author URI: https://github.com/ghost-ng/
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// Prevent direct access to the file.
if (!defined('ABSPATH')) {
    exit;
}

// Register the custom post meta for guest author.
function gam_register_guest_author_meta() {
    register_post_meta('post', '_guest_author_name', array(
        'type' => 'string',
        'description' => 'Guest Author Name',
        'single' => true,
        'show_in_rest' => true, // Ensure it's available in the REST API for Gutenberg.
        'auth_callback' => function() {
            return current_user_can('edit_posts');
        }
    ));
}
add_action('init', 'gam_register_guest_author_meta');

// Add Guest Author Settings Menu
function gam_add_admin_menu() {
    add_menu_page(
        'Guest Author Settings',   // Page Title
        'Guest Author',            // Menu Title
        'manage_options',          // Capability
        'guest-author-settings',   // Menu Slug
        'gam_settings_page',       // Callback Function
        'dashicons-admin-users',   // Icon
        25                         // Position
    );
}
add_action( 'admin_menu', 'gam_add_admin_menu' );

// Render the Settings Page
// Render the Settings Page with Instructions
function gam_settings_page() {
    ?>
    <div class="wrap">
        <h1><?php esc_html_e( 'Guest Author Settings', 'guest-author-meta-block' ); ?></h1>
        <p><?php esc_html_e( 'Use these settings to configure the default guest author name and attribution intro text.', 'guest-author-meta-block' ); ?></p>
        <p><strong><?php esc_html_e( 'Instructions:', 'guest-author-meta-block' ); ?></strong></p>
        <ul style="margin-bottom: 20px;">
            <li><?php esc_html_e( 'The "Default Guest Author Name" will be used if no guest author is set for a post.', 'guest-author-meta-block' ); ?></li>
            <li><?php esc_html_e( 'The "Attribution Intro Text" (e.g., "By:" or "Written by:") will appear before the guest author name.', 'guest-author-meta-block' ); ?></li>
            <li><?php esc_html_e( 'If the guest author and the default guest author name are empty, no intro or guest author will be displayed.', 'guest-author-meta-block' ); ?></li>
            <p><strong><?php esc_html_e( 'Shortcodes:', 'guest-author-meta-block' ); ?></strong></p>
            <li><?php esc_html_e( 'Use the [guest_authors_list] shortcode to display a list of guest authors.', 'guest-author-meta-block' ); ?></li>
            <li><?php esc_html_e( 'The shortcode accepts a "style" attribute with values "grid", "ul", or "ol" for different list styles.', 'guest-author-meta-block' ); ?></li>
            <li><?php esc_html_e( 'Example: [guest_authors_list style="grid"]', 'guest-author-meta-block' ); ?></li>
            <p><strong><?php esc_html_e( 'REST API:', 'guest-author-meta-block' ); ?></strong></p>
            <li><?php esc_html_e( 'Use the /wp-json/custom/v1/guest-author-search endpoint to search for guest authors.', 'guest-author-meta-block' ); ?></li>
            <li><?php esc_html_e( 'Example: /wp-json/custom/v1/guest-author-search?name=John Doe', 'guest-author-meta-block' ); ?></li>
            <li><?php esc_html_e( 'Use the /wp-json/custom/v1/guest-author/{id} endpoint to get or update (POST) guest author by post ID.', 'guest-author-meta-block' ); ?></li>
            <li><?php esc_html_e( 'Example: /wp-json/custom/v1/guest-author/123', 'guest-author-meta-block' ); ?></li>
        </ul>
        
        <form method="post" action="options.php">
            <?php
            settings_fields( 'gam_settings_group' );
            do_settings_sections( 'guest-author-settings' );
            submit_button();
            ?>
        </form>
    </div>
    <?php
}


// Register Settings Section and Fields
function gam_register_plugin_settings() {
    // Register Default Guest Author Name
    register_setting( 'gam_settings_group', 'gam_default_guest_author_name', [
        'type' => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default' => 'Guest Author'
    ]);

    // Register Guest Author Intro Text
    register_setting( 'gam_settings_group', 'gam_guest_author_intro', [
        'type' => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default' => 'By:'
    ]);

    // Add Settings Section
    add_settings_section(
        'gam_general_settings_section',
        'General Settings',
        '__return_false',
        'guest-author-settings'
    );

    // Field: Default Guest Author Name
    add_settings_field(
        'gam_default_guest_author_name',
        'Default Guest Author Name',
        'gam_default_guest_author_name_callback',
        'guest-author-settings',
        'gam_general_settings_section'
    );

    // Field: Attribution Intro Text
    add_settings_field(
        'gam_guest_author_intro',
        'Attribution Intro Text',
        'gam_guest_author_intro_callback',
        'guest-author-settings',
        'gam_general_settings_section'
    );
}
add_action( 'admin_init', 'gam_register_plugin_settings' );

// Callback: Default Guest Author Name Field
function gam_default_guest_author_name_callback() {
    $value = get_option( 'gam_default_guest_author_name' );
    $default = 'Guest Author';
    $value = $value !== false ? $value : $default;
    
    echo '<input type="text" name="gam_default_guest_author_name" value="' . esc_attr( $value ) . '" />';
}

// Callback: Attribution Intro Text Field
function gam_guest_author_intro_callback() {
    $value = get_option( 'gam_guest_author_intro' );
    $default = 'By:';
    $value = $value !== false ? $value : $default;
    
    echo '<input type="text" name="gam_guest_author_intro" value="' . esc_attr( $value ) . '" />';
}



// Add the meta box to the post editing sidebar.
function gam_add_guest_author_meta_box() {
    add_meta_box(
        'guest_author_meta_box',          // Unique ID
        'Guest Author(s)',                // Box title
        'gam_guest_author_meta_box_html', // Content callback
        'post',                           // Post type
        'side',                           // Context (right sidebar)
        'high'                            // Priority
    );
}
add_action('add_meta_boxes', 'gam_add_guest_author_meta_box');

// Render the HTML for the meta box.
function gam_guest_author_meta_box_html($post) {
    // Retrieve the value from the post meta.
    $guest_authors = get_post_meta($post->ID, '_guest_author_name', true);
    
    // Output the nonce field
    wp_nonce_field('guest_author_meta_box', 'guest_author_meta_box_nonce');
    
    ?>
    <label for="guest_author_field">Guest Author(s)</label>
    <input type="text" id="guest_author_field" name="guest_author_field" value="<?php echo esc_attr($guest_authors); ?>" placeholder="Enter guest author name(s)" style="width:100%;">
    <?php
}


// Save the meta box data when the post is saved.
function gam_save_guest_author_meta_box($post_id) {
    // Check if the nonce field is present in the POST data.
    if (!isset($_POST['guest_author_meta_box_nonce'])) {
        return; // If nonce is missing, exit the function.
    }

    // Verify the nonce to ensure the request is valid.
    $nonce = sanitize_text_field( wp_unslash( $_POST['guest_author_meta_box_nonce'] ) );
    if (!wp_verify_nonce(wp_unslash($nonce), 'guest_author_meta_box')) {
        return; // If nonce verification fails, exit the function.
    }

    // Prevent the meta from being saved on autosaves.
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        return;
    }

    // Check if the user has permission to edit the post.
    if (!current_user_can('edit_post', $post_id)) {
        return;
    }

    // Ensure the guest author field is set before saving.
    if (isset($_POST['guest_author_field'])) {
        // Sanitize the input and update the post meta. Must be SQL safe
        update_post_meta($post_id, '_guest_author_name', sanitize_text_field(wp_unslash($_POST['guest_author_field'])));
    }
}
add_action('save_post', 'gam_save_guest_author_meta_box');


// Enqueue the Gutenberg block's JavaScript for the block editor.
function gam_enqueue_block_editor_assets() {
    wp_enqueue_script(
        'gam-guest-author-block',
        plugins_url('guest-author-meta-block.js', __FILE__),
        array('wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-data'),
        filemtime(plugin_dir_path(__FILE__) . 'guest-author-meta-block.js') // Ensure the file is refreshed when modified.
    );
}
add_action('enqueue_block_editor_assets', 'gam_enqueue_block_editor_assets');

// Render the guest author name dynamically for each post on the frontend.
function render_guest_author_block( $attributes, $content, $block ) {
    // Fetch the postId from the block's context
    $post_id = isset( $block->context['postId'] ) ? $block->context['postId'] : get_the_ID();

    // Fetch guest author from post meta
    $guest_author = get_post_meta( $post_id, '_guest_author_name', true );
	$default_guest_author = get_option( 'gam_default_guest_author_name', 'Guest Author' );
	$intro = get_option( 'gam_guest_author_intro', '' );
	
	if ( ! $guest_author ) {
        $guest_author = $default_guest_author;
    }

    // Handle block attributes for formatting
    $font_size = isset( $attributes['fontSize'] ) ? $attributes['fontSize'] : 16;
    $font_weight = isset( $attributes['isBold'] ) && $attributes['isBold'] ? 'bold' : 'normal';
    $font_style = isset( $attributes['isItalic'] ) && $attributes['isItalic'] ? 'italic' : 'normal';
    $text_decoration = isset( $attributes['isUnderline'] ) && $attributes['isUnderline'] ? 'underline' : 'none';
    $is_hidden = isset( $attributes['isHidden'] ) && $attributes['isHidden'];

    // If hidden, return a placeholder
    if ( $is_hidden ) {
        return '<div><hidden placeholder></div>';
    }

    // Style for the output (similar to editor-side)
    $style = sprintf(
        'font-size: %dpx; font-weight: %s; font-style: %s; text-decoration: %s;',
        esc_attr( $font_size ),
        esc_attr( $font_weight ),
        esc_attr( $font_style ),
        esc_attr( $text_decoration )
    );

    // If no guest author is set, use the default or skip intro if empty
    if ( empty( $guest_author ) ) {
        $guest_author = $default_guest_author;
        if ( empty( $guest_author ) ) {
            return '';  // Return empty if no author at all
        }
        return sprintf(
            '<div class="wp-block-guest-author" style="%s">%s</div>',
            esc_attr( $style ),
            esc_html( $guest_author )
        );
    } else {
        return sprintf(
            '<div class="wp-block-guest-author" style="%s">%s %s</div>',
            esc_attr( $style ),
            esc_html( $intro ),
            esc_html( $guest_author )
        );
    }
}


// Register the dynamic block with the rendering callback
register_block_type( 'gam/guest-author-display', array(
    'render_callback' => 'render_guest_author_block',
    'attributes'      => array(
        'fontSize'   => array(
            'type'    => 'number',
            'default' => 16,
        ),
        'isBold'     => array(
            'type'    => 'boolean',
            'default' => false,
        ),
        'isItalic'   => array(
            'type'    => 'boolean',
            'default' => false,
        ),
        'isUnderline'=> array(
            'type'    => 'boolean',
            'default' => false,
        ),
        'isHidden'   => array(
            'type'    => 'boolean',
            'default' => false,
        ),
    ),
    'provides_context' => array(
        'postId' => 'postId',
    ),
) );



// ---- SEARCH Bar INTEGRATION ----

function gam_aggregate_guest_author_search( $query ) {
    if ( is_admin() || ! $query->is_main_query() || ! $query->is_search() ) {
        return;
    }

    $search_term = $query->get( 's' );

    // Step 1: Run a custom query to fetch posts by guest author
    $meta_query = new WP_Query( array(
        'post_type'  => array( 'post', 'page' ),
        'post_status'=> 'publish',
        'meta_query' => array(
            array(
                'key'     => '_guest_author_name',
                'value'   => $search_term,
                'compare' => 'LIKE'
            )
        ),
        'fields'     => 'ids',  // Return post IDs only
    ));

    // Log for debugging
    //error_log( '[Guest Author Search] Meta query post IDs: ' . print_r( $meta_query->posts, true ) );

    // Step 2: Append these results to the main search results
    add_filter( 'the_posts', function( $posts, $query ) use ( $meta_query ) {
        if ( is_search() && ! is_admin() && $query->is_main_query() ) {
            // If meta_query returns posts, fetch the full post objects
            if ( $meta_query->have_posts() ) {
                $additional_posts = get_posts( array(
                    'post__in'    => $meta_query->posts,
                    'post_type'   => array( 'post', 'page' ),
                    'post_status' => 'publish'
                ));

                // Merge and filter duplicates
                $posts = array_merge( $posts, $additional_posts );
                $posts = array_unique( $posts, SORT_REGULAR );

                //error_log( '[Guest Author Search] Final merged posts: ' . print_r( wp_list_pluck( $posts, 'ID' ), true ) );
            }
        }
        return $posts;
    }, 10, 2 );
}
add_action( 'pre_get_posts', 'gam_aggregate_guest_author_search' );


// ------- Shortcode for Author List ----------

function gam_get_guest_authors() {
    global $wpdb;

    // Cache key for storing guest authors
    $cache_key = 'guest_author_list';
    $results = wp_cache_get($cache_key);

    // If cache is empty, query the database
    if ($results === false) {
        $results = $wpdb->get_results(
            $wpdb->prepare("
                SELECT DISTINCT TRIM(meta_value) as meta_value
                FROM {$wpdb->postmeta}
                WHERE meta_key = %s
                AND TRIM(meta_value) != ''
                AND meta_value IS NOT NULL
                ORDER BY meta_value ASC
            ", '_guest_author_name')
        );

        // Cache results for 1 hour
        wp_cache_set($cache_key, $results, '', HOUR_IN_SECONDS);
    }

    // Process results and split authors
    $authors = array();

    foreach ($results as $result) {
        $meta_value = $result->meta_value;

        // Split by "and", commas, but avoid splitting for "Jr." and "Sr."
        $split_authors = preg_split('/(?<!\b(Jr|Sr)\b)[,&](?!\s*\b(Jr|Sr)\b)|\band\b(?![^\(]*\))/', $meta_value);

        foreach ($split_authors as $author) {
            $trimmed_author = trim($author);

            if (!empty($trimmed_author)) {
                $authors[] = sanitize_text_field($trimmed_author);
            }
        }
    }

    // Return unique authors
    return array_unique($authors);
}


function gam_display_guest_authors($atts) {
    global $wpdb;

    // Shortcode attributes with default 'ul' style
    $atts = shortcode_atts(array(
        'style' => 'ul'
    ), $atts);

    // Fetch distinct guest authors
    $authors = gam_get_guest_authors();

    usort($authors, function($a, $b) {
        $a_parts = explode(' ', $a);
        $b_parts = explode(' ', $b);
        return strcmp(end($a_parts), end($b_parts));
    });

    // Return if no authors exist
    if (empty($authors)) {
        return '<p>No guest authors found.</p>';
    }

    // Generate output based on the style parameter
    $output = '';

    if ($atts['style'] === 'ol') {
        $output .= '<ol class="guest-authors-list">';
        foreach ($authors as $author) {
            $encoded_author = rawurlencode($author);
            $url = site_url('guest-author/' . $encoded_author . '/');
            $output .= '<li class="author-name"><a href="' . esc_url($url) . '">' . esc_html($author) . '</a></li>';
        }
        $output .= '</ol>';
    } elseif ($atts['style'] === 'index' || $atts['style'] === 'grid') {
        // Group authors by the first letter of their last name
        $grouped_authors = [];

        foreach ($authors as $author) {
            $parts = explode(' ', $author);
            $last_name = end($parts);
            $first_letter = strtoupper(substr($last_name, 0, 1));
            $grouped_authors[$first_letter][] = $author;
        }

        // Sort groups alphabetically by letter
        ksort($grouped_authors);

        // Apply grid or index class
        $container_class = $atts['style'] === 'grid' ? 'guest-authors-grid' : 'guest-authors-index';

        $output .= '<div class="' . esc_attr($container_class) . '">';
        foreach ($grouped_authors as $letter => $author_list) {
            $output .= '<div class="author-section">';
            $output .= '<h2>' . esc_html($letter) . '</h2>';
            $output .= '<ul>';
            foreach ($author_list as $author) {
                $encoded_author = rawurlencode($author);
                $url = site_url('guest-author/' . $encoded_author . '/');
                $output .= '<li class="author-name"><a href="' . esc_url($url) . '">' . esc_html($author) . '</a></li>';
            }
            $output .= '</ul>';
            $output .= '</div>';
        }
        $output .= '</div>';
    } else {
        // Default to unordered list
        $output .= '<ul class="guest-authors-list">';
        foreach ($authors as $author) {
            $encoded_author = rawurlencode($author);
            $url = site_url('guest-author/' . $encoded_author . '/');
            $output .= '<li class="author-name"><a href="' . esc_url($url) . '">' . esc_html($author) . '</a></li>';
        }
        $output .= '</ul>';
    }

    return $output;
}
add_shortcode('guest_authors_list', 'gam_display_guest_authors');  // shortcode: [guest_authors_list style="ul"]

function gam_enqueue_guest_author_styles() {
    $css_file = plugin_dir_path(__FILE__) . 'styles.css';

    if (file_exists($css_file)) {
        wp_enqueue_style(
            'guest-author-styles',
            plugins_url('styles.css', __FILE__),
            array(),
            filemtime($css_file)  // Use styles.css for filemtime()
        );
    }
}
add_action('wp_enqueue_scripts', 'gam_enqueue_guest_author_styles');





// Shortcode to display guest author list with precise meta field filtering
function gam_guest_author_list_shortcode($atts) {
    $atts = shortcode_atts(array(
        'exact' => 'false'
    ), $atts, 'guest_author_list');

    $exact = filter_var($atts['exact'], FILTER_VALIDATE_BOOLEAN);
    $guest_authors = get_posts(array(
        'meta_key' => '_guest_author_name',
        'post_type' => 'post',
        'posts_per_page' => -1,
        'fields' => 'ids'
    ));

    $guest_author_names = array();
    foreach ($guest_authors as $post_id) {
        $guest_author_name = get_post_meta($post_id, '_guest_author_name', true);
        if ($guest_author_name && !in_array($guest_author_name, $guest_author_names)) {
            $guest_author_names[] = $guest_author_name;
        }
    }

    $output = '<ul class="guest-author-list">';
    foreach ($guest_author_names as $guest_author_name) {
        $url = site_url('guest-author/' . urlencode(sanitize_title($guest_author_name)));
        $output .= '<li><a href="' . esc_url($url) . '">' . esc_html($guest_author_name) . '</a></li>';
    }
    $output .= '</ul>';

    return $output;
}
add_shortcode('guest_author_list', 'gam_guest_author_list_shortcode');





// ----- REST API ------

// Register the custom REST API endpoint
add_action('rest_api_init', function() {
    // Register individual guest author by ID
    register_rest_route('custom/v1', '/guest-author/(?P<id>\d+)', array(
        'methods'             => array('GET', 'POST'),
        'callback'            => 'gam_handle_guest_author_request',
        'permission_callback' => function() {
            return current_user_can('edit_posts');
        }
    ));

    // Register guest author search endpoint
    // GET /wp-json/custom/v1/guest-author-search?name=John Doe
    register_rest_route('custom/v1', '/guest-author-search', array(
        'methods'             => 'GET',
        'callback'            => 'gam_handle_guest_author_search_request',
        'permission_callback' => function() {
            return current_user_can('edit_posts');
        }
    ));
});

// Handle Guest Author Search API Requests
function gam_handle_guest_author_search_request($request) {
    $search_term = sanitize_text_field($request->get_param('name'));

    // Validate that a search term is provided
    if (empty($search_term)) {
        return new WP_Error(
            'no_search_term',
            __('A search term is required.', 'guest-author-meta-block'),
            array('status' => 400)
        );
    }

    // Query for guest authors by meta field
    $query = new WP_Query(array(
        'post_type'      => 'post',
        'posts_per_page' => -1,
        'meta_query'     => array(
            array(
                'key'     => '_guest_author_name',
                'value'   => $search_term,
                'compare' => 'LIKE'
            )
        )
    ));

    // Prepare results
    $results = array();
    foreach ($query->posts as $post) {
        $results[] = array(
            'id'    => $post->ID,
            'title' => get_the_title($post->ID),
            'name'  => get_post_meta($post->ID, '_guest_author_name', true)
        );
    }

    // Return results or a 404 if none are found
    if (empty($results)) {
        return new WP_Error(
            'no_results',
            __('No posts found for the given guest author.', 'guest-author-meta-block'),
            array('status' => 404)
        );
    }

    return rest_ensure_response($results);
}



// Handle Guest Author API Requests (GET and POST)
function gam_handle_guest_author_request($request) {
    $post_id = $request['id'];

    // Handle POST (Update guest author)
    if ('POST' === $request->get_method()) {
        $guest_author = sanitize_text_field($request->get_param('guest_author'));

        if (empty($guest_author)) {
            return new WP_Error('empty_field', 'Guest author is required.', array('status' => 400));
        }

        update_post_meta($post_id, '_guest_author_name', $guest_author);

        return rest_ensure_response(array(
            'post_id'      => $post_id,
            'guest_author' => $guest_author,
            'message'      => 'Guest author updated successfully'
        ));
    }

    // Handle GET (Retrieve guest author)
    $meta = get_post_meta($post_id, '_guest_author_name', true);

    if (empty($meta)) {
        return new WP_Error('no_meta', 'No guest author found.', array('status' => 404));
    }

    return rest_ensure_response(array(
        'post_id'      => $post_id,
        'guest_author' => $meta
    ));
}


// ---- Guest Author Search Results ----


// Register the rewrite rule
function guest_author_rewrite_rule() {
    add_rewrite_rule('^guest-author/([^/]+)/?', 'index.php?guest_author=$matches[1]', 'top');
}
add_action('init', 'guest_author_rewrite_rule');

// Flush rewrite rules on activation
function flush_rewrite_rules_on_activation() {
    flush_rewrite_rules();
}
register_activation_hook(__FILE__, 'flush_rewrite_rules_on_activation');

// Register guest_author as a query variable
function register_var_guest_author($vars) {
    $vars[] = 'guest_author';
    return $vars;
}
add_filter('query_vars', 'register_var_guest_author');


function modify_query_for_guest_author($query) {
    if (!is_admin() && $query->is_main_query() && get_query_var('guest_author')) {
        // Get guest author from the query variable
        $guest_author = get_query_var('guest_author', '');
        $guest_author = sanitize_text_field(urldecode($guest_author));

        if (!empty($guest_author)) {
            //error_log('Filtering posts for Guest Author: ' . $guest_author);

            // Modify the existing main query to filter by _guest_author_name meta field
            $meta_query = array(
                array(
                    'key'     => '_guest_author_name',
                    'value'   => $guest_author,
                    'compare' => 'LIKE'
                )
            );

            $query->set('meta_query', $meta_query);
            
            // Set appropriate query flags
            $query->is_archive = true;
            $query->is_search = false;
            $query->is_home = false;

            //error_log('Guest Author query modified: ' . print_r($query->query_vars, true));
        }
    }
}
add_action('pre_get_posts', 'modify_query_for_guest_author');


function load_theme_search_for_guest_author($template) {
    if (get_query_var('guest_author')) {
        // Check if the theme has search.php, fall back to archive.php
        $custom_template = locate_template(array('search.php', 'archive.php'));

        if ($custom_template) {
            //error_log('Using theme search or archive template for guest author.');
            return $custom_template;
        }
    }
    return $template;
}
add_filter('template_include', 'load_theme_search_for_guest_author');

function modify_guest_author_archive_title($title) {
    // Check if it's a guest author query
    $guest_author = get_query_var('guest_author');

    if (!empty($guest_author)) {
        // translators: %s is the guest author name
        $title = sprintf(esc_html__('By Author: %s', 'guest-author-meta-block'), esc_html(urldecode($guest_author)));
    }
    
    return $title;
}
add_filter('get_the_archive_title', 'modify_guest_author_archive_title');

// Optional: Flush Rewrite Rules on Deactivation
function flush_rewrite_rules_on_deactivation() {
    flush_rewrite_rules();
}
register_deactivation_hook(__FILE__, 'flush_rewrite_rules_on_deactivation');
