<?php

/**
 * Main plugin class file.
 *
 * @package WordPress Plugin Template/Includes
 */
if ( !defined( 'ABSPATH' ) ) {
    exit;
}
/**
 * Main plugin class.
 */
class Guest_Author_Affiliate {
    /**
     * The single instance of Guest_Author_Affiliate.
     *
     * @var     object
     * @access  private
     * @since   1.0.0
     */
    private static $_instance = null;

    //phpcs:ignore
    /**
     * Local instance of Guest_Author_Affiliate_Admin_API
     *
     * @var Guest_Author_Affiliate_Admin_API|null
     */
    public $admin = null;

    /**
     * Settings class object
     *
     * @var     object
     * @access  public
     * @since   1.0.0
     */
    public $settings = null;

    /**
     * The version number.
     *
     * @var     string
     * @access  public
     * @since   1.0.0
     */
    public $_version;

    //phpcs:ignore
    /**
     * The token.
     *
     * @var     string
     * @access  public
     * @since   1.0.0
     */
    public $_token;

    //phpcs:ignore
    /**
     * The main plugin file.
     *
     * @var     string
     * @access  public
     * @since   1.0.0
     */
    public $file;

    /**
     * The main plugin directory.
     *
     * @var     string
     * @access  public
     * @since   1.0.0
     */
    public $dir;

    /**
     * The plugin assets directory.
     *
     * @var     string
     * @access  public
     * @since   1.0.0
     */
    public $assets_dir;

    /**
     * The plugin assets URL.
     *
     * @var     string
     * @access  public
     * @since   1.0.0
     */
    public $assets_url;

    /**
     * Suffix for JavaScripts.
     *
     * @var     string
     * @access  public
     * @since   1.0.0
     */
    public $script_suffix;

    /**
     * Constructor funtion.
     *
     * @param string $file File constructor.
     * @param string $version Plugin version.
     */
    public function __construct( $file = '', $version = '1.0.0' ) {
        $this->_version = $version;
        $this->_token = 'guest_author_affiliate';
        // Load plugin environment variables.
        $this->file = $file;
        $this->dir = dirname( $this->file );
        $this->assets_dir = trailingslashit( $this->dir ) . 'assets';
        $this->assets_url = esc_url( trailingslashit( plugins_url( '/assets/', $this->file ) ) );
        $this->script_suffix = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min' );
        register_activation_hook( $this->file, array($this, 'install') );
        // Load frontend JS & CSS.
        add_action( 'wp_enqueue_scripts', array($this, 'enqueue_styles'), 10 );
        add_action( 'wp_enqueue_scripts', array($this, 'enqueue_scripts'), 10 );
        // Load admin JS & CSS.
        add_action(
            'admin_enqueue_scripts',
            array($this, 'admin_enqueue_scripts'),
            10,
            1
        );
        add_action(
            'admin_enqueue_scripts',
            array($this, 'admin_enqueue_styles'),
            10,
            1
        );
        // Load API for generic admin functions.
        if ( is_admin() ) {
            $this->admin = new Guest_Author_Affiliate_Admin_API();
        }
        // Handle localisation.
        $this->load_plugin_textdomain();
        add_action( 'init', array($this, 'load_localisation'), 0 );
        // add_action( 'init', array( $this, 'process_page_load' ), 1000 );
        add_action( 'loop_start', array($this, '_loop_start') );
        add_action( 'wp_ajax_guest_author_affiliate_set_affiliation_coockie', array($this, 'set_affiliation_coockie') );
        add_action( 'wp_ajax_nopriv_guest_author_affiliate_set_affiliation_coockie', array($this, 'set_affiliation_coockie') );
        // The Affiliates Manager plugin support
        // @see https://wordpress.org/plugins/affiliates-manager/
        add_filter(
            'guest_author_affiliate_get_affiliate_token',
            array($this, 'WPAM_get_affiliate_token'),
            10,
            2
        );
        add_action(
            'guest_author_affiliate_set_affiliation_coockie_handler',
            array($this, 'WPAM_set_affiliation_coockie_handler'),
            10,
            1
        );
        // The YITH WooCommerce Affiliates plugin support
        // @see https://wordpress.org/plugins/yith-woocommerce-affiliates/
        add_filter(
            'guest_author_affiliate_get_affiliate_token',
            array($this, 'YITH_WCAF_get_affiliate_token'),
            10,
            2
        );
        add_action(
            'guest_author_affiliate_set_affiliation_coockie_handler',
            array($this, 'YITH_WCAF_set_affiliation_coockie_handler'),
            10,
            1
        );
        add_filter(
            'wpam_woo_override_refkey',
            array($this, 'WPAM_woo_override_refkey'),
            10,
            2
        );
    }

    // End __construct ()
    /**
     * Register post type function.
     *
     * @param string $post_type Post Type.
     * @param string $plural Plural Label.
     * @param string $single Single Label.
     * @param string $description Description.
     * @param array  $options Options array.
     *
     * @return bool|string|Guest_Author_Affiliate_Post_Type
     */
    public function register_post_type(
        $post_type = '',
        $plural = '',
        $single = '',
        $description = '',
        $options = array()
    ) {
        if ( !$post_type || !$plural || !$single ) {
            return false;
        }
        $post_type = new Guest_Author_Affiliate_Post_Type(
            $post_type,
            $plural,
            $single,
            $description,
            $options
        );
        return $post_type;
    }

    /**
     * Wrapper function to register a new taxonomy.
     *
     * @param string $taxonomy Taxonomy.
     * @param string $plural Plural Label.
     * @param string $single Single Label.
     * @param array  $post_types Post types to register this taxonomy for.
     * @param array  $taxonomy_args Taxonomy arguments.
     *
     * @return bool|string|Guest_Author_Affiliate_Taxonomy
     */
    public function register_taxonomy(
        $taxonomy = '',
        $plural = '',
        $single = '',
        $post_types = array(),
        $taxonomy_args = array()
    ) {
        if ( !$taxonomy || !$plural || !$single ) {
            return false;
        }
        $taxonomy = new Guest_Author_Affiliate_Taxonomy(
            $taxonomy,
            $plural,
            $single,
            $post_types,
            $taxonomy_args
        );
        return $taxonomy;
    }

    /**
     * Load frontend CSS.
     *
     * @access  public
     * @return void
     * @since   1.0.0
     */
    public function enqueue_styles() {
        wp_register_style(
            $this->_token . '-frontend',
            esc_url( $this->assets_url ) . 'css/frontend.css',
            array(),
            $this->_version
        );
        wp_enqueue_style( $this->_token . '-frontend' );
    }

    // End enqueue_styles ()
    /**
     * Load frontend Javascript.
     *
     * @access  public
     * @return  void
     * @since   1.0.0
     */
    public function enqueue_scripts() {
        wp_register_script(
            $this->_token . '-frontend',
            esc_url( $this->assets_url ) . 'js/frontend' . $this->script_suffix . '.js',
            array('jquery'),
            $this->_version,
            true
        );
        wp_enqueue_script( $this->_token . '-frontend' );
    }

    // End enqueue_scripts ()
    /**
     * Admin enqueue style.
     *
     * @param string $hook Hook parameter.
     *
     * @return void
     */
    public function admin_enqueue_styles( $hook = '' ) {
        wp_register_style(
            $this->_token . '-admin',
            esc_url( $this->assets_url ) . 'css/admin.css',
            array(),
            $this->_version
        );
        wp_enqueue_style( $this->_token . '-admin' );
    }

    // End admin_enqueue_styles ()
    /**
     * Load admin Javascript.
     *
     * @access  public
     *
     * @param string $hook Hook parameter.
     *
     * @return  void
     * @since   1.0.0
     */
    public function admin_enqueue_scripts( $hook = '' ) {
        wp_register_script(
            $this->_token . '-admin',
            esc_url( $this->assets_url ) . 'js/admin' . $this->script_suffix . '.js',
            array('jquery'),
            $this->_version,
            true
        );
        wp_enqueue_script( $this->_token . '-admin' );
    }

    // End admin_enqueue_scripts ()
    /**
     * Load plugin localisation
     *
     * @access  public
     * @return  void
     * @since   1.0.0
     */
    public function load_localisation() {
        load_plugin_textdomain( 'guest-author-affiliate', false, dirname( plugin_basename( $this->file ) ) . '/lang/' );
    }

    // End load_localisation ()
    public function _loop_start( $query ) {
        // @see https://wordpress.stackexchange.com/a/141269
        if ( $query->is_main_query() ) {
            add_filter( 'the_content', array($this, '_the_content') );
            add_action( 'loop_end', array($this, '_loop_end') );
        }
    }

    public function WPAM_woo_override_refkey( $wpam_refkey, $order ) {
        return $wpam_refkey;
    }

    public function WPAM_get_affiliate_token( $aff_id, $authorId ) {
        if ( !is_null( $aff_id ) ) {
            // error_log('WPAM_get_affiliate_token: aff_id is not null');
            return $aff_id;
        }
        if ( !defined( 'WPAM_AFFILIATES_TBL' ) ) {
            // error_log('guest_author_affiliate: WPAM_AFFILIATES_TBL is not defined');
            return $aff_id;
        }
        // 0. skip processing if it would be processed in the https://wordpress.org/plugins/affiliates-manager/ plugin itself
        if ( isset( $_REQUEST[WPAM_PluginConfig::$RefKey] ) && !empty( $_REQUEST[WPAM_PluginConfig::$RefKey] ) ) {
            //this is the old tracking code (deprecated). This will be removed once the new tracking system is functional.
            // click already registered in the plugin itself
            WPAM_Logger::log_debug( 'click already registered in the plugin itself' );
            return $aff_id;
        }
        //this will be the new affiliate link. A click will be tracked when wpam_id is present in the URL
        if ( isset( $_REQUEST[WPAM_PluginConfig::$wpam_id] ) && !empty( $_REQUEST[WPAM_PluginConfig::$wpam_id] ) ) {
            // click already registered in the plugin itself
            WPAM_Logger::log_debug( 'click already registered in the plugin itself: new way' );
            return $aff_id;
        }
        global $wpdb;
        $query = "SELECT * FROM " . WPAM_AFFILIATES_TBL . " WHERE userId = %d";
        $affiliate = $wpdb->get_row( $wpdb->prepare( $query, $authorId ) );
        if ( is_null( $affiliate ) || $affiliate->status != "active" ) {
            WPAM_Logger::log_debug( 'page author is not registered as an affiliate' );
            return;
        }
        $aff_id = $affiliate->affiliateId;
        return $aff_id;
    }

    public function WPAM_set_affiliation_coockie_handler( $aff_id ) {
        if ( !defined( 'WPAM_AFFILIATES_TBL' ) ) {
            // error_log('guest_author_affiliate: WPAM_AFFILIATES_TBL is not defined');
            return;
        }
        $aff_id = intval( $aff_id );
        $this->_wpam_register_click( $aff_id );
    }

    public function YITH_WCAF_get_affiliate_token( $aff_id, $authorId ) {
        return $aff_id;
    }

    public function YITH_WCAF_set_affiliation_coockie_handler( $aff_id ) {
    }

    public function _the_content( $content ) {
        $authorId = get_the_author_meta( 'ID' );
        // @see https://stackoverflow.com/a/27884391
        if ( empty( $authorId ) ) {
            // error_log('empty author id for page');
            return $content;
        }
        $aff_id = apply_filters( 'guest_author_affiliate_get_affiliate_token', null, $authorId );
        if ( is_null( $aff_id ) ) {
            // error_log('guest_author_affiliate: aff_id is null');
            return $content;
        }
        $nonce = wp_create_nonce( 'guest_author_affiliate_set_affiliation_coockie_nonce_' . $aff_id );
        ob_start();
        ?>
		<script  type='text/javascript'>
		<!--
		// When the document loads do everything inside here ...
		jQuery(document).ready(function(){
			jQuery.ajax({
				type: "post",url: "<?php 
        echo admin_url( 'admin-ajax.php', 'admin' );
        ?>",data: { action: 'guest_author_affiliate_set_affiliation_coockie', aff_id: <?php 
        echo esc_attr( $aff_id );
        ?>, _ajax_nonce: '<?php 
        echo esc_attr( $nonce );
        ?>' },
				success: function(html){ //so, if data is retrieved, store it in html
				}
			}); //close jQuery.ajax(
		})
		-->
		</script>
		<?php 
        $s = ob_get_flush();
        $content .= $s;
        return $content;
    }

    public function set_affiliation_coockie() {
        if ( !isset( $_POST['aff_id'] ) || empty( $_POST['aff_id'] ) ) {
            error_log( 'set_affiliation_coockie: empty aff id for page' );
            ?>ERROR<?php 
            die;
            return;
        }
        $aff_id = sanitize_key( $_POST['aff_id'] );
        check_ajax_referer( 'guest_author_affiliate_set_affiliation_coockie_nonce_' . $aff_id );
        do_action( 'guest_author_affiliate_set_affiliation_coockie_handler', $aff_id );
        ?>SUCCESS<?php 
        die;
    }

    public function _loop_end() {
        remove_action( 'the_content', array($this, '_the_content') );
    }

    private function _wpam_register_click( $aff_id ) {
        $cookie_life_time = wpam_get_cookie_life_time();
        setcookie(
            'wpam_id',
            $aff_id,
            $cookie_life_time,
            "/",
            COOKIE_DOMAIN
        );
        $args = array();
        $args['dateCreated'] = date( "Y-m-d H:i:s", time() );
        $args['sourceAffiliateId'] = $aff_id;
        $args['trackingKey'] = uniqid();
        //save a unique ID to avoid error
        $args['sourceCreativeId'] = '';
        // remove this column from the click tracking menu in the settings
        $args['referer'] = ( isset( $_SERVER['HTTP_REFERER'] ) ? $_SERVER['HTTP_REFERER'] : '' );
        $args['affiliateSubCode'] = '';
        $args['ipAddress'] = WPAM_Click_Tracking::get_user_ip();
        WPAM_Logger::log_debug( 'guest_author_affiliate inserting click: ' . print_r( $args, true ) );
        WPAM_Click_Tracking::insert_click_data( $args );
    }

    /**
     * Load plugin textdomain
     *
     * @access  public
     * @return  void
     * @since   1.0.0
     */
    public function load_plugin_textdomain() {
        $domain = 'guest-author-affiliate';
        $locale = apply_filters( 'plugin_locale', get_locale(), $domain );
        load_textdomain( $domain, WP_LANG_DIR . '/' . $domain . '/' . $domain . '-' . $locale . '.mo' );
        load_plugin_textdomain( $domain, false, dirname( plugin_basename( $this->file ) ) . '/lang/' );
    }

    // End load_plugin_textdomain ()
    /**
     * Main Guest_Author_Affiliate Instance
     *
     * Ensures only one instance of Guest_Author_Affiliate is loaded or can be loaded.
     *
     * @param string $file File instance.
     * @param string $version Version parameter.
     *
     * @return Object Guest_Author_Affiliate instance
     * @see Guest_Author_Affiliate()
     * @since 1.0.0
     * @static
     */
    public static function instance( $file = '', $version = '1.0.0' ) {
        if ( is_null( self::$_instance ) ) {
            self::$_instance = new self($file, $version);
        }
        return self::$_instance;
    }

    // End instance ()
    /**
     * Cloning is forbidden.
     *
     * @since 1.0.0
     */
    public function __clone() {
        _doing_it_wrong( __FUNCTION__, esc_html( __( 'Cloning of Guest_Author_Affiliate is forbidden' ) ), esc_attr( $this->_version ) );
    }

    // End __clone ()
    /**
     * Unserializing instances of this class is forbidden.
     *
     * @since 1.0.0
     */
    public function __wakeup() {
        _doing_it_wrong( __FUNCTION__, esc_html( __( 'Unserializing instances of Guest_Author_Affiliate is forbidden' ) ), esc_attr( $this->_version ) );
    }

    // End __wakeup ()
    /**
     * Installation. Runs on activation.
     *
     * @access  public
     * @return  void
     * @since   1.0.0
     */
    public function install() {
        $this->_log_version_number();
    }

    // End install ()
    /**
     * Log the plugin version number.
     *
     * @access  public
     * @return  void
     * @since   1.0.0
     */
    private function _log_version_number() {
        //phpcs:ignore
        update_option( $this->_token . '_version', $this->_version );
    }

    // End _log_version_number ()
}
