<?php
if (!defined('ABSPATH')) exit;

// Enqueue styles and scripts
add_action('admin_enqueue_scripts', function ($hook) {
    if ($hook !== 'toplevel_page_gsseo-meta-generator') return;
    
    wp_enqueue_style(
        'gsseo-general-report-style',
        plugin_dir_url(__FILE__) . 'assets/css/general-report.css',
        [],
        filemtime(plugin_dir_path(__FILE__) . 'assets/css/general-report.css')
    );

    // Load Chart.js locally
    wp_enqueue_script(
        'chartjs',
        plugins_url('assets/js/chart.js', __FILE__),
        [],
        '4.4.0',
        true
    );
});

function gsseo_settings_page() {
    $selected_type = isset($_GET['post_type_filter']) ? sanitize_text_field(wp_unslash($_GET['post_type_filter'])) : 'all';
    $post_types = ['post', 'page'];
    $args = [
        'post_type'      => ($selected_type === 'all') ? $post_types : [$selected_type],
        'post_status'    => 'publish',
        'posts_per_page' => -1,
        'no_found_rows'  => true,
        'update_post_meta_cache' => true,
        'update_post_term_cache' => false,
    ];
    
    $posts = get_posts($args);
    $score_counts = ['high' => 0, 'medium' => 0, 'low' => 0];
    $scored_posts = ['high' => [], 'medium' => [], 'low' => []];

    foreach ($posts as $post) {
        $post_id = $post->ID;
        
        // Clear any cached meta data to ensure fresh values
        wp_cache_delete($post_id, 'post_meta');
        
        // Initialize with defaults
        $title = get_the_title($post_id);
        $description = '';
        
        // Get description fallback
        if (has_excerpt($post_id)) {
            $description = get_the_excerpt($post_id);
        } else {
            $content = strip_shortcodes($post->post_content);
            $content = wp_strip_all_tags($content);
            $description = wp_trim_words($content, 25);
        }

        // FIRST check for GSSEO generated meta (priority)
        $gsseo_title = get_post_meta($post_id, 'gsseo_generated_title', true);
        $gsseo_desc = get_post_meta($post_id, 'gsseo_generated_description', true);
        
        // Only check other plugins if GSSEO meta doesn't exist
        if (empty($gsseo_title) || empty($gsseo_desc)) {
            $active_seo_plugin = gsseo_detect_active_seo_plugin();
            
            switch ($active_seo_plugin) {
                case 'yoast':
                    $yoast_title = get_post_meta($post_id, '_yoast_wpseo_title', true);
                    $yoast_desc = get_post_meta($post_id, '_yoast_wpseo_metadesc', true);
                    
                    if (!empty($yoast_title)) {
                        if (function_exists('wpseo_replace_vars')) {
                            global $post;
                            $post = get_post($post_id);
                            setup_postdata($post);
                            $title = wpseo_replace_vars($yoast_title, $post);
                            wp_reset_postdata();
                        } else {
                            $title = $yoast_title;
                        }
                    }
                    
                    if (!empty($yoast_desc)) {
                        if (function_exists('wpseo_replace_vars')) {
                            global $post;
                            $post = get_post($post_id);
                            setup_postdata($post);
                            $description = wpseo_replace_vars($yoast_desc, $post);
                            wp_reset_postdata();
                        } else {
                            $description = $yoast_desc;
                        }
                    }
                    break;
                    
                case 'rankmath':
                    $rank_title = get_post_meta($post_id, 'rank_math_title', true);
                    $rank_desc = get_post_meta($post_id, 'rank_math_description', true);
                    $title = !empty($rank_title) ? $rank_title : $title;
                    $description = !empty($rank_desc) ? $rank_desc : $description;
                    break;
                    
                case 'aioseo':
                    $aio_title = get_post_meta($post_id, '_aioseo_title', true);
                    $aio_desc = get_post_meta($post_id, '_aioseo_description', true);
                    $title = !empty($aio_title) ? $aio_title : $title;
                    $description = !empty($aio_desc) ? $aio_desc : $description;
                    break;
                    
                case 'seopress':
                    $seo_title = get_post_meta($post_id, '_seopress_titles_title', true);
                    $seo_desc = get_post_meta($post_id, '_seopress_titles_desc', true);
                    $title = !empty($seo_title) ? $seo_title : $title;
                    $description = !empty($seo_desc) ? $seo_desc : $description;
                    break;
            }
        }
        
        // Use GSSEO meta if available, otherwise use whatever we got from plugins/fallback
        $title = !empty($gsseo_title) ? $gsseo_title : $title;
        $description = !empty($gsseo_desc) ? $gsseo_desc : $description;
        
        // Clean and trim the values
        $title = wp_strip_all_tags(trim($title));
        $description = wp_strip_all_tags(trim($description));
        
        // Calculate lengths
        $title_length = mb_strlen($title);
        $desc_length = mb_strlen($description);
        
        // Score Calculation
        $score = 0;
        if ($title_length >= 50 && $title_length <= 60) $score += 2;
        elseif (($title_length >= 30 && $title_length < 50) || ($title_length > 60 && $title_length <= 70)) $score += 1;

        if ($desc_length >= 150 && $desc_length <= 160) $score += 2;
        elseif (($desc_length >= 120 && $desc_length < 150) || ($desc_length > 160 && $desc_length <= 300)) $score += 1;

        // Categorize the score
        if ($score >= 3) {
            $score_category = 'high';
        } elseif ($score >= 1) {
            $score_category = 'medium';
        } else {
            $score_category = 'low';
        }

        $score_counts[$score_category]++;
        $scored_posts[$score_category][] = [
            'post' => $post,
            'title_length' => $title_length,
            'desc_length' => $desc_length,
            'score' => $score,
            'title_text' => $title,
            'desc_text' => $description,
        ];
    }

    $total = array_sum($score_counts);
    $percent = fn($count) => $total ? round(($count / $total) * 100) : 0;
    $bulk_generator_url = admin_url('admin.php?page=gsseo-title-meta');
    ?>
    <div class="wrap">
        <h1 style="margin-bottom: 30px;">GSSEO: SEO Title & Description Health Score</h1>
        <a target="_blank" href="<?php echo esc_url('https://aboutgaurishankar.wordpress.com/contact/'); ?>"
           style="animation: blink-animation 10s steps(10, start) infinite;
                  color: #0008ff;
                  text-decoration: underline;
                  font-size: 20px;
                  font-weight: 500;">
           Contact Us for Custom Support
        </a>

        <style>
        @keyframes blink-animation {
            to { visibility: hidden; }
        }
        </style>

        <form method="get" style="margin-bottom: 30px;">
            <input type="hidden" name="page" value="gsseo-meta-generator">
            <label for="post_type_filter"><strong>Filter by Post Type:</strong></label>
            <select name="post_type_filter" id="post_type_filter" onchange="this.form.submit();">
                <option value="all" <?php selected($selected_type, 'all'); ?>>All</option>
                <option value="post" <?php selected($selected_type, 'post'); ?>>Posts</option>
                <option value="page" <?php selected($selected_type, 'page'); ?>>Pages</option>
            </select>
        </form>

        <div class="gsseo-two-column-wrapper">
            <div class="gsseo-left-column">
                <?php foreach (['high', 'medium', 'low'] as $type): ?>
                    <div class="gsseo-box <?php echo esc_attr($type); ?>">
                        <strong><?php echo esc_html(ucfirst($type)); ?> Score</strong>
                         <?php echo esc_html($score_counts[$type]); ?> (<?php echo esc_html($percent($score_counts[$type])); ?>%)
                        <div class="gsseo-progress">
                            <div class="gsseo-progress-bar" style="width: <?php echo esc_attr($percent($score_counts[$type])); ?>%;"></div>
                        </div>
                        <?php if (!empty($scored_posts[$type])): ?>
                            <button class="toggle-table-button" data-target="table-<?php echo esc_attr($type); ?>">
                                <?php echo $type === 'high' ? 'Hide List' : 'View List'; ?>
                            </button>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>

                <canvas id="seoScoreChart" style="width: 50%; height: 350px; margin-bottom: 30px;"></canvas>

                <div class="gsseo-score-info">
                    <h3>Scoring Criteria:</h3>
                    <ul>
                        <li><strong>High Score (3-4 points):</strong> Title 50-60 chars, Description 150-160 chars</li>
                        <li><strong>Medium Score (1-2 points):</strong> Title 30-49 or 61-70, Description 120-149 or 161-300</li>
                        <li><strong>Low Score (0 points):</strong> Title &lt;30 or &gt;70, Description &lt;120 or &gt;300</li>
                    </ul>
                </div>
            </div>

            <div class="gsseo-right-column">
                <?php foreach (['high', 'medium', 'low'] as $type): ?>
                    <?php if (!empty($scored_posts[$type])): ?>
                        <div class="score-table-wrapper" id="table-<?php echo esc_attr($type); ?>" style="display: <?php echo $type === 'high' ? 'block' : 'none'; ?>;">
                            <button class="go-to-bulk-generator" data-url="<?php echo esc_url($bulk_generator_url); ?>">Go to Bulk Generator</button>
                            <h2>
                              <?php echo esc_html(ucfirst($type)); ?> Score - 
                              <?php echo esc_html(($selected_type === 'all') ? 'Posts/Pages' : ucfirst($selected_type) . 's'); ?>
                            </h2>
                            <table class="widefat fixed striped">
                                <thead>
                                    <tr>
                                        <th>Title</th>
                                        <th>Post Type</th>
                                        <th>Title Length</th>
                                        <th>Desc Length</th>
                                        <th>Score</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($scored_posts[$type] as $item): ?>
                                        <tr>
                                            <td><?php echo esc_html(get_the_title($item['post'])); ?></td>
                                            <td><?php echo esc_html(ucfirst(get_post_type($item['post']))); ?></td>
                                            <td><?php echo esc_html($item['title_length']); ?></td>
                                            <td><?php echo esc_html($item['desc_length']); ?></td>
                                            <td>
                                                <span class="score-badge score-<?php echo esc_attr($type); ?>">
                                                <?php echo esc_html($item['score']); ?>/4
                                                </span>
                                            </td>
                                            <td>
                                               <a href="<?php echo esc_url(get_permalink($item['post'])); ?>" target="_blank">View</a> |
                                                <a href="<?php echo esc_url(get_edit_post_link($item['post'])); ?>" target="_blank">Edit</a>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                <?php endforeach; ?>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener("DOMContentLoaded", function () {
            new Chart(document.getElementById('seoScoreChart').getContext('2d'), {
                type: 'doughnut',
                data: {
                    labels: ['High Score', 'Medium Score', 'Low Score'],
                    datasets: [{
                        data: [<?php echo (int)$score_counts['high']; ?>, <?php echo (int)$score_counts['medium']; ?>, <?php echo (int)$score_counts['low']; ?>],
                        backgroundColor: ['#28a745', '#ffc107', '#dc3545']
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: { position: 'bottom' },
                        title: {
                            display: true,
                            text: 'SEO Health Score Distribution (<?php echo esc_js($total); ?> <?php echo esc_js(($selected_type === 'all') ? "Posts/Pages" : ucfirst($selected_type) . "s"); ?>)'
                        }
                    }
                }
            });

            document.querySelectorAll('.toggle-table-button').forEach(btn => {
                btn.addEventListener('click', function () {
                    const targetId = this.getAttribute('data-target');
                    const table = document.getElementById(targetId);
                    const isVisible = table.style.display === 'block';
                    document.querySelectorAll('.score-table-wrapper').forEach(t => t.style.display = 'none');
                    document.querySelectorAll('.toggle-table-button').forEach(b => b.textContent = 'View List');
                    table.style.display = isVisible ? 'none' : 'block';
                    this.textContent = isVisible ? 'View List' : 'Hide List';
                });
            });

            jQuery(document).on('click', '.go-to-bulk-generator', function () {
                window.location.href = jQuery(this).data('url');
            });
        });
    </script>

    <style>
        .score-badge {
            padding: 3px 8px;
            border-radius: 10px;
            font-weight: bold;
            font-size: 12px;
            color: white;
        }
        .score-high { background-color: #28a745; }
        .score-medium { background-color: #ffc107; color: #000; }
        .score-low { background-color: #dc3545; }

        .gsseo-score-info {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 5px;
            margin-top: 20px;
            border-left: 4px solid #0073aa;
        }
    </style>
<?php
}

/**
 * Detect which SEO plugin is active
 */
function gsseo_detect_active_seo_plugin() {
    if (defined('YOASTSEO_VERSION')) {
        return 'yoast';
    } elseif (defined('RANK_MATH_VERSION')) {
        return 'rankmath';
    } elseif (defined('AIOSEO_VERSION')) {
        return 'aioseo';
    } elseif (defined('SEOPRESS_VERSION')) {
        return 'seopress';
    }
    return false;
}