jQuery(document).ready(function ($) {
    // Show inline message under a row
    function setInlineMessage(postId, text, type = 'info') {
        const $row = $(`#gsseo-meta-title-${postId}`).closest('tr');
        $row.find('.gsseo-inline-message').remove();

        const $message = $(`<div class="gsseo-inline-message gsseo-${type}">${text}</div>`);
        $row.find('td:last').append($message);

        if (type !== 'error') {
            setTimeout(() => {
                $message.fadeOut(300, function () { $(this).remove(); });
            }, 5000);
        }
    }

    // Refresh a single row with updated data
    function refreshRow(postId) {
        const postType = $('#gsseo-post-type-filter').val();
        const search = $('#gsseo-search').val();
        const currentPage = $('.gsseo-page-btn.active').data('page') || 1;

        $.post(gsseo_ajax.ajax_url, {
            action: 'gsseo_get_all_posts',
            nonce: gsseo_ajax.nonce,
            type: postType,
            search: search,
            page: currentPage
        }, function (res) {
            if (res.success) {
                const post = res.data.posts.find(p => p.id == postId);
                if (post) {
                    const $row = $(`#gsseo-meta-title-${postId}`).closest('tr');
                    $row.replaceWith(renderRow(post));
                }
            }
        });
    }

    function renderRow(post) {
        const titleHtml = `
            <strong>${post.title}</strong><br>
            <a href="${post.edit_link}" target="_blank">Edit</a> |
            <a href="${post.view_link}" target="_blank">View</a>
        `;

        return `
            <tr>
                <td>${titleHtml}</td>
                <td>
                    <textarea id="gsseo-meta-title-${post.id}" class="gsseo-meta-title form-control" data-post-id="${post.id}">${post.meta_title || ''}</textarea>
                </td>
                <td>
                    <textarea id="gsseo-meta-description-${post.id}" class="gsseo-meta-description form-control" data-post-id="${post.id}">${post.meta_description || ''}</textarea>
                </td>
                <td>
                    <button class="btn btn-primary gsseo-save-meta" data-post-id="${post.id}">Save</button>
                    <button class="btn btn-secondary gsseo-generate-ai-meta" data-post-id="${post.id}">Generate AI</button>
                </td>
            </tr>
        `;
    }

    function renderPagination(currentPage, totalPages) {
        let paginationHtml = '<div class="gsseo-pagination">';
        
        // Previous button
        if (currentPage > 1) {
            paginationHtml += `<button class="gsseo-page-btn" data-page="${currentPage - 1}">&laquo; Previous</button>`;
        }
        
        // Page numbers
        const maxVisiblePages = 5;
        let startPage = Math.max(1, currentPage - Math.floor(maxVisiblePages / 2));
        let endPage = Math.min(totalPages, startPage + maxVisiblePages - 1);
        
        if (endPage - startPage < maxVisiblePages - 1) {
            startPage = Math.max(1, endPage - maxVisiblePages + 1);
        }
        
        if (startPage > 1) {
            paginationHtml += `<button class="gsseo-page-btn" data-page="1">1</button>`;
            if (startPage > 2) {
                paginationHtml += '<span class="gsseo-page-dots">...</span>';
            }
        }
        
        for (let i = startPage; i <= endPage; i++) {
            paginationHtml += `<button class="gsseo-page-btn ${i === currentPage ? 'active' : ''}" data-page="${i}">${i}</button>`;
        }
        
        if (endPage < totalPages) {
            if (endPage < totalPages - 1) {
                paginationHtml += '<span class="gsseo-page-dots">...</span>';
            }
            paginationHtml += `<button class="gsseo-page-btn" data-page="${totalPages}">${totalPages}</button>`;
        }
        
        // Next button
        if (currentPage < totalPages) {
            paginationHtml += `<button class="gsseo-page-btn" data-page="${currentPage + 1}">Next &raquo;</button>`;
        }
        
        paginationHtml += '</div>';
        return paginationHtml;
    }

    // Load posts and populate table
    function loadPosts(postType = 'all', search = '', page = 1) {
        const $table = $('#gsseo-posts-table');
        const $pagination = $('.gsseo-pagination-container');
        const $countContainer = $('.gsseo-count-container');
        $table.html('<tr><td colspan="4">Loading...</td></tr>');
        showLoader();

        $.post(gsseo_ajax.ajax_url, {
            action: 'gsseo_get_all_posts',
            nonce: gsseo_ajax.nonce,
            type: postType,
            search: search,
            page: page
        }, function (res) {
            $table.empty();
            $pagination.empty();
            $countContainer.empty();
            hideLoader();

            if (res.success && res.data.posts.length > 0) {
                res.data.posts.forEach(post => $table.append(renderRow(post)));
                $countContainer.html(renderItemCount(
                res.data.pagination.current_page,
                10, // matches the $per_page in PHP
                res.data.pagination.total_posts
            ));
                if (res.data.pagination.total_pages > 1) {
                    $pagination.html(renderPagination(
                        res.data.pagination.current_page,
                        res.data.pagination.total_pages
                    ));
                }
            } else {
                $table.append('<tr><td colspan="4">No posts found.</td></tr>');
                $countContainer.html('<div class="gsseo-item-count">0 items</div>'); 
            }
        }).fail(() => {
            hideLoader();
            alert('Failed to load posts. Please refresh the page.');
        });
    }

    // Post type filter
    $('#gsseo-post-type-filter').on('change', function () {
        loadPosts($(this).val(), $('#gsseo-search').val(), 1);
    });

    // Search filter
    $('#gsseo-search').on('input', function () {
        loadPosts($('#gsseo-post-type-filter').val(), $(this).val(), 1);
    });

    // Save meta
    $(document).on('click', '.gsseo-save-meta', function () {
        const postId = $(this).data('post-id');
        const title = $(`#gsseo-meta-title-${postId}`).val();
        const description = $(`#gsseo-meta-description-${postId}`).val();

        if (!postId) return;

        setInlineMessage(postId, 'Saving...', 'info');

        $.post(gsseo_ajax.ajax_url, {
            action: 'gsseo_save_post_meta',
            nonce: gsseo_ajax.nonce,
            post_id: postId,
            title,
            description
        }, function (res) {
            if (res.success) {
                setInlineMessage(postId, `Meta updated successfully using ${res.data.plugin || 'your SEO plugin'}.`, 'success');
                refreshRow(postId);
            } else {
                setInlineMessage(postId, 'Failed to save meta data.', 'error');
            }
        }).fail(() => {
            setInlineMessage(postId, 'Failed to save meta data. Please try again.', 'error');
        });
    });

    // Generate AI meta
    $(document).on('click', '.gsseo-generate-ai-meta', function () {
        const postId = $(this).data('post-id');
        if (!postId) return;

        setInlineMessage(postId, 'Generating AI meta data...', 'info');

        $.post(gsseo_ajax.ajax_url, {
            action: 'gsseo_generate_ai_meta',
            nonce: gsseo_ajax.nonce,
            post_id: postId
        }, function (res) {
            if (res.success) {
                $(`#gsseo-meta-title-${postId}`).val(res.data.title || '');
                $(`#gsseo-meta-description-${postId}`).val(res.data.description || '');
                setInlineMessage(postId, 'AI-generated meta has been applied!', 'success');
            }
 else {
                setInlineMessage(postId, res.data?.message || 'AI generation failed.', 'error');
            }
        }).fail((xhr) => {
            const errorMsg = xhr.responseJSON?.data?.message || xhr.responseText || 'Failed to generate AI meta.';
            setInlineMessage(postId, errorMsg, 'error');
        });
    });


    // Pagination click handler
    $(document).on('click', '.gsseo-page-btn', function() {
        const page = $(this).data('page');
        loadPosts($('#gsseo-post-type-filter').val(), $('#gsseo-search').val(), page);
    });

    // Initial load
    loadPosts();

    // Global loader functions
    function showLoader() { $('#gsseo-loading-overlay').fadeIn(200); }
    function hideLoader() { $('#gsseo-loading-overlay').fadeOut(200); }
});

function renderItemCount(currentPage, perPage, totalPosts) {
    const start = ((currentPage - 1) * perPage) + 1;
    const end = Math.min(currentPage * perPage, totalPosts);
    return `<div class="gsseo-item-count">Showing ${start}-${end} of ${totalPosts} items</div>`;
}