<?php

namespace GS_BOOKS;

use function GS_BOOKS_PRO\is_pro_valid;

defined( 'ABSPATH' ) || exit;

class ShortcodeBuilder {

	private $option_name          = 'gs_bookshowcase_settings';
	private $level_option_name    = 'gs_bookshowcase_level_settings';
	private $taxonomy_option_name = 'gs_books_taxonomy_settings';
	private $fields_visibility_option_name = 'gs_books_fields_visibility_settings';

	public function __construct() {

		add_action( 'admin_menu', array( $this, 'register_sub_menu' ), 30 );
		add_action( 'admin_enqueue_scripts', array( $this, 'scripts' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'preview_scripts' ) );

		add_action( 'wp_ajax_gs_books_create_shortcode', array( $this, 'create_shortcode' ) );
		add_action( 'wp_ajax_gs_books_clone_shortcode', array( $this, 'clone_shortcode' ) );
		add_action( 'wp_ajax_gs_books_get_shortcode', array( $this, 'get_shortcode' ) );
		add_action( 'wp_ajax_gs_books_update_shortcode', array( $this, 'update_shortcode' ) );
		add_action( 'wp_ajax_gs_books_delete_shortcodes', array( $this, 'delete_shortcodes' ) );
		add_action( 'wp_ajax_gs_books_temp_save_shortcode_settings', array( $this, 'temp_save_shortcode_settings' ) );
		add_action( 'wp_ajax_gs_books_get_shortcodes', array( $this, 'get_shortcodes' ) );

		add_action( 'wp_ajax_gs_books_get_shortcode_pref', array( $this, 'get_shortcode_pref' ) );
		add_action( 'wp_ajax_gs_books_save_shortcode_pref', array( $this, 'save_shortcode_pref' ) );

		add_action( 'wp_ajax_gs_books_get_localization', array( $this, 'get_localization' ) );
		add_action( 'wp_ajax_gs_books_save_localization', array( $this, 'save_localization' ) );

		add_action( 'wp_ajax_gsbooks_get_taxonomy_settings', array($this, 'get_taxonomy_settings') );
		add_action( 'wp_ajax_gsbooks_save_taxonomy_settings', array($this, 'save_taxonomy_settings') );

		add_action( 'wp_ajax_gsbooks_get_fields_visibility_settings', array($this, 'get_fields_visibility_settings') );
		add_action( 'wp_ajax_gsbooks_save_fields_visibility_settings', array($this, 'save_fields_visibility_settings') );

		add_action( 'template_include', array( $this, 'populate_shortcode_preview' ) );
		add_action( 'show_admin_bar', array( $this, 'hide_admin_bar_from_preview' ) );

		// add_filter( 'body_class', array($this, 'add_shortcode_body_class') );

		return $this;
	}

	public function is_gs_books_shortcode_preview() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		return isset( $_REQUEST['gs_books_shortcode_preview'] ) && ! empty( $_REQUEST['gs_books_shortcode_preview'] );
	}

	public function hide_admin_bar_from_preview( $visibility ) {
		if ( $this->is_gs_books_shortcode_preview() ) {
			return false;
		}

		return $visibility;
	}

	public function get_taxonomy_settings() {
		return $this->_get_taxonomy_settings( wp_doing_ajax() );
	}

	public function get_fields_visibility_settings() {
		return $this->_get_fields_visibility_settings( wp_doing_ajax() );
	}

	public function _get_fields_visibility_settings( $is_ajax ) {

		$settings = (array) get_option( $this->fields_visibility_option_name, [] );
		$defaults = $this->get_visibility_defaults( $this->detail_visibility_settings_exclude() );

		$settings = shortcode_atts( $defaults, $settings );

		$settings = $this->validate_fields_visibility_settings( $settings );

		if ( $is_ajax ) {
			wp_send_json_success( $settings );
		}

		return $settings;

	}

	public function get_sorted_fields_visibility_settings() {

		$exclude_visibility = $this->detail_visibility_settings_exclude();

		// Remove book_related_section from $exclude_visibility to keep it in the list
		$exclude_visibility = array_filter($exclude_visibility, function($item) {
			return $item !== 'book_related_section';
		});

		$fields_visibility = $this->get_visibility_defaults( $exclude_visibility );

		$fields_visibility_saved = get_option( 'gs_book_fields_visibility_order', [] );

		if ( !empty($fields_visibility_saved) ) {
			$fields_visibility_merged = array();
			foreach ( $fields_visibility_saved as $field => $values ) {
				if ( ! array_key_exists( $field, $fields_visibility ) ) continue;
				$fields_visibility_merged[ $field ] = array_merge(
					isset( $fields_visibility[ $field ] ) ? $fields_visibility[ $field ] : array(),
					$values
				);
			}
			return array_merge( $fields_visibility_merged, array_diff_key( $fields_visibility, $fields_visibility_merged ) );
		}

		return $fields_visibility;
	}

	function get_scoped_fields( array $keys, array $sorted_fields = [] ) {
		$scoped = array();

		if ( empty( $sorted_fields ) ) {
			$sorted_fields = $this->get_sorted_fields_visibility_settings();
		}

		foreach ( $sorted_fields as $field => $settings ) {
			if ( in_array( $field, $keys, true ) && Helpers::is_visible( $settings ) ) {
				$scoped[ $field ] = $settings;
			}
		}

		return $scoped;
	}

	public function apply_default_visibility_order( $fields ) {

		$default_order = ['book_name', 'book_author_by', 'book_short_description', 'book_publish_date', 'book_co_publisher', 'book_translator', 'book_categories', 'book_tags', 'book_author', 'book_genres', 'book_series', 'book_languages', 'book_publisher', 'book_countries', 'book_isbn', 'book_isbn_thirteen', 'book_asin', 'book_doi', 'book_llcn', 'book_pages', 'book_dimension', 'book_weight', 'book_file_size', 'book_download_url', 'book_rating', 'book_cover', 'book_regular_price', 'book_sale_price', 'book_availability', 'book_preorder_available', 'book_age_group', 'book_reading_level', 'book_edition', 'book_edition_features', 'book_awards', 'book_reading_time', 'book_accessibility_features', 'book_store', 'book_formats', 'book_details', 'readers_review', 'book_author_section', 'book_author_social', 'book_author_img', 'book_related_section'];

		$sorted = array();

		// loop over the default order
		foreach ( $default_order as $key ) {
			if ( isset( $fields[ $key ] ) ) {
				$sorted[ $key ] = $fields[ $key ];
			}
		}

		return $sorted;
	}

	public function validate_fields_visibility_settings( $settings ) {

		$visibility_defaults = $this->get_visibility_defaults();

		foreach( $settings as $setting_key => &$setting ) {
			if ( array_key_exists( 'translation_key', $setting ) ) unset( $setting['translation_key'] );
			$setting = shortcode_atts( $visibility_defaults[ $setting_key ], $setting );
			$setting['desktop'] = wp_validate_boolean( $setting['desktop'] );
			$setting['tablet'] = wp_validate_boolean( $setting['tablet'] );
			$setting['mobile_landscape'] = wp_validate_boolean( $setting['mobile_landscape'] );
			$setting['mobile'] = wp_validate_boolean( $setting['mobile'] );
		}

		return $settings;

	}

	public function get_localization() {

		$localizations = $this->_get_localization( wp_doing_ajax() );

		if ( wp_doing_ajax() ) {
			wp_send_json_success( $localizations );
		}

		return $localizations;
	}

	public function _get_taxonomy_settings( $is_ajax ) {

		$settings = (array) get_option( $this->taxonomy_option_name, [] );
		$settings = $this->validate_taxonomy_settings( $settings );

		if ( $is_ajax ) {
			wp_send_json_success( $settings );
		}

		return $settings;

	}

	public function validate_taxonomy_settings( $settings ) {

		$defaults = $this->get_taxonomy_default_settings();

		if ( empty($settings) ) {
			$settings = $defaults;
		} else {
			foreach ( $settings as $setting_key => $setting_val ) {
				if ( str_contains($setting_key, '_label') && empty($setting_val) ) {
					$settings[$setting_key] = $defaults[$setting_key];
				}
			}
		}
		
		return array_map( 'sanitize_text_field', $settings );
	}

	public function get_tax_option( $option, $default = '' ) {
		$options  = (array) get_option( $this->taxonomy_option_name, [] );
		$defaults = $this->get_taxonomy_default_settings();
		$options  = shortcode_atts($defaults, $options);

		if ( str_contains($option, '_label') && ( getoption('gs_member_enable_multilingual', 'off') == 'on' ) ) {
			return $defaults[$option];
		}

		if ( str_contains($option, '_label') && empty($options[$option]) ) {
			return $defaults[$option] ?? $default;
		}

		if ( isset($options[$option]) ) return $options[$option];
		return $default;
	}

	public function save_taxonomy_settings() {
		
		check_ajax_referer( '_gs_books_admin_nonce_gs_' );
		
		if ( empty($_POST['tax_settings']) ) {
			wp_send_json_error( __('No settings provided', 'gs-books-showcase'), 400 );
		}

		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$this->_save_taxonomy_settings( $_POST['tax_settings'], true );
	}

	public function _save_taxonomy_settings( $settings, $is_ajax ) {

		if ( empty($settings) ) $settings = [];

		$settings = $this->validate_taxonomy_settings( $settings );
		update_option( $this->taxonomy_option_name, $settings, 'yes' );
		
		// Clean permalink flush
		delete_option( 'GS_book_permalinks_flushed' );

		do_action( 'gs_book_tax_settings_update' );
		do_action( 'gsp_tax_settings_update' );
	
		if ( $is_ajax ) wp_send_json_success( __('Taxonomy settings saved', 'gs-books-showcase') );
	}

	public function add_shortcode_body_class( $classes ) {
		if ( $this->is_gs_books_shortcode_preview() ) {
			return array_merge( $classes, array( 'gsbooks-shortcode-preview--page' ) );
		}

		return $classes;
	}

	public function populate_shortcode_preview( $template ) {
		global $wp, $wp_query;

		if ( $this->is_gs_books_shortcode_preview() ) {

			// Create our fake post
			$post_id              = wp_rand( 1, 99999 ) - 9999999;
			$post                 = new \stdClass();
			$post->ID             = $post_id;
			$post->post_author    = 1;
			$post->post_date      = current_time( 'mysql' );
			$post->post_date_gmt  = current_time( 'mysql', 1 );
			$post->post_title     = esc_html__( 'Shortcode Preview', 'gs-books-showcase' );
			$post->post_content   = sprintf( '[gs_bookshowcase preview="yes" id="%s"]', esc_attr( sanitize_key( $_REQUEST['gs_books_shortcode_preview'] ) ) );
			$post->post_status    = 'publish';
			$post->comment_status = 'closed';
			$post->ping_status    = 'closed';
			$post->post_name      = 'fake-page-' . wp_rand( 1, 99999 ); // append random number to avoid clash
			$post->post_type      = 'page';
			$post->filter         = 'raw'; // important!

			// Convert to WP_Post object
			$wp_post = new \WP_Post( $post );

			// Add the fake post to the cache
			wp_cache_add( $post_id, $wp_post, 'posts' );

			// Update the main query
			$wp_query->post                 = $wp_post;
			$wp_query->posts                = array( $wp_post );
			$wp_query->queried_object       = $wp_post;
			$wp_query->queried_object_id    = $post_id;
			$wp_query->found_posts          = 1;
			$wp_query->post_count           = 1;
			$wp_query->max_num_pages        = 1;
			$wp_query->is_page              = true;
			$wp_query->is_singular          = true;
			$wp_query->is_single            = false;
			$wp_query->is_attachment        = false;
			$wp_query->is_archive           = false;
			$wp_query->is_category          = false;
			$wp_query->is_tag               = false;
			$wp_query->is_tax               = false;
			$wp_query->is_author            = false;
			$wp_query->is_date              = false;
			$wp_query->is_year              = false;
			$wp_query->is_month             = false;
			$wp_query->is_day               = false;
			$wp_query->is_time              = false;
			$wp_query->is_search            = false;
			$wp_query->is_feed              = false;
			$wp_query->is_comment_feed      = false;
			$wp_query->is_trackback         = false;
			$wp_query->is_home              = false;
			$wp_query->is_embed             = false;
			$wp_query->is_404               = false;
			$wp_query->is_paged             = false;
			$wp_query->is_admin             = false;
			$wp_query->is_preview           = false;
			$wp_query->is_robots            = false;
			$wp_query->is_posts_page        = false;
			$wp_query->is_post_type_archive = false;

			// Update globals
			$GLOBALS['wp_query'] = $wp_query;
			$wp->register_globals();

			include GS_BOOKS_PLUGIN_DIR . 'includes/shortcode-builder/preview.php';

			return;
		}

		return $template;
	}

	public function register_sub_menu() {

		add_submenu_page(
			'edit.php?post_type=gs_bookshowcase',
			__( 'Shortcode', 'gs-books-showcase' ),
			__( 'Shortcode', 'gs-books-showcase' ),
			'manage_options',
			'gs-books-shortcode',
			array( $this, 'view' )
		);

		add_submenu_page(
			'edit.php?post_type=gs_bookshowcase',
			__( 'Install Demo', 'gs-books-showcase' ),
			__( 'Install Demo', 'gs-books-showcase' ),
			'manage_options',
			'gs-books-shortcode#/demo-data',
			array( $this, 'view' ), 20
		);
	}

	public function view() {
		include GS_BOOKS_PLUGIN_DIR . 'includes/shortcode-builder/page.php';
	}

	public function get_book_categories() {
		$_terms = get_terms(
			'bookshowcase_group',
			array(
				'hide_empty' => false,
			)
		);

		$terms = array();

		if ( ! is_wp_error( $_terms ) ) {
			foreach ( $_terms as $term ) {
				$terms[] = array(
					'label' => $term->name,
					'value' => $term->slug,
				);
			}
		}

		return $terms;
	}

	public function get_book_tags() {
		$_terms = get_terms(
			'gsb_tag',
			array(
				'hide_empty' => false,
			)
		);

		$terms = array();

		if ( ! is_wp_error( $_terms ) ) {
			foreach ( $_terms as $term ) {
				$terms[] = array(
					'label' => $term->name,
					'value' => $term->slug,
				);
			}
		}

		return $terms;
	}

	public function get_book_authors() {
		$_terms = get_terms(
			'gsb_author',
			array(
				'hide_empty' => false,
			)
		);

		$terms = array();

		if ( ! is_wp_error( $_terms ) ) {
			foreach ( $_terms as $term ) {
				$terms[] = array(
					'label' => $term->name,
					'value' => $term->slug,
				);
			}
		}

		return $terms;
	}

	public function get_book_genres() {
		$_terms = get_terms(
			'gsb_genre',
			array(
				'hide_empty' => false,
			)
		);

		$terms = array();

		if ( ! is_wp_error( $_terms ) ) {
			foreach ( $_terms as $term ) {
				$terms[] = array(
					'label' => $term->name,
					'value' => $term->slug,
				);
			}
		}

		return $terms;
	}

	public function get_book_series() {
		$_terms = get_terms(
			'gsb_series',
			array(
				'hide_empty' => false,
			)
		);

		$terms = array();

		if ( ! is_wp_error( $_terms ) ) {
			foreach ( $_terms as $term ) {
				$terms[] = array(
					'label' => $term->name,
					'value' => $term->slug,
				);
			}
		}

		return $terms;
	}

	public function get_book_languages() {
		$_terms = get_terms(
			'gsb_languages',
			array(
				'hide_empty' => false,
			)
		);

		$terms = array();

		if ( ! is_wp_error( $_terms ) ) {
			foreach ( $_terms as $term ) {
				$terms[] = array(
					'label' => $term->name,
					'value' => $term->slug,
				);
			}
		}

		return $terms;
	}

	public function get_book_publishers() {

		$_terms = get_terms(
			'gsb_publishers',
			array(
				'hide_empty' => false,
			)
		);

		$terms = array();

		if ( ! is_wp_error( $_terms ) ) {
			foreach ( $_terms as $term ) {
				$terms[] = array(
					'label' => $term->name,
					'value' => $term->slug,
				);
			}
		}

		return $terms;
	}

	public function get_book_countries() {

		$_terms = get_terms(
			'gsb_countries',
			array(
				'hide_empty' => false,
			)
		);

		$terms = array();

		if ( ! is_wp_error( $_terms ) ) {
			foreach ( $_terms as $term ) {
				$terms[] = array(
					'label' => $term->name,
					'value' => $term->slug,
				);
			}
		}

		return $terms;
	}

	public function scripts( $hook ) {

		if ( 'gs_bookshowcase_page_gs-books-shortcode' != $hook ) {
			return;
		}

		wp_register_style(
			'gs-zmdi-fonts',
			GS_BOOKS_PLUGIN_URI . '/assets/libs/material-design-iconic-font/css/material-design-iconic-font.min.css',
			'',
			GS_BOOKS_VERSION,
			'all'
		);

		wp_enqueue_style(
			'gs-books-shortcode',
			GS_BOOKS_PLUGIN_URI . '/assets/admin/css/gs-books-shortcode.min.css',
			array( 'gs-zmdi-fonts' ),
			GS_BOOKS_VERSION,
			'all'
		);

		$data = array(
			'nonce'    => wp_create_nonce( '_gs_books_admin_nonce_gs_' ),
			'ajaxurl'  => admin_url( 'admin-ajax.php' ),
			'adminurl' => admin_url(),
			'siteurl'  => home_url(),
		);

		$data['shortcode_settings'] = $this->get_shortcode_default_settings();
		$data['shortcode_options']  = $this->get_shortcode_default_options();
		$data['translations']       = $this->get_translation_strings();
		$data['preference']         = $this->get_shortcode_default_prefs();
		$data['preference_options'] = $this->get_shortcode_prefs_options();
		$data['taxonomy_settings']  = $this->get_taxonomy_default_settings();
		$data['is_multilingual']    = $this->is_multilingual_enabled();
		$data['is_pro']    		    = is_pro_active() && is_pro_valid();

		$data['demo_data'] = array(
			'book_data'      => wp_validate_boolean( get_option( 'gsbooks_dummy_book_data_created' ) ),
			'shortcode_data' => wp_validate_boolean( get_option( 'gsbooks_dummy_shortcode_data_created' ) ),
		);

		if ( is_pro_active() && is_pro_valid() ) {
			wp_enqueue_script(
				'gs-books-shortcode',
				GS_BOOKS_PRO_PLUGIN_URI . '/assets/admin/js/gs-books-shortcode.min.js',
				array( 'jquery', 'wp-color-picker' ),
				GS_BOOKS_PRO_VERSION,
				true
			);
		} else {
			wp_enqueue_script(
				'gs-books-shortcode',
				GS_BOOKS_PLUGIN_URI . '/assets/admin/js/gs-books-shortcode.min.js',
				array( 'jquery', 'wp-color-picker' ),
				GS_BOOKS_VERSION,
				true
			);
		}

		wp_localize_script( 'gs-books-shortcode', '_gs_books_data', $data );
	}

	public function preview_scripts() {
		if ( ! $this->is_gs_books_shortcode_preview() ) {
			return;
		}

		wp_enqueue_style(
			'gs-books-shortcode-preview',
			GS_BOOKS_PLUGIN_URI . '/assets/css/gs-books-shortcode-preview.min.css',
			[],
			GS_BOOKS_VERSION,
			'all'
		);
	}

	public function gs_books_get_wpdb() {
		global $wpdb;

		if ( wp_doing_ajax() ) {
			$wpdb->show_errors = false;
		}

		return $wpdb;
	}

	public function gs_books_check_db_error() {
		$wpdb = $this->gs_books_get_wpdb();

		if ( $wpdb->last_error === '' ) {
			return false;
		}

		return true;
	}

	public function is_multilingual_enabled() {
		$localizations = $this->_get_localization( false );
		return wp_validate_boolean( $localizations['gs_member_enable_multilingual'] );
	}

	public function validate_shortcode_settings( $shortcode_settings ) {

		$shortcode_settings = shortcode_atts( $this->get_shortcode_default_settings(), $shortcode_settings );

		$shortcode_settings['theme']                           = sanitize_text_field( $shortcode_settings['theme'] );
		$shortcode_settings['view_type']                       = sanitize_text_field( $shortcode_settings['view_type'] );
		$shortcode_settings['gsb_filter_by']                   = sanitize_text_field( $shortcode_settings['gsb_filter_by'] );
		$shortcode_settings['filter_position']                 = sanitize_text_field( $shortcode_settings['filter_position'] );
		$shortcode_settings['filter_style']                    = sanitize_text_field( $shortcode_settings['filter_style'] );
		$shortcode_settings['gs_book_title_font']              = sanitize_text_field( $shortcode_settings['gs_book_title_font'] );
		$shortcode_settings['gs_book_fz']                      = intval( $shortcode_settings['gs_book_fz'] );
		$shortcode_settings['columns']                         = sanitize_text_field( $shortcode_settings['columns'] );
		$shortcode_settings['columns_tablet']                  = sanitize_text_field( $shortcode_settings['columns_tablet'] );
		$shortcode_settings['columns_mobile_portrait']         = sanitize_text_field( $shortcode_settings['columns_mobile_portrait'] );
		$shortcode_settings['columns_mobile']                  = sanitize_text_field( $shortcode_settings['columns_mobile'] );
		$shortcode_settings['link_type']                       = sanitize_text_field( $shortcode_settings['link_type'] );
		$shortcode_settings['search_by_name']                  = wp_validate_boolean( $shortcode_settings['search_by_name'] );
		$shortcode_settings['search_by_isbn']                  = wp_validate_boolean( $shortcode_settings['search_by_isbn'] );
		$shortcode_settings['search_by_asin']                  = wp_validate_boolean( $shortcode_settings['search_by_asin'] );
		$shortcode_settings['search_by_countries']             = wp_validate_boolean( $shortcode_settings['search_by_countries'] );
		$shortcode_settings['search_by_publishers']            = wp_validate_boolean( $shortcode_settings['search_by_publishers'] );
		$shortcode_settings['search_by_languages']             = wp_validate_boolean( $shortcode_settings['search_by_languages'] );
		$shortcode_settings['search_by_tags']                  = wp_validate_boolean( $shortcode_settings['search_by_tags'] );
		$shortcode_settings['search_by_categories']            = wp_validate_boolean( $shortcode_settings['search_by_categories'] );
		$shortcode_settings['search_by_genres']                = wp_validate_boolean( $shortcode_settings['search_by_genres'] );
		$shortcode_settings['search_by_authors']               = wp_validate_boolean( $shortcode_settings['search_by_authors'] );
		$shortcode_settings['search_by_series']                = wp_validate_boolean( $shortcode_settings['search_by_series'] );
		$shortcode_settings['gs_book_pagination']              = wp_validate_boolean( $shortcode_settings['gs_book_pagination'] );
		$shortcode_settings['posts_per_page']                  = intval( $shortcode_settings['posts_per_page'] );
		$shortcode_settings['show_all']                        = wp_validate_boolean( $shortcode_settings['show_all'] );
		$shortcode_settings['show_all_text']                   = sanitize_text_field( $shortcode_settings['show_all_text'] );
		$shortcode_settings['gs_book_enable_multi_select']     = wp_validate_boolean( $shortcode_settings['gs_book_enable_multi_select'] );
		$shortcode_settings['gs_book_multi_select_ellipsis']   = wp_validate_boolean( $shortcode_settings['gs_book_multi_select_ellipsis'] );
		$shortcode_settings['gs_book_enable_clear_filters']    = wp_validate_boolean( $shortcode_settings['gs_book_enable_clear_filters'] );
		$shortcode_settings['desc_limit']                      = intval( $shortcode_settings['desc_limit'] );
		$shortcode_settings['gs_book_category']                = wp_validate_boolean( $shortcode_settings['gs_book_category'] );
		$shortcode_settings['gs_child_tax']                    = wp_validate_boolean( $shortcode_settings['gs_child_tax'] );
		$shortcode_settings['gs_book_translator']              = wp_validate_boolean( $shortcode_settings['gs_book_translator'] );
		$shortcode_settings['gs_book_isbn']                    = wp_validate_boolean( $shortcode_settings['gs_book_isbn'] );
		$shortcode_settings['gs_book_isbn_thirteen']           = wp_validate_boolean($shortcode_settings['gs_book_isbn_thirteen']);
		$shortcode_settings['gs_book_asin']                    = wp_validate_boolean( $shortcode_settings['gs_book_asin'] );
		$shortcode_settings['gs_book_pages']                   = wp_validate_boolean( $shortcode_settings['gs_book_pages'] );
		$shortcode_settings['gs_book_country']                 = wp_validate_boolean( $shortcode_settings['gs_book_country'] );
		$shortcode_settings['gs_book_language']                = wp_validate_boolean( $shortcode_settings['gs_book_language'] );
		$shortcode_settings['gs_book_dimension']               = wp_validate_boolean( $shortcode_settings['gs_book_dimension'] );
		$shortcode_settings['gs_book_fsize']                   = wp_validate_boolean( $shortcode_settings['gs_book_fsize'] );
		$shortcode_settings['gsb_slider_navs']                 = wp_validate_boolean( $shortcode_settings['gsb_slider_navs'] );
		$shortcode_settings['gsb_slider_dots']                 = wp_validate_boolean( $shortcode_settings['gsb_slider_dots'] );
		$shortcode_settings['gs_book_authordes']               = wp_validate_boolean( $shortcode_settings['gs_book_authordes'] );
		$shortcode_settings['gs_store_img']                    = wp_validate_boolean( $shortcode_settings['gs_store_img'] );
		$shortcode_settings['gs_book_sin_nxt_prev']            = wp_validate_boolean( $shortcode_settings['gs_book_sin_nxt_prev'] );
		$shortcode_settings['num']                             = intval( $shortcode_settings['num'] );
		$shortcode_settings['order']                           = sanitize_text_field( $shortcode_settings['order'] );
		$shortcode_settings['gs_filter_cat_pos']               = sanitize_text_field( $shortcode_settings['gs_filter_cat_pos'] );
		$shortcode_settings['group']                           = array_map( 'sanitize_text_field', $shortcode_settings['group'] );
		$shortcode_settings['exclude_group']                   = array_map( 'sanitize_text_field', $shortcode_settings['exclude_group'] );
		$shortcode_settings['include_tags']                    = array_map( 'sanitize_text_field', $shortcode_settings['include_tags'] );
		$shortcode_settings['exclude_tags']                    = array_map( 'sanitize_text_field', $shortcode_settings['exclude_tags'] );
		$shortcode_settings['orderby']                         = sanitize_text_field( $shortcode_settings['orderby'] );
		$shortcode_settings['group_order_by']                  = sanitize_text_field( $shortcode_settings['group_order_by'] );
		$shortcode_settings['group_order']                     = sanitize_text_field( $shortcode_settings['group_order'] );
		$shortcode_settings['gs_book_fntw']                    = sanitize_text_field( $shortcode_settings['gs_book_fntw'] );
		$shortcode_settings['gs_book_fnstyl']                  = sanitize_text_field( $shortcode_settings['gs_book_fnstyl'] );
		$shortcode_settings['gs_sin_book_txt_align']           = sanitize_text_field( $shortcode_settings['gs_sin_book_txt_align'] );
		$shortcode_settings['gs_book_thumbnail_sizes']         = sanitize_text_field( $shortcode_settings['gs_book_thumbnail_sizes'] );
		$shortcode_settings['gs_book_thumbnail_sizes_details'] = sanitize_text_field( $shortcode_settings['gs_book_thumbnail_sizes_details'] );
		$shortcode_settings['popup_style']                     = sanitize_text_field( $shortcode_settings['popup_style'] );
		// $shortcode_settings['store_display']                   = sanitize_text_field( $shortcode_settings['store_display'] );
		$shortcode_settings['gs_book_name_color']              = sanitize_text_field( $shortcode_settings['gs_book_name_color'] );
		$shortcode_settings['gs_book_rating_color']            = sanitize_text_field( $shortcode_settings['gs_book_rating_color'] );
		$shortcode_settings['gs_book_label_fz']                = intval( $shortcode_settings['gs_book_label_fz'] );
		$shortcode_settings['gs_book_label_color']             = sanitize_text_field( $shortcode_settings['gs_book_label_color'] );
		$shortcode_settings['gs_books_btn_nav_cls_color']      = sanitize_text_field( $shortcode_settings['gs_books_btn_nav_cls_color'] );
		$shortcode_settings['gs_filter_enabled']               = wp_validate_boolean( $shortcode_settings['gs_filter_enabled'] );
		$shortcode_settings['shortcode_settings']              = wp_validate_boolean( $shortcode_settings['shortcode_settings'] );
		$shortcode_settings['isAutoplay']                      = wp_validate_boolean( $shortcode_settings['isAutoplay'] );
		$shortcode_settings['gsb_navs_style']                  = sanitize_text_field( $shortcode_settings['gsb_navs_style'] );
		$shortcode_settings['gsb_navs_pos']                    = sanitize_text_field( $shortcode_settings['gsb_navs_pos'] );
		$shortcode_settings['gsb_dots_style']                  = sanitize_text_field( $shortcode_settings['gsb_dots_style'] );
		$shortcode_settings['speed']                           = intval( $shortcode_settings['speed'] );
		$shortcode_settings['ticker_speed']                    = intval( $shortcode_settings['ticker_speed'] );
		$shortcode_settings['autoplay_delay']                  = intval( $shortcode_settings['autoplay_delay'] );
		$shortcode_settings['slides_per_group']                = intval( $shortcode_settings['slides_per_group'] );
		$shortcode_settings['pause_on_hover']                  = wp_validate_boolean( $shortcode_settings['pause_on_hover'] );
		$shortcode_settings['infinite_loop']                   = wp_validate_boolean( $shortcode_settings['infinite_loop'] );
		$shortcode_settings['reverse_direction']               = wp_validate_boolean( $shortcode_settings['reverse_direction'] );

		$shortcode_settings['visibility_settings'] 			   = shortcode_atts( $this->get_visibility_defaults( $this->visibility_settings_exclude() ), $shortcode_settings['visibility_settings'] );
		$shortcode_settings['detail_visibility_settings'] 	   = shortcode_atts( $this->get_visibility_defaults( $this->detail_visibility_settings_exclude() ), $shortcode_settings['detail_visibility_settings'] );

        $shortcode_settings['visibility_settings'] = $this->validate_fields_visibility_settings( $shortcode_settings['visibility_settings'] );
        $shortcode_settings['detail_visibility_settings'] = $this->validate_fields_visibility_settings( $shortcode_settings['detail_visibility_settings'] );

		return $shortcode_settings;
	}

	public function get_taxonomy_default_settings() {

		$free_tax = [

			// Category Taxonomy
			'enable_group_tax'         => 'on',
			'group_tax_label'          => __('Category', 'gs-books-showcase'),
			'group_tax_plural_label'   => __('Categories', 'gs-books-showcase'),
			'enable_group_tax_archive' => 'on',
			'group_tax_archive_slug'   => 'bookshowcase_group',

			// Tag Taxonomy
			'enable_tag_tax'           => 'on',
			'tag_tax_label'            => __('Tag', 'gs-books-showcase'),
			'tag_tax_plural_label'     => __('Tags', 'gs-books-showcase'),
			'enable_tag_tax_archive'   => 'on',
			'tag_tax_archive_slug'     => 'gsb_tag',
		];

		$pro_tax = [
			// Author Taxonomy
			'enable_author_tax'           => 'on',
			'author_tax_label'            => __('Author', 'gs-books-showcase'),
			'author_tax_plural_label'     => __('Authors', 'gs-books-showcase'),
			'enable_author_tax_archive'   => 'on',
			'author_tax_archive_slug'     => 'gsb_author',

			// Genre Taxonomy
			'enable_genre_tax'           => 'on',
			'genre_tax_label'            => __('Genre', 'gs-books-showcase'),
			'genre_tax_plural_label'     => __('Genres', 'gs-books-showcase'),
			'enable_genre_tax_archive'   => 'on',
			'genre_tax_archive_slug'     => 'gsb_genre',

			// Series Taxonomy
			'enable_series_tax'           => 'on',
			'series_tax_label'            => __('Series', 'gs-books-showcase'),
			'series_tax_plural_label'     => __('Series', 'gs-books-showcase'),
			'enable_series_tax_archive'   => 'on',
			'series_tax_archive_slug'     => 'gsb_series',

			// Languages Taxonomy
			'enable_language_tax'           => 'on',
			'language_tax_label'            => __('Language', 'gs-books-showcase'),
			'language_tax_plural_label'     => __('Languages', 'gs-books-showcase'),
			'enable_language_tax_archive'   => 'on',
			'language_tax_archive_slug'     => 'gsb_languages',

			// Publishers Taxonomy
			'enable_publisher_tax'           => 'on',
			'publisher_tax_label'            => __('Publisher', 'gs-books-showcase'),
			'publisher_tax_plural_label'     => __('Publishers', 'gs-books-showcase'),
			'enable_publisher_tax_archive'   => 'on',
			'publisher_tax_archive_slug'     => 'gsb_publishers',

			// Countries Taxonomy
			'enable_country_tax'           => 'on',
			'country_tax_label'            => __('Country', 'gs-books-showcase'),
			'country_tax_plural_label'     => __('Countries', 'gs-books-showcase'),
			'enable_country_tax_archive'   => 'on',
			'country_tax_archive_slug'     => 'gsb_countries',
		];

		return ( is_pro_active() && is_pro_valid() ) ? array_merge($free_tax, $pro_tax) : $free_tax;

	}

	protected function get_gs_books_shortcode_db_columns() {
		return array(
			'shortcode_name'     => '%s',
			'shortcode_settings' => '%s',
			'created_at'         => '%s',
			'updated_at'         => '%s',
		);
	}

	public function _get_shortcode( $shortcode_id, $is_ajax = false ) {

		if ( empty( $shortcode_id ) ) {
			if ( $is_ajax ) {
				wp_send_json_error( __( 'Shortcode ID missing', 'gs-books-showcase' ), 400 );
			}
			return false;
		}

		$shortcode = wp_cache_get( 'gs_book_shortcode' . $shortcode_id, 'gs_book_showcase' );

		// Return the cache if found
		if ( $shortcode !== false ) {
			if ( $is_ajax ) {
				wp_send_json_success( $shortcode );
			}
			return $shortcode;
		}

		$wpdb      = $this->gs_books_get_wpdb();
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
		$shortcode = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}gs_books WHERE id = %d LIMIT 1", absint( $shortcode_id ) ), ARRAY_A );

		if ( $shortcode ) {
			$shortcode['shortcode_settings'] = json_decode( $shortcode['shortcode_settings'], true );
			$shortcode['shortcode_settings'] = $this->validate_shortcode_settings( $shortcode['shortcode_settings'] );

			wp_cache_add( 'gs_book_shortcode' . $shortcode_id, $shortcode, 'gs_book_showcase' );

			if ( $is_ajax ) {
				wp_send_json_success( $shortcode );
			}
			return $shortcode;
		}

		if ( $is_ajax ) {
			wp_send_json_error( __( 'No shortcode found', 'gs-books-showcase' ), 404 );
		}

		return false;
	}

	public function _get_shortcodes( $shortcode_ids = array(), $is_ajax = false, $minimal = false ) {
		$wpdb   = $this->gs_books_get_wpdb();
		$fields = $minimal ? 'id, shortcode_name' : '*';

		if ( empty( $shortcode_ids ) ) {
			$shortcodes = $wpdb->get_results( "SELECT {$fields} FROM {$wpdb->prefix}gs_books ORDER BY id DESC", ARRAY_A );
		} else {
			$how_many     = count( $shortcode_ids );
			$placeholders = array_fill( 0, $how_many, '%d' );
			$format       = implode( ', ', $placeholders );
			$query        = "SELECT {$fields} FROM {$wpdb->prefix}gs_books WHERE id IN($format)";
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
			$shortcodes   = $wpdb->get_results( $wpdb->prepare( $query, $shortcode_ids ), ARRAY_A );
		}

		// check for database error
		if ( $this->gs_books_check_db_error() ) {
			// translators: %s: Database error message from $wpdb->last_error
			wp_send_json_error( sprintf( esc_html__( 'Database Error: %s','gs-books-showcase' ), $wpdb->last_error ) );
		}

		if ( $is_ajax ) {
			wp_send_json_success( $shortcodes );
		}

		return $shortcodes;
	}

	public function create_shortcode() {

		// validate nonce && check permission
		if ( ! check_admin_referer( '_gs_books_admin_nonce_gs_' ) || ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Unauthorised Request', 'gs-books-showcase' ), 401 );
		}

		$shortcode_settings = ! empty( $_POST['shortcode_settings'] ) ? wp_unslash($_POST['shortcode_settings']) : '';
		$shortcode_name = ! empty( $_POST['shortcode_name'] ) ? sanitize_text_field( wp_unslash( $_POST['shortcode_name'] ) ) : __( 'Undefined', 'gs-books-showcase' );
		
		if ( empty( $shortcode_settings ) || ! is_array( $shortcode_settings ) ) {
			wp_send_json_error( __( 'Please configure the settings properly', 'gs-books-showcase' ), 206 );
		}

		$shortcode_settings = $this->validate_shortcode_settings( $shortcode_settings );
		$wpdb               = $this->gs_books_get_wpdb();

		$data = array(
			'shortcode_name'     => $shortcode_name,
			'shortcode_settings' => json_encode( $shortcode_settings ),
			'created_at'         => current_time( 'mysql' ),
			'updated_at'         => current_time( 'mysql' ),
		);

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
		$wpdb->insert(
			"{$wpdb->prefix}gs_books",
			$data,
			$this->get_gs_books_shortcode_db_columns()
		);

		// check for database error
		if ( $this->gs_books_check_db_error() ) {
			// translators: %s: Database error message from $wpdb->last_error
			wp_send_json_error( sprintf( __( 'Database Error: %s','gs-books-showcase' ), $wpdb->last_error ), 500 );
		}

		// send success response with inserted id
		wp_send_json_success(
			array(
				'message'      => __( 'Shortcode created successfully', 'gs-books-showcase' ),
				'shortcode_id' => $wpdb->insert_id,
			)
		);
	}

	public function clone_shortcode() {
		// validate nonce && check permission
		if ( ! check_admin_referer( '_gs_books_admin_nonce_gs_' ) || ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Unauthorised Request', 'gs-books-showcase' ), 401 );
		}

		$clone_id = ! empty( $_POST['clone_id'] )?  wp_unslash( $_POST['clone_id'] ) : '';

		if ( empty( $clone_id ) ) {
			wp_send_json_error( __( 'Clone Id not provided', 'gs-books-showcase' ), 400 );
		}

		$clone_shortcode = $this->_get_shortcode( $clone_id, false );

		if ( empty( $clone_shortcode ) ) {
			wp_send_json_error( __( 'Clone shortcode not found', 'gs-books-showcase' ), 404 );
		}

		$shortcode_settings = $clone_shortcode['shortcode_settings'];
		$shortcode_name     = $clone_shortcode['shortcode_name'] . ' ' . __( '- Cloned', 'gs-books-showcase' );
		$shortcode_settings = $this->validate_shortcode_settings( $shortcode_settings );

		$wpdb = $this->gs_books_get_wpdb();

		$data = array(
			'shortcode_name'     => $shortcode_name,
			'shortcode_settings' => json_encode( $shortcode_settings ),
			'created_at'         => current_time( 'mysql' ),
			'updated_at'         => current_time( 'mysql' ),
		);

		 // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
		$wpdb->insert(
			"{$wpdb->prefix}gs_books",
			$data,
			$this->get_gs_books_shortcode_db_columns()
		);

		// check for database error
		if ( $this->gs_books_check_db_error() ) {
			// translators: %s: Database error message from $wpdb->last_error
			wp_send_json_error( sprintf( __( 'Database Error: %s','gs-books-showcase' ), $wpdb->last_error ), 500 );
		}

		// Get the cloned shortcode
		$shotcode = $this->_get_shortcode( $wpdb->insert_id, false );

		// send success response with inserted id
		wp_send_json_success(
			array(
				'message'   => __( 'Shortcode cloned successfully', 'gs-books-showcase' ),
				'shortcode' => $shotcode,
			)
		);
	}

	public function get_shortcode() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$shortcode_id = ! empty( $_GET['id'] ) ? absint( $_GET['id'] ) : null;
		$this->_get_shortcode( $shortcode_id, wp_doing_ajax() );
	}

	public function update_shortcode( $shortcode_id = null, $nonce = null ) {

		if ( ! check_admin_referer( '_gs_books_admin_nonce_gs_' ) || ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Unauthorised Request', 'gs-books-showcase' ), 401 );
		}

		$shortcode_id = isset( $_POST['id'] ) ? sanitize_text_field( wp_unslash( $_POST['id'] ) ): null;

		if ( empty( $shortcode_id ) ) {
			wp_send_json_error( __( 'Shortcode ID missing', 'gs-books-showcase' ), 400 );
		}

		$_shortcode = $this->_get_shortcode( $shortcode_id, false );

		if ( empty( $_shortcode ) ) {
			wp_send_json_error( __( 'No shortcode found to update', 'gs-books-showcase' ), 404 );
		}

		$shortcode_name = ! empty( $_POST['shortcode_name'] ) ? sanitize_text_field( wp_unslash( $_POST['shortcode_name'] ) ) : sanitize_text_field( $_shortcode['shortcode_name'] );
		$shortcode_settings = ! empty( $_POST['shortcode_settings'] ) ? $_POST['shortcode_settings'] : $_shortcode['shortcode_settings'];

		// Remove dummy indicator on update
		if ( isset( $shortcode_settings['gsbooks-demo_data'] ) ) {
			unset( $shortcode_settings['gsbooks-demo_data'] );
		}

		$shortcode_settings = $this->validate_shortcode_settings( $shortcode_settings );
		$wpdb               = $this->gs_books_get_wpdb();

		$data = array(
			'shortcode_name'     => $shortcode_name,
			'shortcode_settings' => json_encode( $shortcode_settings ),
			'updated_at'         => current_time( 'mysql' ),
		);

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$num_row_updated = $wpdb->update(
			"{$wpdb->prefix}gs_books",
			$data,
			array( 'id' => absint( $shortcode_id ) ),
			$this->get_gs_books_shortcode_db_columns()
		);

		if ( $this->gs_books_check_db_error() ) {
			// translators: %s: Database error message from $wpdb->last_error
			wp_send_json_error( sprintf( __( 'Database Error: %1$s', 'gs-books-showcase' ), $wpdb->last_error ), 500 );
			return false;
		}

		do_action( 'gs_books_shortcode_updated', $num_row_updated );
		do_action( 'gsp_shortcode_updated', $num_row_updated );

		wp_send_json_success(
			array(
				'message'      => __( 'Shortcode updated', 'gs-books-showcase' ),
				'shortcode_id' => $num_row_updated,
			)
		);
	}

	public function delete_shortcodes() {
		if ( ! check_admin_referer( '_gs_books_admin_nonce_gs_' ) || ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Unauthorised Request', 'gs-books-showcase' ), 401 );
		}

		$ids = isset( $_POST['ids'] ) ? wp_unslash( $_POST['ids'] ) : null;

		if ( empty( $ids ) ) {
			wp_send_json_error( __( 'No shortcode ids provided', 'gs-books-showcase' ), 400 );
		}

		$wpdb  = $this->gs_books_get_wpdb();
		$count = count( $ids );
		$ids   = implode( ',', array_map( 'absint', $ids ) );

		$wpdb->query( "DELETE FROM {$wpdb->prefix}gs_books WHERE ID IN($ids)" );

		if ( $this->gs_books_check_db_error() ) {
			// translators: %s: Database error message from $wpdb->last_error
			wp_send_json_error( sprintf( __( 'Database Error: %s','gs-books-showcase' ), $wpdb->last_error ), 500 );
		}

		$m = _n( 'Shortcode has been deleted', 'Shortcodes have been deleted', $count, 'gs-books-showcase' );

		wp_send_json_success( array( 'message' => $m ) );
	}

	public function get_shortcodes() {
		$this->_get_shortcodes( null, wp_doing_ajax() );
	}

	public function get_shortcode_default_settings() {

		return array(
			'theme'                           => 'style-01',
			'view_type'                       => 'grid',
			'gsb_filter_by'                   => 'category',
			'filter_position'                 => 'left',
			'filter_style'                    => 'filter--default',
			'gs_book_title_font'              => 'Roboto, sans-serif',
			'gs_book_fz'                      => '',
			'columns'                         => '3',
			'columns_tablet'                  => '4',
			'columns_mobile_portrait'         => '6',
			'columns_mobile'                  => '12',
			'link_type'                       => 'single_page',
			'show_all'                        => true,
			'search_by_name'                  => true,
			'search_by_isbn'                  => true,
			'search_by_asin'                  => true,
			'search_by_countries'             => true,
			'search_by_publishers'            => true,
			'search_by_languages'             => true,
			'search_by_tags'                  => true,
			'search_by_categories'            => true,
			'search_by_authors'               => true,
			'search_by_series'                => true,
			'search_by_genres'                => true,
			'gs_book_pagination'              => false,
			'posts_per_page'                  => 10,
			'gs_book_enable_multi_select'     => false,
			'gs_book_multi_select_ellipsis'   => false,
			'gs_book_enable_clear_filters'    => true,
			'desc_limit'                      => 100,
			'gs_book_category'                => true,
			'gs_child_tax'                    => false,
			'gs_book_translator'              => true,
			'book_short_description'          => true,
			'gs_book_isbn'                    => true,
			'gs_book_isbn_thirteen'			  => true,
			'gs_book_asin'                    => true,
			'gs_book_pages'                   => true,
			'gs_book_country'                 => true,
			'gs_book_language'                => true,
			'gs_book_dimension'               => true,
			'gs_book_fsize'                   => true,
			'gsb_slider_navs'                 => true,
			'gsb_slider_dots'                 => true,
			'gs_book_authordes'               => true,
			'gs_store_img'                    => true,
			'gs_book_sin_nxt_prev'            => true,
			'num'                             => -1,
			'order'                           => 'DESC',
			'gs_filter_cat_pos'               => 'left',
			'group'                           => array(),
			'exclude_group'                   => array(),
			'include_tags'                    => array(),
			'exclude_tags'                    => array(),
			'include_authors'                 => array(),
			'exclude_authors'                 => array(),
			'include_genres'                  => array(),
			'exclude_genres'                  => array(),
			'include_series'                  => array(),
			'exclude_series'                  => array(),
			'include_languages'               => array(),
			'exclude_languages'               => array(),
			'include_publishers'              => array(),
			'exclude_publishers'              => array(),
			'include_countries'               => array(),
			'exclude_countries'               => array(),
			'orderby'                         => 'menu_order',
			'group_order_by'                  => 'term_order',
			'group_order'                     => 'ASC',
			'gs_book_fntw'                    => 'normal',
			'gs_book_fnstyl'                  => 'normal',
			'gs_sin_book_txt_align'           => 'left',
			'gs_book_thumbnail_sizes'         => 'large',
			'gs_book_thumbnail_sizes_details' => 'details',
			'popup_style'                     => 'style_one',
			'show_all_text'                   => 'Show All',
			'gs_book_name_color'              => '',
			'gs_book_rating_color'            => '',
			'gs_book_label_fz'                => 17,
			'gs_book_label_color'             => '',
			'gs_books_btn_nav_cls_color'      => '',
			'gs_filter_enabled'               => false,
			'shortcode_settings'              => false,
			'isAutoplay'                      => false,
			'gsb_navs_style'                  => 'nav--default',
			'gsb_navs_pos'                    => 'carousel-navs-pos--bottom',
			'gsb_dots_style'                  => 'dot--default',
			'image_filter'                    => 'none',
			'hover_image_filter'              => 'none',
			'speed'                           => 5000,
			'ticker_speed'                    => 1000,
			'autoplay_delay'                  => 5000,
			'slides_per_group'                => 1,
			'pause_on_hover'                  => false,
			'isAutoplay'                      => false,
			'infinite_loop'                   => false,
			'reverse_direction'               => false,
			'visibility_settings'             => $this->get_visibility_defaults( $this->visibility_settings_exclude() ),
			'detail_visibility_settings' 	  => $this->get_visibility_defaults( $this->detail_visibility_settings_exclude() ),
		);
	}

	public function visibility_settings_exclude() {
		return [
			'book_formats',
			'readers_review',
			'book_short_description',
			'book_translator',
			'book_author',
			'book_categories',
			'book_tags',
			'book_genres',
			'book_series',
			'book_languages',
			'book_countries',
			'book_isbn',
			'book_isbn_thirteen',
			'book_asin',
			'book_doi',
			'book_llcn',
			'book_pages',
			'book_dimension',
			'book_weight',
			'book_file_size',
			'book_download_url',
			'book_cover',
			'book_regular_price',
			'book_sale_price',
			'book_age_group',
			'book_reading_level',
			'book_edition',
			'book_edition_features',
			'book_awards',
			'book_reading_time',
			'book_author_social',
			'book_author_section',
			'book_author_img',
			'book_co_publisher',
			'book_availability',
			'book_preorder_available',
			'book_accessibility_features',
			'book_related_section'
		];
	}

	public function detail_visibility_settings_exclude() {
		return ['book_related_section'];
	}

    public function get_visibility_defaults( $exclude = [] ) {
        $fields = [
            'book_name' => [
				'translation_key' => 'name',
                'desktop' => true,
                'tablet' => true,
				'mobile_landscape' => true,
                'mobile' => true,
            ],
            'book_author_by' => [
				'translation_key' => 'book-author-by',
                'desktop' => true,
                'tablet' => true,
				'mobile_landscape' => true,
                'mobile' => true,
            ],
			'book_publish_date' => [
				'translation_key' => 'book-publish-date',
				'desktop' => true,
				'tablet' => true,
				'mobile_landscape' => true,
				'mobile' => true,
			],
			'book_isbn' => [
				'translation_key' => 'gs_book_isbn',
				'desktop' => true,
				'tablet' => true,
				'mobile_landscape' => true,
				'mobile' => true,
			],
			'book_isbn_thirteen' => [
				'translation_key' => 'gs_book_isbn_thirteen',
				'desktop' => true,
				'tablet' => true,
				'mobile_landscape' => true,
				'mobile' => true,
			],
			'book_details' => [
				'translation_key' => 'book-details',
				'desktop' => true,
				'tablet' => true,
				'mobile_landscape' => true,
				'mobile' => true,
			],
			'book_dimension' => [
				'translation_key' => 'gs_book_dimension',
				'desktop' => true,
				'tablet' => true,
				'mobile_landscape' => true,
				'mobile' => true,
			],
			'book_file_size' => [
				'translation_key' => 'file-size-e-book',
				'desktop' => true,
				'tablet' => true,
				'mobile_landscape' => true,
				'mobile' => true,
			],
			'book_pages' => [
            	'translation_key' => 'pages',
            	'desktop' => true,
            	'tablet' => true,
            	'mobile_landscape' => true,
            	'mobile' => true,
			],
			'book_download_url' => [
				'translation_key' => 'download-url',
				'desktop' => true,
				'tablet' => true,
				'mobile_landscape' => true,
				'mobile' => true,
			],
			'book_categories' => [
				'translation_key' => 'categories',
				'desktop' => true,
				'tablet' => true,
				'mobile_landscape' => true,
				'mobile' => true,
			],
			'book_tags' => [
				'translation_key' => 'tags',
				'desktop' => true,
				'tablet' => true,
				'mobile_landscape' => true,
				'mobile' => true,
			],
			
		];

		$fields = apply_filters('gs_book_visibility_fields', $fields, $exclude );

		foreach ( $exclude as $key ) {
			if ( isset( $fields[ $key ] ) ) {
				unset( $fields[ $key ] );
			}
		}

		$fields = $this->apply_default_visibility_order( $fields );

		return $fields;
	}

	public function temp_save_shortcode_settings() {

		if ( ! check_admin_referer( '_gs_books_admin_nonce_gs_' ) || ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Unauthorised Request', 'gs-books-showcase' ), 401 );
		}

		$temp_key = isset( $_POST['temp_key'] ) ? sanitize_text_field( wp_unslash( $_POST['temp_key'] ) ) : null;
		$shortcode_settings = isset( $_POST['shortcode_settings'] ) ? $_POST['shortcode_settings'] : null;

		if ( empty( $temp_key ) ) {
			wp_send_json_error( __( 'No temp key provided', 'gs-books-showcase' ), 400 );
		}

		if ( empty( $shortcode_settings ) ) {
			wp_send_json_error( __( 'No temp settings provided', 'gs-books-showcase' ), 400 );
		}

		delete_transient( $temp_key );

		$shortcode_settings = $this->validate_shortcode_settings( $shortcode_settings );
		set_transient( $temp_key, $shortcode_settings, 86400 ); // save the transient for 1 day

		wp_send_json_success(
			array(
				'message' => __( 'Temp data saved', 'gs-books-showcase' ),
			)
		);
	}

	public function get_translation_strings() {
		return array(
			'books_slug'                           => __( 'Books Slug', 'gs-books-showcase' ),
			'books_slug_info'                      => __( 'Customize Books Post Type Slug, by default it is set to books', 'gs-books-showcase' ),
			'replace_custom_slug' 				   => __( 'Ignore Base Permalink Prefix', 'gs-books-showcase' ),
			'replace_custom_slug_info' 			   => __( 'Enable this option to use a custom structure without the base prefix.', 'gs-books-showcase' ),

			'archive_page_slug' 				   => __('Archive Page Slug', 'gs-books-showcase'),
			'archive_page_slug_details' 		   => __('Set Custom Archive Page Slug, now it is set to', 'gs-books-showcase') . ' ' . get_post_type_archive_link( 'gs_bookshowcase' ),

			'archive_page_title' 				   => __('Archive Page Title', 'gs-books-showcase'),
			'archive_page_title_details' 		   => __('Set Custom Archive Page Title, now it is set to', 'gs-books-showcase') . ' ' . gs_get_post_type_archive_title(),

			'single_page'                          => __( 'Single Page', 'gs-books-showcase' ),
			'single_page_info'                     => __( 'Single Page', 'gs-books-showcase' ),
			'popup_style'                          => __( 'PopUp Style', 'gs-books-showcase' ),
			'store_display'                        => __( 'Store', 'gs-books-showcase' ),
			'popup_style__details'                 => __( 'PopUp Style', 'gs-books-showcase' ),
			'gs_book_thumbnail_sizes'              => __( 'Thumbnail', 'gs-books-showcase' ),
			'details-control'                      => __( 'Details Control', 'gs-books-showcase' ),
			'define-maximum-number-of-characters'  => __( 'Define maximum number of characters in Member details. Default 100', 'gs-books-showcase' ),
			'enable-multilingual'                  => __( 'Enable Multilingual', 'gs-books-showcase' ),
			'enable-multilingual--details'         => __( 'Enable Multilingual mode to translate strings using any Multilingual plugin, such as WPML or Loco Translate.', 'gs-books-showcase' ),
			'add_formats'                          => __( 'Add Formats', 'gs-books-showcase' ),
			'add_formats_help'                     => __( 'Add Formats', 'gs-books-showcase' ),
			'allowed_book_format'                  => __( 'Allowed Formats', 'gs-books-showcase' ),
			'allowed_book_format_info'             => __( 'Allowed Formats', 'gs-books-showcase' ),
			'slides_per_group'                     => __( 'Per Group', 'gs-books-showcase' ),
			'carousel_enabled'                     => __( 'Enable Carousel', 'gs-books-showcase' ),
			'carousel_enabled_details'             => __( 'Enable carousel for this theme, it may not available for certain theme', 'gs-books-showcase' ),
			'filter_enabled'                       => __( 'Enable Filter', 'gs-books-showcase' ),
			'filter_enabled_details'               => __( 'Enable filter for this theme, it may not available for certain theme', 'gs-books-showcase' ),
			'filter-position'                      => __( 'Filter Position', 'gs-books-showcase' ),
			'gs_child_tax'                         => __( 'Child Taxonomy', 'gs-books-showcase' ),
			'link_type'                            => __( 'Link Type', 'gs-books-showcase' ),
			'book-single'                          => __( 'Book Single', 'gs-books-showcase' ),
			'book-single-info'                     => __( 'Select Book Single Style', 'gs-books-showcase' ),
			'author-single'                        => __( 'Author Single', 'gs-books-showcase' ),
			'author-single-info'                   => __( 'Select Author Single Style', 'gs-books-showcase' ),
			'gs-book-author'                       => __( 'Author', 'gs-books-showcase' ),
			'gs-book-author-info'                  => __( 'Select Book Author Style', 'gs-books-showcase' ),
			'title-color'                          => __( 'Title Color', 'gs-books-showcase' ),
			'title-color-desc'                     => __( 'Select Title Color for Book Name.', 'gs-books-showcase' ),
			'label-font-size'                      => __( 'Label Font Size', 'gs-books-showcase' ),
			'label-font-size-desc'                 => __( 'Set Font Size for Book Label', 'gs-books-showcase' ),
			'font-style'                           => __( 'Font Style', 'gs-books-showcase' ),
			'font-style-desc'                      => __( 'Select Font Weight for Book Name', 'gs-books-showcase' ),
			'font-weight'                          => __( 'Font Weight', 'gs-books-showcase' ),
			'font-weight-info'                     => __( 'Select Font Weight for Book Name', 'gs-books-showcase' ),
			'font-size'                            => __( 'Font Size', 'gs-books-showcase' ),
			'font-size-desc'                       => __( 'Set Font Size for Book Name', 'gs-books-showcase' ),
			'desc-char-control'                    => __( 'Description Character Control', 'gs-books-showcase' ),
			'desc-char-control-info'               => __( 'Set max no of characters in Book details. Default 100 & Max 300', 'gs-books-showcase' ),
			'book-link-target'                     => __( 'Book Link Target', 'gs-books-showcase' ),
			'gsb_currency'                         => __( 'Currency', 'gs-books-showcase' ),
			'book-link-target-info'                => __( 'Specify target to load the Links, Default New Tab', 'gs-books-showcase' ),
			'book-name'                            => __( 'Book Name', 'gs-books-showcase' ),
			'book-info'                            => __( 'Show or Hide All Books Name', 'gs-books-showcase' ),
			'book-details-info'                    => __( 'Show or Hide All Books Details', 'gs-books-showcase' ),
			'book-author'                          => __( 'Book Author', 'gs-books-showcase' ),
			'book-author-by'                       => __( 'Book Author By', 'gs-books-showcase' ),
			'book-author-info'                     => __( 'Show or Hide All Books Author', 'gs-books-showcase' ),
			'book-publish-date'                    => __( 'Publish Date', 'gs-books-showcase' ),
			'book-publish-date-info'               => __( 'Show or Hide All Books Publish Date', 'gs-books-showcase' ),
			'book-publisher'                       => __( 'Book Publisher', 'gs-books-showcase' ),
			'book-publisher-info'                  => __( 'Show or Hide All Books Publisher', 'gs-books-showcase' ),
			'book-co-publisher'                    => __( 'Book Co-Publisher', 'gs-books-showcase' ),
			'book-co-publisher-info'               => __( 'Show or Hide All Books Co-Publisher', 'gs-books-showcase' ),
			'localization-title'                   => __( 'Localization', 'gs-books-showcase' ),
			'localization-subtitle'                => __( 'Global setting for localization strings', 'gs-books-showcase' ),
			'categories' 							=> __( 'Categories', 'gs-books-showcase' ),
			'tags' 									=> __( 'Tags', 'gs-books-showcase' ),
			'ratings' 								=> __( 'Ratings', 'gs-books-showcase' ),
			'book_store' 							=> __( 'Store', 'gs-books-showcase' ),
			'authors' 								=> __( 'Authors', 'gs-books-showcase' ),
			'genres' 								=> __( 'Genres', 'gs-books-showcase' ),
			'series' 								=> __( 'Series', 'gs-books-showcase' ),
			'languages' 							=> __( 'Languages', 'gs-books-showcase' ),
			'publishers' 							=> __( 'Publishers', 'gs-books-showcase' ),
			'co-publisher'                         	=> __( 'Co-Publisher', 'gs-books-showcase' ),
			'countries' 							=> __( 'Countries', 'gs-books-showcase' ),

			'more'                                 => __( 'More', 'gs-books-showcase' ),
			'author'                               => __( 'Author', 'gs-books-showcase' ),
			'publish'                              => __( 'Publish Date', 'gs-books-showcase' ),
			'translator'                           => __( 'Translator', 'gs-books-showcase' ),
			'format'                               => __( 'Format', 'gs-books-showcase' ),
			'isbn'                                 => __( 'ISBN', 'gs-books-showcase' ),
			'asin'                                 => __( 'ASIN', 'gs-books-showcase' ),
			'save-settings'						   => __( 'Save Settings', 'gs-books-showcase' ),

			'bulk-import'                 		   => __('Bulk Import', 'gs-books-showcase'),
			'bulk-import-description' 			   => __('Add Book Showcase faster by GS Bulk Import feature', 'gs-books-showcase'),

			'export-data' 						   => __('Export Data', 'gs-books-showcase'),
			'export-data--description' 			   => __('Export GS Books Plugins data', 'gs-books-showcase'),

			'import-data' 						   => __('Import Data', 'gs-books-showcase'),
			'import-data--description' 			   => __('Import GS Books Plugins data', 'gs-books-showcase'),

			'image_filter' 						   => __( 'Image Filter', 'gs-books-showcase' ),
			'hover_image_filter' 				   => __( 'Image Filter Hover', 'gs-books-showcase' ),

			// Group Taxonomy
			'enable_group_tax'					   => __( 'Enable Taxonomy', 'gs-books-showcase' ),
			'group_tax_label'					   => __( 'Taxonomy Label', 'gs-books-showcase' ),
			'group_tax_plural_label'			   => __( 'Taxonomy Plural Label', 'gs-books-showcase' ),
			'enable_group_tax_archive'			   => __( 'Enable Taxonomy Archive', 'gs-books-showcase' ),
			'group_tax_archive_slug'			   => __( 'Taxonomy Archive Slug', 'gs-books-showcase' ),

			// Tag Taxonomy
			'enable_tag_tax'					   => __( 'Enable Taxonomy', 'gs-books-showcase' ),
			'tag_tax_label'					       => __( 'Taxonomy Label', 'gs-books-showcase' ),
			'tag_tax_plural_label'			       => __( 'Taxonomy Plural Label', 'gs-books-showcase' ),
			'enable_tag_tax_archive'			   => __( 'Enable Taxonomy Archive', 'gs-books-showcase' ),
			'tag_tax_archive_slug'			       => __( 'Taxonomy Archive Slug', 'gs-books-showcase' ),

			// Author Taxonomy
			'enable_author_tax'					   => __( 'Enable Taxonomy', 'gs-books-showcase' ),
			'author_tax_label'					   => __( 'Taxonomy Label', 'gs-books-showcase' ),
			'author_tax_plural_label'			   => __( 'Taxonomy Plural Label', 'gs-books-showcase' ),
			'enable_author_tax_archive'			   => __( 'Enable Taxonomy Archive', 'gs-books-showcase' ),
			'author_tax_archive_slug'			   => __( 'Taxonomy Archive Slug', 'gs-books-showcase' ),

			// Genre Taxonomy
			'enable_genre_tax'					   => __( 'Enable Taxonomy', 'gs-books-showcase' ),
			'genre_tax_label'					   => __( 'Taxonomy Label', 'gs-books-showcase' ),
			'genre_tax_plural_label'			   => __( 'Taxonomy Plural Label', 'gs-books-showcase' ),
			'enable_genre_tax_archive'			   => __( 'Enable Taxonomy Archive', 'gs-books-showcase' ),
			'genre_tax_archive_slug'			   => __( 'Taxonomy Archive Slug', 'gs-books-showcase' ),

			// Series Taxonomy
			'enable_series_tax'					   => __( 'Enable Taxonomy', 'gs-books-showcase' ),
			'series_tax_label'					   => __( 'Taxonomy Label', 'gs-books-showcase' ),
			'series_tax_plural_label'			   => __( 'Taxonomy Plural Label', 'gs-books-showcase' ),
			'enable_series_tax_archive'			   => __( 'Enable Taxonomy Archive', 'gs-books-showcase' ),
			'series_tax_archive_slug'			   => __( 'Taxonomy Archive Slug', 'gs-books-showcase' ),

			// Publisher Taxonomy
			'enable_publisher_tax'					   => __( 'Enable Taxonomy', 'gs-books-showcase' ),
			'publisher_tax_label'					   => __( 'Taxonomy Label', 'gs-books-showcase' ),
			'publisher_tax_plural_label'			   => __( 'Taxonomy Plural Label', 'gs-books-showcase' ),
			'enable_publisher_tax_archive'			   => __( 'Enable Taxonomy Archive', 'gs-books-showcase' ),
			'publisher_tax_archive_slug'			   => __( 'Taxonomy Archive Slug', 'gs-books-showcase' ),			// Publisher Taxonomy
			
			// Langugae Taxonomy
			'enable_language_tax'					   => __( 'Enable Taxonomy', 'gs-books-showcase' ),
			'language_tax_label'					   => __( 'Taxonomy Label', 'gs-books-showcase' ),
			'language_tax_plural_label'			       => __( 'Taxonomy Plural Label', 'gs-books-showcase' ),
			'enable_language_tax_archive'			   => __( 'Enable Taxonomy Archive', 'gs-books-showcase' ),
			'language_tax_archive_slug'			       => __( 'Taxonomy Archive Slug', 'gs-books-showcase' ),

			// Country Taxonomy
			'enable_country_tax'					   => __( 'Enable Taxonomy', 'gs-books-showcase' ),
			'country_tax_label'					       => __( 'Taxonomy Label', 'gs-books-showcase' ),
			'country_tax_plural_label'			       => __( 'Taxonomy Plural Label', 'gs-books-showcase' ),
			'enable_country_tax_archive'			   => __( 'Enable Taxonomy Archive', 'gs-books-showcase' ),
			'country_tax_archive_slug'			       => __( 'Taxonomy Archive Slug', 'gs-books-showcase' ),

			'taxonomies-page' 					      => __('Taxonomies', 'gs-books-showcase'),
			'taxonomies-page--des' 				     => __('Global settings for Taxonomies', 'gs-books-showcase'),
			'taxonomy_group' 					    => $this->get_tax_option( 'group_tax_plural_label' ),
			'taxonomy_tag' 						    => $this->get_tax_option( 'tag_tax_plural_label' ),
			'author_tag' 						    => $this->get_tax_option( 'author_tax_plural_label', 'Authors' ),
			'genre_tag' 						    => $this->get_tax_option( 'genre_tax_plural_label', 'Genres' ),
			'series_tag' 						    => $this->get_tax_option( 'series_tax_plural_label', 'Series' ),
			'languages_tag' 						 => $this->get_tax_option( 'language_tax_plural_label', 'Languages' ),
			'publisher_tag' 						 => $this->get_tax_option( 'publisher_tax_plural_label', 'Publishers' ),
			'countries_tag' 					    => $this->get_tax_option( 'country_tax_plural_label', 'Countries' ),
			'pages'                                => __( 'Pages', 'gs-books-showcase' ),
			'show_all'                             => __( 'Show All', 'gs-books-showcase' ),
			'show_all_text'                        => __( 'Show All', 'gs-books-showcase' ),
			'country'                              => __( 'Country', 'gs-books-showcase' ),
			'language'                             => __( 'Language', 'gs-books-showcase' ),
			'dimension'                            => __( 'Dimension', 'gs-books-showcase' ),
			'filter_style'                         => __( 'Filter Styles', 'gs-books-showcase' ),
			'file-size-e-book'                     => __( 'File size(e-book)', 'gs-books-showcase' ),
			'reader-s-review'                      => __( 'Reader\'s Review', 'gs-books-showcase' ),
			'rating'                               => __( 'Rating', 'gs-books-showcase' ),
			'store'                                => __( 'Store', 'gs-books-showcase' ),
			'description'                          => __( 'Description', 'gs-books-showcase' ),
			'author-details'                       => __( 'Author Details', 'gs-books-showcase' ),
			'book-details'                         => __( 'Book Details', 'gs-books-showcase' ),
			'download-url'                         => __( 'Download URL', 'gs-books-showcase' ),
			'download'                             => __( 'Download', 'gs-books-showcase' ),
			'short-description' 				   => __( 'Short Description', 'gs-books-showcase' ),
			'book-author-section' 				   => __( 'Author Section', 'gs-books-showcase' ),
			'book-related-section' 				   => __( 'Related Section', 'gs-books-showcase' ),
			'book-author-social' 				   => __( 'Author Social', 'gs-books-showcase' ),
			'book-author-image' 				   => __( 'Author Image', 'gs-books-showcase' ),
			'export-books-data' 				   => __( 'Export Books', 'gs-books-showcase'),
			'export-shortcodes-data' 			   => __( 'Export Shortcodes', 'gs-books-showcase'),
			'export-settings-data' 				   => __( 'Export Settings', 'gs-books-showcase'),
			'readers-rating' 			   		   => __( 'Readers Rating', 'gs-books-showcase' ),
			'accessibility-features' 			   => __( 'Accessibility Features', 'gs-books-showcase' ),
			'by'                      			   => __( 'by', 'gs-books-showcase' ),
			'age-group'               			   => __( 'Age Group', 'gs-books-showcase' ),
			'book-availability'       			   => __( 'Book Availability', 'gs-books-showcase' ),
			'book-awards'             			   => __( 'Book Awards', 'gs-books-showcase' ),
			'book-cover'              			   => __( 'Book Cover', 'gs-books-showcase' ),
			'doi'                     			   => __( 'DOI', 'gs-books-showcase' ),
			'book-edition-features'   			   => __( 'Book Edition Features', 'gs-books-showcase' ),
			'book-formats'            			   => __( 'Book Formats', 'gs-books-showcase' ),
			'llcn'                    			   => __( 'LLCN', 'gs-books-showcase' ),
			'pre-order-availability'  			   => __( 'Pre Order Availability', 'gs-books-showcase' ),
			'reading-level'           			   => __( 'Reading Level', 'gs-books-showcase' ),
			'book-reading-time'       			   => __( 'Book Reading Time', 'gs-books-showcase' ),
			'regular-price'           			   => __( 'Regular Price', 'gs-books-showcase' ),
			'sale-price'              			   => __( 'Sale Price', 'gs-books-showcase' ),
			'weight'                  			   => __( 'Weight', 'gs-books-showcase' ),
			'book-edition'            			   => __( 'Book Edition', 'gs-books-showcase' ),
			'search-by-name'          			   => __( 'Search By Name', 'gs-books-showcase' ),
			'search-by-isbn'          			   => __( 'Search By ISBN', 'gs-books-showcase' ),
			'search-by-asin'          			   => __( 'Search By ASIN', 'gs-books-showcase' ),
			
			'readers-rating--details'      		   => __( 'Replace preferred text for Readers Rating', 'gs-books-showcase' ),
			'accessibility-features--details'      => __( 'Replace preferred text for Accessibility Features', 'gs-books-showcase' ),
			'by--details'                          => __( 'Replace preferred text for by', 'gs-books-showcase' ),
			'age-group--details'                   => __( 'Replace preferred text for Age Group', 'gs-books-showcase' ),
			'book-availability--details'           => __( 'Replace preferred text for Book Availability', 'gs-books-showcase' ),
			'book-awards--details'                 => __( 'Replace preferred text for Book Awards', 'gs-books-showcase' ),
			'book-cover--details'                  => __( 'Replace preferred text for Book Cover', 'gs-books-showcase' ),
			'doi--details'                         => __( 'Replace preferred text for DOI', 'gs-books-showcase' ),
			'book-edition-features--details'       => __( 'Replace preferred text for Book Edition Features', 'gs-books-showcase' ),
			'book-formats--details'                => __( 'Replace preferred text for Book Formats', 'gs-books-showcase' ),
			'llcn--details'                        => __( 'Replace preferred text for LLCN', 'gs-books-showcase' ),
			'pre-order-availability--details'      => __( 'Replace preferred text for Pre Order Availability', 'gs-books-showcase' ),
			'reading-level--details'               => __( 'Replace preferred text for Reading Level', 'gs-books-showcase' ),
			'book-reading-time--details'           => __( 'Replace preferred text for Book Reading Time', 'gs-books-showcase' ),
			'regular-price--details'               => __( 'Replace preferred text for Regular Price', 'gs-books-showcase' ),
			'sale-price--details'                  => __( 'Replace preferred text for Sale Price', 'gs-books-showcase' ),
			'weight--details'                      => __( 'Replace preferred text for Weight', 'gs-books-showcase' ),
			'book-edition--details'                => __( 'Replace preferred text for Book Edition', 'gs-books-showcase' ),
			'search-by-name--details'              => __( 'Replace preferred text for Search By Name', 'gs-books-showcase' ),
			'search-by-isbn--details'              => __( 'Replace preferred text for Search By ISBN', 'gs-books-showcase' ),
			'search-by-asin--details'              => __( 'Replace preferred text for Search By ASIN', 'gs-books-showcase' ),

			'show-all-publisher'                   => __( 'Show All Publisher', 'gs-books-showcase' ),
			'search-by-book-name'                  => __( 'Search By Book Name', 'gs-books-showcase' ),
			'all'                                  => __( 'All', 'gs-books-showcase' ),
			'select-book-name'                     => __( 'Select Book Name', 'gs-books-showcase' ),
			'similar_category_books' 			   => __( 'Similar Category Books', 'gs-books-showcase' ),
			'other_authors' 			   		   => __( 'Other Authors', 'gs-books-showcase' ),
			'more_books_by' 			   		   => __( 'More books by', 'gs-books-showcase' ),
			'categories--details' 				   => __( 'Replace preferred text for Categories', 'gs-books-showcase' ),
			'tags--details' 					   => __( 'Replace preferred text for Tags', 'gs-books-showcase' ),
			'authors--details' 					   => __( 'Replace preferred text for Authors', 'gs-books-showcase' ),
			'genres--details' 					   => __( 'Replace preferred text for Genres', 'gs-books-showcase' ),
			'series--details' 					   => __( 'Replace preferred text for Series', 'gs-books-showcase' ),
			'languages--details' 				   => __( 'Replace preferred text for Languages', 'gs-books-showcase' ),
			'publishers--details' 				   => __( 'Replace preferred text for Publishers', 'gs-books-showcase' ),
			'countries--details' 				   => __( 'Replace preferred text for Countries', 'gs-books-showcase' ),
			'more--details'                        => __( 'Replace preferred text for More', 'gs-books-showcase' ),
			'publish--details'                     => __( 'Replace preferred text for Publish', 'gs-books-showcase' ),
			'publisher--details'                   => __( 'Replace preferred text for Publisher', 'gs-books-showcase' ),
			'translator--details'                  => __( 'Replace preferred text for Translator', 'gs-books-showcase' ),
			'format--details'                      => __( 'Replace preferred text for Format', 'gs-books-showcase' ),
			'isbn--details'                        => __( 'Replace preferred text for ISBN', 'gs-books-showcase' ),
			'asin--details'                        => __( 'Replace preferred text for ASIN', 'gs-books-showcase' ),
			'pages--details'                       => __( 'Replace preferred text for Pages', 'gs-books-showcase' ),
			'country--details'                     => __( 'Replace preferred text for Country', 'gs-books-showcase' ),
			'language--details'                    => __( 'Replace preferred text for Language', 'gs-books-showcase' ),
			'dimension--details'                   => __( 'Replace preferred text for Dimension', 'gs-books-showcase' ),
			'file-size-e-book--details'            => __( 'Replace preferred text for File size(e-book)', 'gs-books-showcase' ),
			'reader-s-review--details'             => __( 'Replace preferred text for Reader\'s Review', 'gs-books-showcase' ),
			'rating--details'                      => __( 'Replace preferred text for Rating', 'gs-books-showcase' ),
			'store--details'                       => __( 'Replace preferred text for Store', 'gs-books-showcase' ),
			'description--details'                 => __( 'Replace preferred text for Description', 'gs-books-showcase' ),
			'author-details--details'              => __( 'Replace preferred text for Author Details', 'gs-books-showcase' ),
			'book-details--details'                => __( 'Replace preferred text for Book Details (Theme 15 & 16 : Flip & Flip 3D)', 'gs-books-showcase' ),
			'download-url--details'                => __( 'Replace preferred text for Download URL', 'gs-books-showcase' ),
			'download--details'                    => __( 'Replace preferred text for Download', 'gs-books-showcase' ),
			'show-all-publisher--details'          => __( 'Replace preferred text (Theme 14 & 18 : Filter & Single, Filter & Search)', 'gs-books-showcase' ),
			'search-by-book-name--details'         => __( 'Replace preferred text (Theme 18 : Filter & Search)', 'gs-books-showcase' ),
			'all--details'                         => __( 'Replace preferred text for All (Filter Theme)', 'gs-books-showcase' ),
			'select-book-name--details'            => __( 'Replace preferred text (Theme 14 : Filter & Single)', 'gs-books-showcase' ),
			'similar_category_books--details' 	   => __( 'Replace preferred text Similar Category Books', 'gs-books-showcase' ),
			'other_authors--details' 	   		   => __( 'Replace preferred text Other Authors', 'gs-books-showcase' ),
			'more_books_by--details' 	   		   => __( 'Replace preferred text More books by', 'gs-books-showcase' ),
			'save-localization'                    => __( 'Save Localization', 'gs-books-showcase' ),
			'search_by_name'                       => __( 'Filter By Name', 'gs-books-showcase' ),
			'search_by_isbn'                       => __( 'Filter By ISBN', 'gs-books-showcase' ),
			'search_by_asin'                       => __( 'Filter By ASIN', 'gs-books-showcase' ),
			'search_by_countries'                  => __( 'Filter By Countries', 'gs-books-showcase' ),
			'search_by_publishers'                 => __( 'Filter By Publishers', 'gs-books-showcase' ),
			'search_by_languages'                  => __( 'Filter By Languages', 'gs-books-showcase' ),
			'search_by_tags'                       => __( 'Filter By Tags', 'gs-books-showcase' ),
			'search_by_categories'                 => __( 'Filter By Categories', 'gs-books-showcase' ),
			'search_by_genres'                     => __( 'Filter By Genres', 'gs-books-showcase' ),
			'search_by_authors'                    => __( 'Filter By Authors', 'gs-books-showcase' ),
			'search_by_series'                     => __( 'Filter By Series', 'gs-books-showcase' ),
			'gs_book_pagination'                   => __( 'Pagination', 'gs-books-showcase' ),
			'posts_per_page'                       => __( 'Per Page', 'gs-books-showcase' ),
			'gs_book_enable_multi_select'          => __( 'Multi Select', 'gs-books-showcase' ),
			'gs_book_multi_select_ellipsis'        => __( 'Multi Select Elipsis', 'gs-books-showcase' ),
			'gs_book_category'                     => __( 'Book Category', 'gs-books-showcase' ),
			'gs_book_category__details'            => __( 'Show or Hide All Books Category', 'gs-books-showcase' ),
			'gs_book_translator'                   => __( 'Book Translator', 'gs-books-showcase' ),
			'gs_book_translator__details'          => __( 'Show or Hide All Books Translator', 'gs-books-showcase' ),
			'gs_book_isbn'                         => __( 'ISBN 10', 'gs-books-showcase' ),
			'gs_book_isbn_thirteen'				   => __( 'ISBN 13', 'gs-books-showcase'),
			'gs_book_isbn__details'                => __( 'Show or Hide All Books ISBN', 'gs-books-showcase' ),
			'gs_book_asin'                         => __( 'Book ASIN', 'gs-books-showcase' ),
			'include_tags'                         => __( 'Tags', 'gs-books-showcase' ),
			'exclude_tags'                         => __( 'Tags', 'gs-books-showcase' ),
			'gs_book_asin__details'                => __( 'Show or Hide All Books ASIN', 'gs-books-showcase' ),
			'gs_book_pages'                        => __( 'Book Pages', 'gs-books-showcase' ),
			'gs_book_pages__details'               => __( 'Show or Hide All Books Pages', 'gs-books-showcase' ),
			'gs_book_country'                      => __( 'Book Country', 'gs-books-showcase' ),
			'gs_book_country__details'             => __( 'Show or Hide All Books Country', 'gs-books-showcase' ),
			'gs_book_language'                     => __( 'Book Language', 'gs-books-showcase' ),
			'gs_book_language__details'            => __( 'Show or Hide All Books Language', 'gs-books-showcase' ),
			'gs_book_dimension'                    => __( 'Book Dimension', 'gs-books-showcase' ),
			'gs_book_dimension__details'           => __( 'Show or Hide All Books dimension', 'gs-books-showcase' ),
			'gs_book_fsize'                        => __( 'Book File Size (e-book)', 'gs-books-showcase' ),
			'gs_book_fsize__details'               => __( 'Show or Hide All Books File Size for e-book', 'gs-books-showcase' ),
			'gs_book_authordes'                    => __( 'Book Author Details', 'gs-books-showcase' ),
			'gs_book_authordes__details'           => __( 'Show or Hide All Books Author Details', 'gs-books-showcase' ),
			'gs-filter-by'                         => __( 'Filter By', 'gs-books-showcase' ),
			'gs_store_img'                         => __( 'Store Image', 'gs-books-showcase' ),
			'gs_book_sin_nxt_prev'                 => __( 'Next / Previous', 'gs-books-showcase' ),
			'gs_book_sin_nxt_prev__details'        => __( 'Show or Hide Next / Previous link at Single Book Template', 'gs-books-showcase' ),
			'enable-clear-filters' 				   => __( 'Reset Filters Button', 'gs-books-showcase' ),
			'enable-clear-filters--help' 		   => __( 'Enable Reset all filters button in filter themes, Default is Off', 'gs-books-showcase' ),

			// STYLE
			'include_authors'                      => __( 'Include Authors', 'gs-books-showcase' ),
			'exclude_authors'                      => __( 'Exclude Authors', 'gs-books-showcase' ),
			'include_genres'                       => __( 'Include Genres', 'gs-books-showcase' ),
			'exclude_genres'                       => __( 'Exclude Genres', 'gs-books-showcase' ),
			'include_series'                       => __( 'Include Series', 'gs-books-showcase' ),
			'exclude_series'                       => __( 'Exclude Series', 'gs-books-showcase' ),
			'include_languages'                    => __( 'Include Languages', 'gs-books-showcase' ),
			'exclude_languages'                    => __( 'Exclude Languages', 'gs-books-showcase' ),
			'include_publishers'                   => __( 'Include Publishers', 'gs-books-showcase' ),
			'exclude_publishers'                   => __( 'Exclude Publishers', 'gs-books-showcase' ),
			'include_countries'                    => __( 'Include Countries', 'gs-books-showcase' ),
			'exclude_countries'                    => __( 'Exclude Countries', 'gs-books-showcase' ),
			'gs_book_title_font'                   => __( 'Font Family', 'gs-books-showcase' ),
			'gs-book-fz'                           => __( 'Font Size', 'gs-books-showcase' ),
			'gs-book-fz--details'                  => __( 'Set Font Size for Book Name', 'gs-books-showcase' ),
			'gs-book-fntw'                         => __( 'Font Weight', 'gs-books-showcase' ),
			'gs-book-fntw--details'                => __( 'Select Font Weight for Book Name', 'gs-books-showcase' ),
			'gs-book-fnstyl'                       => __( 'Font Style', 'gs-books-showcase' ),
			'speed'                                => __( 'Speed', 'gs-books-showcase' ),
			'autoplay'                             => __( 'Autoplay', 'gs-books-showcase' ),
			'autoplay_delay'                       => __( 'Autoplay Delay', 'gs-books-showcase' ),
			'gsb-slider-navs'                      => __( 'Slider Navs', 'gs-books-showcase' ),
			'gsb-slider-dots'                      => __( 'Slider Dots', 'gs-books-showcase' ),
			'gsb-navs-pos'                         => __( 'Navs Position', 'gs-books-showcase' ),
			'gsb_dots_style'                       => __( 'Dot Style', 'gs-books-showcase' ),
			'gsb_navs_style'                       => __( 'Navs Style', 'gs-books-showcase' ),
			'pause_on_hover'                       => __( 'Pause on Hover', 'gs-books-showcase' ),
			'infinite_loop'                        => __( 'Infinite Loop', 'gs-books-showcase' ),
			'reverse_direction'                    => __( 'Reverse Direction', 'gs-books-showcase' ),
			'gs-book-fnstyl--details'              => __( 'Select Font Weight for Book Name', 'gs-books-showcase' ),
			'gs-book-name-color'                   => __( 'Font Color', 'gs-books-showcase' ),
			'gs_book_rating_color'                 => __( 'Rating Color', 'gs-books-showcase' ),
			'gs-book-rating-color--details'        => __( 'Select Rating Color', 'gs-books-showcase' ),
			'gs-book-name-color--details'          => __( 'Select color for Book Name', 'gs-books-showcase' ),
			'gs-book-label-fz'                     => __( 'Label Font Size', 'gs-books-showcase' ),
			'gs-book-label-fz--details'            => __( 'Set Font Size for Book Label', 'gs-books-showcase' ),
			'gs-book-label-color'                  => __( 'Label Font Color', 'gs-books-showcase' ),
			'gs-book-label-color--details'         => __( 'Select color for Book Label', 'gs-books-showcase' ),
			'gs-books-btn-nav-ctrl-color'          => __( 'Popup Btn, Nav & Close Color', 'gs-books-showcase' ),
			'gs-books-btn-nav-ctrl-color--details' => __( 'Select color for Popup Btn, Nav & Close Button', 'gs-books-showcase' ),
			'gs-filter-cat-pos'                    => __( 'Filter Category Position', 'gs-books-showcase' ),
			'gs-filter-cat-pos--details'           => __( 'Select Filter Category Position', 'gs-books-showcase' ),
			'gs_sin_book_txt_align'                => __( 'Book Details Text Alignment', 'gs-books-showcase' ),
			'gs_sin_book_txt_align--details'       => __( 'Select Book Details Text Alignment at Single Book Template', 'gs-books-showcase' ),

			// OLD
			'theme'                                => __( 'Style & Theming', 'gs-books-showcase' ),
			'theme--placeholder'                   => __( 'Select Theme', 'gs-books-showcase' ),
			'theme--help'                          => __( 'Select preferred Style & Theme', 'gs-books-showcase' ),
			'view_type'                            => __( 'View Type', 'gs-books-showcase' ),
			'view-type--help'                      => __( 'Select preferred View Type', 'gs-books-showcase' ),
			'posts'                                => __( 'Books/ Authors', 'gs-books-showcase' ),
			'posts--placeholder'                   => __( 'Books', 'gs-books-showcase' ),
			'posts--help'                          => __( 'Set max book numbers you want to show, set -1 for all books', 'gs-books-showcase' ),
			'order'                                => __( 'Order', 'gs-books-showcase' ),
			'order--placeholder'                   => __( 'Order', 'gs-books-showcase' ),
			'order-by'                             => __( 'Order By', 'gs-books-showcase' ),
			'group_order'                          => __( 'Group Order', 'gs-books-showcase' ),
			'group_order_by'                       => __( 'Group Order By', 'gs-books-showcase' ),
			'order-by--placeholder'                => __( 'Order By', 'gs-books-showcase' ),
			'group'                                => __( 'Categories', 'gs-books-showcase' ),
			'group__help'                          => __( 'Select specific book group to show that specific group books', 'gs-books-showcase' ),
			'exclude_group'                        => __( 'Categories', 'gs-books-showcase' ),
			'exclude_group__help'                  => __( 'Select specific book group to hide that specific group books', 'gs-books-showcase' ),
			'install-demo-data'                    => __( 'Install Demo Data', 'gs-books-showcase' ),
			'install-demo-data-description'        => __( 'Quick start with GS Book Showcase by installing the demo data', 'gs-books-showcase' ),
			'preference'                           => __( 'Preference', 'gs-books-showcase' ),
			'save-preference'                      => __( 'Save Preference', 'gs-books-showcase' ),
			'custom-css'                           => __( 'Custom CSS', 'gs-books-showcase' ),
			'shortcodes'                           => __( 'Shortcodes', 'gs-books-showcase' ),
			'create-shortcode'                     => __( 'Create Shortcode', 'gs-books-showcase' ),
			'create-new-shortcode'                 => __( 'Create New Shortcode', 'gs-books-showcase' ),
			'shortcode'                            => __( 'Shortcode', 'gs-books-showcase' ),
			'name'                                 => __( 'Name', 'gs-books-showcase' ),
			'action'                               => __( 'Action', 'gs-books-showcase' ),
			'actions'                              => __( 'Actions', 'gs-books-showcase' ),
			'edit'                                 => __( 'Edit', 'gs-books-showcase' ),
			'clone'                                => __( 'Clone', 'gs-books-showcase' ),
			'delete'                               => __( 'Delete', 'gs-books-showcase' ),
			'delete-all'                           => __( 'Delete All', 'gs-books-showcase' ),
			'pref-search-all-fields' 				=> __('Include fields when search', 'gs-books-showcase'),
			'pref-search-all-fields-details' 		=> __('Enable searching through all fields', 'gs-books-showcase'),
			'create-a-new-shortcode-and'           => __( 'Create a new shortcode & save it to use globally in anywhere', 'gs-books-showcase' ),
			'edit-shortcode'                       => __( 'Edit Shortcode', 'gs-books-showcase' ),
			'general-settings'                     => __( 'General', 'gs-books-showcase' ),
			'style-settings'                       => __( 'Style', 'gs-books-showcase' ),
			'query-settings'                       => __( 'Query', 'gs-books-showcase' ),
			'visibility-settings' 				   => __( 'Visibility', 'gs-books-showcase' ),
			'shortcode-name'                       => __( 'Shortcode Name', 'gs-books-showcase' ),
			'name-of-the-shortcode'                => __( 'Name of the Shortcode', 'gs-books-showcase' ),
			'save-shortcode'                       => __( 'Save Shortcode', 'gs-books-showcase' ),
			'preview-shortcode'                    => __( 'Preview Shortcode', 'gs-books-showcase' ),

			'desktop'                              => __( 'Desktop', 'gs-books-showcase' ),
			'tablet'                               => __( 'Tablet', 'gs-books-showcase' ),
			'mobile_landscape'                     => __( 'Large Mobile', 'gs-books-showcase' ),
			'mobile'                               => __( 'Mobile', 'gs-books-showcase' ),

			'columns'                              => __( 'Columns', 'gs-books-showcase' ),
			'columns__details'                     => __( 'Select column for desktop', 'gs-books-showcase' ),
			'columns_tablet__details'              => __( 'Select column for tablet', 'gs-books-showcase' ),
			'columns_mobile_portrait__details'     => __( 'Select column for large display mobile', 'gs-books-showcase' ),
			'columns_mobile__details'              => __( 'Select column for mobile', 'gs-books-showcase' ),
		);
	}

	public function get_shortcode_default_localization() {
		return array(
			'gsb_categories_text_modify' 		=> __( 'Categories', 'gs-books-showcase' ),
			'gsb_tags_text_modify' 				=> __( 'Tags', 'gs-books-showcase' ),
			'gsb_authors_text_modify' 			=> __( 'Authors', 'gs-books-showcase' ),
			'gsb_genres_text_modify' 			=> __( 'Genres', 'gs-books-showcase' ),
			'gsb_series_text_modify' 			=> __( 'Series', 'gs-books-showcase' ),
			'gsb_languages_text_modify' 		=> __( 'Languages', 'gs-books-showcase' ),
			'gsb_publishers_text_modify' 		=> __( 'Publishers', 'gs-books-showcase' ),
			'gsb_countries_text_modify' 		=> __( 'Countries', 'gs-books-showcase' ),

			'gsb_more_text_modify'             => __( 'More', 'gs-books-showcase' ),
			'gsb_publish_text_modify'          => __( 'Publish Date', 'gs-books-showcase' ),
			'gsb_co_publisher_text_modify'     => __( 'Co-Publisher', 'gs-books-showcase' ),
			'gsb_translator_text_modify'       => __( 'Translator', 'gs-books-showcase' ),
			'gsb_format_text_modify'           => __( 'Format', 'gs-books-showcase' ),
			'gsb_isbn_text_modify'             => __( 'ISBN 10', 'gs-books-showcase' ),
			'gsb_isbn13_text_modify'           => __( 'ISBN 13', 'gs-books-showcase'),
			'gsb_asin_text_modify'             => __( 'ASIN', 'gs-books-showcase' ),
			'gsb_pages_text_modify'            => __( 'Pages', 'gs-books-showcase' ),
			'gsb_country_text_modify'          => __( 'Country', 'gs-books-showcase' ),
			'gsb_language_text_modify'         => __( 'Language', 'gs-books-showcase' ),
			'gsb_dimension_text_modify'        => __( 'Dimension', 'gs-books-showcase' ),
			'gsb_filesize_text_modify'         => __( 'File size(e-book)', 'gs-books-showcase' ),
			'gsb_readers_text_modify'          => __( "Reader's Review", 'gs-books-showcase' ),
			'gsb_rating_text_modify'           => __( 'Rating', 'gs-books-showcase' ),
			'gsb_store_text_modify'            => __( 'Store', 'gs-books-showcase' ),
			'gsb_description_text_modify'      => __( 'Description', 'gs-books-showcase' ),
			'gsb_author_details_text_modify'   => __( 'Author Details', 'gs-books-showcase' ),
			'gsb_book_details_text_modify'     => __( 'Book Details', 'gs-books-showcase' ),
			'gsb_download_url_text_modify'     => __( 'Download URL', 'gs-books-showcase' ),
			'gsb_download_text_modify'         => __( 'Download', 'gs-books-showcase' ),
			'gsb_show_all_publisher_text_modify'      => __( 'Show All Publisher', 'gs-books-showcase' ),
			'gsb_search_by_text_modify'               => __( 'Search By Book Name', 'gs-books-showcase' ),
			'gsb_book_name_text_modify'       		  => __( 'Select Book Name', 'gs-books-showcase' ),
			'gsb_sim_cat_books_text_modify'    		  => __( 'Similar Category Books', 'gs-books-showcase' ),
			'gsb_other_authors_text_modify'    		  => __( 'Other Authors', 'gs-books-showcase' ),
			'gsb_more_books_by_text_modify'    		  => __( 'More books by', 'gs-books-showcase' ),
			'gsb_readers_rating_text_modify'   		  => __( 'Readers Rating', 'gs-books-showcase' ),
			'gsb_accessibility_features_text_modify'  => __( 'Accessibility Features', 'gs-books-showcase' ),
			'gsb_by_text_modify'                      => __( 'by', 'gs-books-showcase' ),
			'gsb_age_group_text_modify'               => __( 'Age Group', 'gs-books-showcase' ),
			'gsb_book_availability_text_modify'       => __( 'Book Availability', 'gs-books-showcase' ),
			'gsb_book_awards_text_modify'             => __( 'Book Awards', 'gs-books-showcase' ),
			'gsb_book_cover_text_modify'              => __( 'Book Cover', 'gs-books-showcase' ),
			'gsb_doi_text_modify'                     => __( 'DOI', 'gs-books-showcase' ),
			'gsb_book_edition_features_text_modify'   => __( 'Book Edition Features', 'gs-books-showcase' ),
			'gsb_book_formats_text_modify'            => __( 'Book Formats', 'gs-books-showcase' ),
			'gsb_llcn_text_modify'                    => __( 'LLCN', 'gs-books-showcase' ),
			'gsb_pre_order_availability_text_modify'  => __( 'Pre Order Availability', 'gs-books-showcase' ),
			'gsb_reading_level_text_modify'           => __( 'Reading Level', 'gs-books-showcase' ),
			'gsb_book_reading_time_text_modify'       => __( 'Book Reading Time', 'gs-books-showcase' ),
			'gsb_regular_price_text_modify'           => __( 'Regular Price', 'gs-books-showcase' ),
			'gsb_sale_price_text_modify'              => __( 'Sale Price', 'gs-books-showcase' ),
			'gsb_weight_text_modify'                  => __( 'Weight', 'gs-books-showcase' ),
			'gsb_book_edition_text_modify'            => __( 'Book Edition', 'gs-books-showcase' ),
			'gsb_search_by_name_text_modify'          => __( 'Search By Name', 'gs-books-showcase' ),
			'gsb_search_by_isbn_text_modify'          => __( 'Search By ISBN', 'gs-books-showcase' ),
			'gsb_search_by_asin_text_modify'          => __( 'Search By ASIN', 'gs-books-showcase' ),
			'gs_member_enable_multilingual' 		  => false
		);
	}

	public function get_shortcode_options_themes() {

		$free_themes = array(
			array(
				'label' => __( 'Style 01 (New)', 'gs-books-showcase' ),
				'value' => 'style-01',
			),
			array(
				'label' => __( 'Horizontal 03 (New)', 'gs-books-showcase' ),
				'value' => 'horizontal-03',
			),
			array(
				'label' => __( 'Theme 01 (Square)', 'gs-books-showcase' ),
				'value' => 'gs_book_theme1',
			)
		);

		$pro_themes = array(			

			array(
				'label' => __( 'Style 02 (New)', 'gs-books-showcase' ),
				'value' => 'style-02',
			),
			array(
				'label' => __( 'Style 03 (New)', 'gs-books-showcase' ),
				'value' => 'style-03',
			),
			array(
				'label' => __( 'Style 04 (New)', 'gs-books-showcase' ),
				'value' => 'style-04',
			),
			array(
				'label' => __( 'Style 05 (New)', 'gs-books-showcase' ),
				'value' => 'style-05',
			),
			array(
				'label' => __( 'Style 06 (New)', 'gs-books-showcase' ),
				'value' => 'style-06',
			),
			array(
				'label' => __( 'Style 07 (New)', 'gs-books-showcase' ),
				'value' => 'style-07',
			),
			array(
				'label' => __( 'Style 08 (New)', 'gs-books-showcase' ),
				'value' => 'style-08',
			),
			array(
				'label' => __( 'Style 09 (New)', 'gs-books-showcase' ),
				'value' => 'style-09',
			),
			array(
				'label' => __( 'Horizontal 01 (New)', 'gs-books-showcase' ),
				'value' => 'horizontal-01',
			),
			array(
				'label' => __( 'Horizontal 02 (New)', 'gs-books-showcase' ),
				'value' => 'horizontal-02',
			),
			array(
				'label' => __( 'Horizontal 04 (New)', 'gs-books-showcase' ),
				'value' => 'horizontal-04',
			),
			array(
				'label' => __( 'Flip 01 (New)', 'gs-books-showcase' ),
				'value' => 'flip-01',
			),
			array(
				'label' => __( 'Flip 02 (New)', 'gs-books-showcase' ),
				'value' => 'flip-02',
			),
			array(
				'label' => __( 'Flip 03 (New)', 'gs-books-showcase' ),
				'value' => 'flip-03',
			),
			array(
				'label' => __( 'Flip 04 (New)', 'gs-books-showcase' ),
				'value' => 'flip-04',
			),
			array(
				'label' => __( 'Flip 05 (New)', 'gs-books-showcase' ),
				'value' => 'flip-05',
			),
			array(
				'label' => __( '3D Style 01 (New)', 'gs-books-showcase' ),
				'value' => 'three-d-style-01',
			),
			array(
				'label' => __( '3D Style 02 (New)', 'gs-books-showcase' ),
				'value' => 'three-d-style-02',
			),
			array(
				'label' => __( '3D Style 03 (New)', 'gs-books-showcase' ),
				'value' => 'three-d-style-03',
			),
			array(
				'label' => __( '3D Style 04 (New)', 'gs-books-showcase' ),
				'value' => 'three-d-style-04',
			),
			array(
				'label' => __( 'Author Style 01', 'gs-books-showcase' ),
				'value' => 'author-style-01',
			),
			array(
				'label' => __( 'Author Style 02', 'gs-books-showcase' ),
				'value' => 'author-style-02',
			),
			array(
				'label' => __( 'Author Style 03', 'gs-books-showcase' ),
				'value' => 'author-style-03',
			),
			array(
				'label' => __( 'Widget 01', 'gs-books-showcase' ),
				'value' => 'widget-01',
			),
			array(
				'label' => __( 'Widget 02', 'gs-books-showcase' ),
				'value' => 'widget-02',
			),
			array(
				'label' => __( 'Widget 03', 'gs-books-showcase' ),
				'value' => 'widget-03',
			)

		);

		if ( ! is_pro_active() || ! is_pro_valid() ) {
			$pro_themes = array_map( function( $item ) {
				$item['pro'] = true;
				return $item;
			}, $pro_themes );
		}

		return array_merge( $free_themes, $pro_themes );
	}

	public function get_view_types() {

		$free_types = [
			array(
				'label' => __( 'Grid', 'gs-books-showcase' ),
				'value' => 'grid',
			),
			array(
				'label' => __( 'Slider', 'gs-books-showcase' ),
				'value' => 'slider',
			)
		];

		$pro_types = [
			array(
				'label' => __( 'Filter', 'gs-books-showcase' ),
				'value' => 'filter',
			),
			array(
				'label' => __('Ticker', 'gs-books-showcase'),
				'value' => 'ticker',
			)
		];

		if ( ! is_pro_active() || ! is_pro_valid() ) {
			$pro_types = array_map( function( $item ) {
				$item['pro'] = true;
				return $item;
			}, $pro_types );
		}

		return array_merge( $free_types, $pro_types );
		
	}

	public function get_google_fonts() {

		$google_fonts = array(
			array(
				'label' => __( 'Roboto', 'gs-books-showcase' ),
				'value' => 'Roboto, sans-serif',
			),
			array(
				'label' => __( 'Open Sans', 'gs-books-showcase' ),
				'value' => 'Open Sans, sans-serif',
			),
			array(
				'label' => __( 'Poppins', 'gs-books-showcase' ),
				'value' => 'Poppins, sans-serif',
			),
		);

		return $google_fonts;
	}

	public function get_shortcode_options_image_sizes() {

		$sizes = get_intermediate_image_sizes();

		$result = array();

		foreach ( $sizes as $size ) {
			$result[] = array(
				'label' => ucwords( preg_replace( '/_|-/', ' ', $size ) ),
				'value' => $size,
			);
		}

		return $result;
	}

	public function get_link_types() {
		return array(
			array(
				'label' => __( 'Single Page', 'gs-books-showcase' ),
				'value' => 'single_page',
			),
			array(
				'label' => __( 'Popup', 'gs-books-showcase' ),
				'value' => 'popup',
			),
		);
	}

	public function get_columns() {

		return array(
			array(
				'label' => __( '1 Column', 'gs-books-showcase' ),
				'value' => '12',
			),
			array(
				'label' => __( '2 Columns', 'gs-books-showcase' ),
				'value' => '6',
			),
			array(
				'label' => __( '3 Columns', 'gs-books-showcase' ),
				'value' => '4',
			),
			array(
				'label' => __( '4 Columns', 'gs-books-showcase' ),
				'value' => '3',
			),
			array(
				'label' => __( '5 Columns', 'gs-books-showcase' ),
				'value' => '2_4',
			),
			array(
				'label' => __( '6 Columns', 'gs-books-showcase' ),
				'value' => '2',
			),
		);
	}

	public function get_shortcode_default_options() {
		return array(
			'gs_book_title_font'      => $this->get_google_fonts(),
			'theme'                   => $this->get_shortcode_options_themes(),
			'view_type'               => $this->get_view_types(),
			'columns'                 => $this->get_columns(),
			'columns_tablet'          => $this->get_columns(),
			'gsb_navs_style'          => $this->get_navs_styles(),
			'filter_style'            => $this->get_filter_styles(),
			'gsb_filter_by'           => array(
				array(
					'label' => __( 'Categories', 'gs-books-showcase' ),
					'value' => 'category',
				),
				array(
					'label' => __( 'Tags', 'gs-books-showcase' ),
					'value' => 'tag',
				),
			),
			'filter_position'         => array(
				array(
					'label' => __( 'Left', 'gs-books-showcase' ),
					'value' => 'left',
				),
				array(
					'label' => __( 'Center', 'gs-books-showcase' ),
					'value' => 'center',
				),
				array(
					'label' => __( 'Right', 'gs-books-showcase' ),
					'value' => 'right',
				),
			),
			'gsb_navs_pos'            => $this->get_navs_pos(),
			'gsb_dots_style'          => $this->get_dot_style(),
			'columns_mobile_portrait' => $this->get_columns(),
			'columns_mobile'          => $this->get_columns(),
			'link_type'               => $this->get_link_types(),
			'gs_book_thumbnail_sizes' => $this->get_shortcode_options_image_sizes(),
			'popup_style'             => $this->get_popup_styles(),
			'gs_filter_cat_pos'       => array(
				array(
					'label' => __( 'Center', 'gs-books-showcase' ),
					'value' => 'center',
				),
				array(
					'label' => __( 'Left', 'gs-books-showcase' ),
					'value' => 'left',
				),
				array(
					'label' => __( 'Right', 'gs-books-showcase' ),
					'value' => 'right',
				),
			),
			'group'                   => $this->get_book_categories(),
			'exclude_group'           => $this->get_book_categories(),
			'include_tags'            => $this->get_book_tags(),
			'exclude_tags'            => $this->get_book_tags(),
			'include_authors'         => $this->get_book_authors(),
			'exclude_authors'         => $this->get_book_authors(),
			'include_genres'          => $this->get_book_genres(),
			'exclude_genres'          => $this->get_book_genres(),
			'include_series'          => $this->get_book_series(),
			'exclude_series'          => $this->get_book_series(),
			'include_languages'       => $this->get_book_languages(),
			'exclude_languages'       => $this->get_book_languages(),
			'include_publishers'      => $this->get_book_publishers(),
			'exclude_publishers'      => $this->get_book_publishers(),
			'include_countries'       => $this->get_book_countries(),
			'exclude_countries'       => $this->get_book_countries(),
			'orderby'                 => array(
				array(
					'label' => __( 'Custom Order', 'gs-books-showcase' ),
					'value' => 'menu_order',
				),
				array(
					'label' => __( 'Book ID', 'gs-books-showcase' ),
					'value' => 'ID',
				),
				array(
					'label' => __( 'Book Name', 'gs-books-showcase' ),
					'value' => 'title',
				),
				array(
					'label' => __( 'Date', 'gs-books-showcase' ),
					'value' => 'date',
				),
				array(
					'label' => __( 'Random', 'gs-books-showcase' ),
					'value' => 'rand',
				),
			),
			'group_order_by'          => array(
				array(
					'label' => __( 'Custom Order', 'gs-books-showcase' ),
					'value' => 'term_order',
				),
				array(
					'label' => __( 'Group ID', 'gs-books-showcase' ),
					'value' => 'term_id',
				),
				array(
					'label' => __( 'Group Name', 'gs-books-showcase' ),
					'value' => 'name',
				),
			),
			'group_order'             => array(
				array(
					'label' => __( 'DESC', 'gs-books-showcase' ),
					'value' => 'DESC',
				),
				array(
					'label' => __( 'ASC', 'gs-books-showcase' ),
					'value' => 'ASC',
				),
			),
			'order'                   => array(
				array(
					'label' => __( 'DESC', 'gs-books-showcase' ),
					'value' => 'DESC',
				),
				array(
					'label' => __( 'ASC', 'gs-books-showcase' ),
					'value' => 'ASC',
				),
			),
			'gs_book_fntw'            => array(
				array(
					'label' => __( 'Normal', 'gs-books-showcase' ),
					'value' => 'normal',
				),
				array(
					'label' => __( 'Bold', 'gs-books-showcase' ),
					'value' => 'bold',
				),
				array(
					'label' => __( 'Light', 'gs-books-showcase' ),
					'value' => 'light',
				),
			),
			'gs_book_fnstyl'          => array(
				array(
					'label' => __( 'Normal', 'gs-books-showcase' ),
					'value' => 'normal',
				),
				array(
					'label' => __( 'Italic', 'gs-books-showcase' ),
					'value' => 'italic',
				),
			),
			'gs_sin_book_txt_align'   => array(
				array(
					'label' => __( 'Left', 'gs-books-showcase' ),
					'value' => 'left',
				),
				array(
					'label' => __( 'Center', 'gs-books-showcase' ),
					'value' => 'center',
				),
				array(
					'label' => __( 'Right', 'gs-books-showcase' ),
					'value' => 'right',
				),
			),
			'image_filter' 			  => $this->get_image_filter_effects(),
			'hover_image_filter' 	  => $this->get_image_filter_effects(),
			'gs_member_enable_clear_filters'  => false,
		);
	}

	public function get_popup_styles() {

		$free_popups = array(				
			array(
				'label' => __( 'Style One', 'gs-books-showcase' ),
				'value' => 'style_one',
			),
		);

		$pro_popups = array(
			array(
				'label' => __( 'Style Two', 'gs-books-showcase' ),
				'value' => 'style_two',
			),
			array(
				'label' => __( 'Style Three', 'gs-books-showcase' ),
				'value' => 'style_three',
			),
			array(
				'label' => __( 'Style Four', 'gs-books-showcase' ),
				'value' => 'style_four',
			),
			array(
				'label' => __( 'Style Five', 'gs-books-showcase' ),
				'value' => 'style_five',
			),
		);

		if ( ! is_pro_active() || ! is_pro_valid() ) {
			$pro_popups = array_map( function( $item ) {
				$item['pro'] = true;
				return $item;
			}, $pro_popups );
		}

		return array_merge( $free_popups, $pro_popups );
	} 

	public function get_image_filter_effects() {

		$effects = [
			[
				'label' => __( 'None', 'gs-books-showcase' ),
				'value' => 'none'
			],
			[
				'label' => __( 'Blur', 'gs-books-showcase' ),
				'value' => 'blur'
			],
			[
				'label' => __( 'Brightness', 'gs-books-showcase' ),
				'value' => 'brightness'
			],
			[
				'label' => __( 'Contrast', 'gs-books-showcase' ),
				'value' => 'contrast'
			],
			[
				'label' => __( 'Grayscale', 'gs-books-showcase' ),
				'value' => 'grayscale'
			],
			[
				'label' => __( 'Hue Rotate', 'gs-books-showcase' ),
				'value' => 'hue_rotate'
			],
			[
				'label' => __( 'Invert', 'gs-books-showcase' ),
				'value' => 'invert'
			],
			[
				'label' => __( 'Opacity', 'gs-books-showcase' ),
				'value' => 'opacity'
			],
			[
				'label' => __( 'Saturate', 'gs-books-showcase' ),
				'value' => 'saturate'
			],
			[
				'label' => __( 'Sepia', 'gs-books-showcase' ),
				'value' => 'sepia'
			]
		];

		return $effects;
	}

	public function get_navs_styles() {

		return array(

			array(
				'label' => __( 'Default', 'gs-books-showcase' ),
				'value' => 'nav--default',
			),
			array(
				'label' => __( 'Style 01', 'gs-books-showcase' ),
				'value' => 'nav--style-01',
			),
			array(
				'label' => __( 'Style 02', 'gs-books-showcase' ),
				'value' => 'nav--style-02',
			),
			array(
				'label' => __( 'Style 03', 'gs-books-showcase' ),
				'value' => 'nav--style-03',
			),
			array(
				'label' => __( 'Style 04', 'gs-books-showcase' ),
				'value' => 'nav--style-04',
			),
			array(
				'label' => __( 'Style 05', 'gs-books-showcase' ),
				'value' => 'nav--style-05',
			),
			array(
				'label' => __( 'Style 06', 'gs-books-showcase' ),
				'value' => 'nav--style-06',
			),
			array(
				'label' => __( 'Style 07', 'gs-books-showcase' ),
				'value' => 'nav--style-07',
			),
			array(
				'label' => __( 'Style 08', 'gs-books-showcase' ),
				'value' => 'nav--style-08',
			),
			array(
				'label' => __( 'Style 09', 'gs-books-showcase' ),
				'value' => 'nav--style-09',
			),
			array(
				'label' => __( 'Style 10', 'gs-books-showcase' ),
				'value' => 'nav--style-10',
			),
			array(
				'label' => __( 'Style 11', 'gs-books-showcase' ),
				'value' => 'nav--style-11',
			),
		);
	}

	public function get_filter_styles() {
		return array(
			array(
				'label' => __( 'Default', 'gs-books-showcase' ),
				'value' => 'filter--default',
			),
			array(
				'label' => __( 'Style 01', 'gs-books-showcase' ),
				'value' => 'filter--style-01',
			),
			array(
				'label' => __( 'Style 02', 'gs-books-showcase' ),
				'value' => 'filter--style-02',
			),
			array(
				'label' => __( 'Style 03', 'gs-books-showcase' ),
				'value' => 'filter--style-03',
			),
			array(
				'label' => __( 'Style 04', 'gs-books-showcase' ),
				'value' => 'filter--style-04',
			),
			array(
				'label' => __( 'Style 05', 'gs-books-showcase' ),
				'value' => 'filter--style-05',
			),
			array(
				'label' => __( 'Style 06', 'gs-books-showcase' ),
				'value' => 'filter--style-06',
			),
			array(
				'label' => __( 'Style 07', 'gs-books-showcase' ),
				'value' => 'filter--style-07',
			),
			array(
				'label' => __( 'Style 08', 'gs-books-showcase' ),
				'value' => 'filter--style-08',
			),
			array(
				'label' => __( 'Style 09', 'gs-books-showcase' ),
				'value' => 'filter--style-09',
			),
			array(
				'label' => __( 'Style 10', 'gs-books-showcase' ),
				'value' => 'filter--style-10',
			),
		);
	}

	public function get_dot_style() {

		return array(
			array(
				'label' => __( 'Default', 'gs-books-showcase' ),
				'value' => 'dot--default',
			),
			array(
				'label' => __( 'Style 01', 'gs-books-showcase' ),
				'value' => 'dot--style-01',
			),
			array(
				'label' => __( 'Style 02', 'gs-books-showcase' ),
				'value' => 'dot--style-02',
			),
			array(
				'label' => __( 'Style 03', 'gs-books-showcase' ),
				'value' => 'dot--style-03',
			),
			array(
				'label' => __( 'Style 04', 'gs-books-showcase' ),
				'value' => 'dot--style-04',
			),
			array(
				'label' => __( 'Style 05', 'gs-books-showcase' ),
				'value' => 'dot--style-05',
			),
			array(
				'label' => __( 'Style 06', 'gs-books-showcase' ),
				'value' => 'dot--style-06',
			),
			array(
				'label' => __( 'Style 07', 'gs-books-showcase' ),
				'value' => 'dot--style-07',
			),
			array(
				'label' => __( 'Style 08', 'gs-books-showcase' ),
				'value' => 'dot--style-08',
			),
			array(
				'label' => __( 'Style 09', 'gs-books-showcase' ),
				'value' => 'dot--style-09',
			),
			array(
				'label' => __( 'Style 10', 'gs-books-showcase' ),
				'value' => 'dot--style-10',
			),
		);
	}

	public function get_navs_pos() {

		return array(
			array(
				'label' => __( 'Bottom', 'gs-books-showcase' ),
				'value' => 'carousel-navs-pos--bottom',
			),
			array(
				'label' => __( 'Center', 'gs-books-showcase' ),
				'value' => 'carousel-navs-pos--center',
			),
			array(
				'label' => __( 'Center Outside', 'gs-books-showcase' ),
				'value' => 'carousel-navs-pos--center-outside',
			),
			array(
				'label' => __( 'Center Inside', 'gs-books-showcase' ),
				'value' => 'carousel-navs-pos--center-inside',
			),
			array(
				'label' => __( 'Top Right', 'gs-books-showcase' ),
				'value' => 'carousel-navs-pos--top-right',
			),
			array(
				'label' => __( 'Top Center', 'gs-books-showcase' ),
				'value' => 'carousel-navs-pos--top-center',
			),
			array(
				'label' => __( 'Top Left', 'gs-books-showcase' ),
				'value' => 'carousel-navs-pos--top-left',
			),
			array(
				'label' => __( 'Verticle Right', 'gs-books-showcase' ),
				'value' => 'carousel-navs-pos--verticle-right',
			),
			array(
				'label' => __( 'Verticle Left', 'gs-books-showcase' ),
				'value' => 'carousel-navs-pos--verticle-left',
			),
		);
	}

	public function get_shortcode_default_prefs() {
		return array(
			'books_slug'              		=> 'books',
			'replace_custom_slug' 			=> false,
			'archive_page_slug'             => '',
			'archive_page_title'            => '',
			'gs_book_nxt_prev'              => false,
			'gs_member_enable_multilingual' => false,
			'gs_book_search_all_fields'     => false,
			'gs_book_enable_multi_select'   => false,
			'gs_child_tax'                  => false,
			'gs_store_img'                  => true,
			'gs_book_showcase_custom_css'   => '',
			'gs_book_font_size'             => '',
			'gs_book_char_control'          => 40,
			'gs_book_label_font_size'       => 20,
			'bs_style'                      => 'bs_style_two',
			'single_page'                   => 'default',
			'style_themeing'                => 'grid',
			'at_style'                      => 'at_style_one',
			'gs_author'                     => 'gs_author_one',
			'link_target'                   => 'new_tab',
			'gsb_currency'                  => '&#36;_usd',
			'store_display'                 => 'image',
			'font_weight'                   => 'normal',
			'font_style'                    => 'normal_font',
		);
	}

	public function get_shortcode_prefs_options() {

		return array(

			'single_page'     => $this->get_single_pages(),

			'theme_and_style' => array(
				array(
					'label' => __( 'Hover(LITE)', 'gs-books-showcase' ),
					'value' => 'hover',
				),
				array(
					'label' => __( 'Grid(Pro)', 'gs-books-showcase' ),
					'value' => 'grid',
				),
				array(
					'label' => __( 'Left Sqr(Pro)', 'gs-books-showcase' ),
					'value' => 'left_sqr',
				),
				array(
					'label' => __( 'Right Sqr(Pro)', 'gs-books-showcase' ),
					'value' => 'right_sqr',
				),
				array(
					'label' => __( 'Slider(Pro)', 'gs-books-showcase' ),
					'value' => 'slider',
				),
				array(
					'label' => __( 'Slider and Hover(Pro)', 'gs-books-showcase' ),
					'value' => 'slider_hover',
				),
				array(
					'label' => __( 'Popup(Pro)', 'gs-books-showcase' ),
					'value' => 'popup',
				),
				array(
					'label' => __( 'Filter(Pro)', 'gs-books-showcase' ),
					'value' => 'filter',
				),
				array(
					'label' => __( 'Grey(Pro)', 'gs-books-showcase' ),
					'value' => 'grey',
				),
				array(
					'label' => __( 'To Single(Pro)', 'gs-books-showcase' ),
					'value' => 'to_single',
				),
			),
			'book_single'     => array(
				array(
					'label' => __( 'Style 01', 'gs-books-showcase' ),
					'value' => 'bs_style_one',
				),
				array(
					'label' => __( 'Style 02', 'gs-books-showcase' ),
					'value' => 'bs_style_two',
				),
				array(
					'label' => __( 'Style 03', 'gs-books-showcase' ),
					'value' => 'bs_style_three',
				),
			),
			'at_single'       => $this->get_author_singles(),
			'gsb_author'      => array(
				array(
					'label' => __( 'Style 01', 'gs-books-showcase' ),
					'value' => 'gs_author_one',
				),
				array(
					'label' => __( 'Style 02', 'gs-books-showcase' ),
					'value' => 'gs_author_two',
				),
				array(
					'label' => __( 'Style 03', 'gs-books-showcase' ),
					'value' => 'gs_author_three',
				),
			),
			'gsb_currency'        => array(
				array(
					'label' => __( '🇺🇸 United States - USD', 'gs-books-showcase' ),
					'value' => '&#36;_usd',
				),
				array(
					'label' => __( '🇨🇦 Canada - CAD', 'gs-books-showcase' ),
					'value' => '&#36;_canada',
				),
				array(
					'label' => __( '🇬🇧 United Kingdom - GBP', 'gs-books-showcase' ),
					'value' => '&#163;',
				),
				array(
					'label' => __( '🇪🇺 Eurozone - EUR', 'gs-books-showcase' ),
					'value' => '&#8364;',
				),
				array(
					'label' => __( '🇦🇺 Australia - AUD', 'gs-books-showcase' ),
					'value' => '&#36;_aus',
				),
				array(
					'label' => __( '🇮🇳 India - INR', 'gs-books-showcase' ),
					'value' => '&#8377;',
				),
				array(
					'label' => __( '🇧🇩 Bangladesh - BDT', 'gs-books-showcase' ),
					'value' => '&#2547;',
				),
				array(
					'label' => __( '🇯🇵 Japan - JPY', 'gs-books-showcase' ),
					'value' => '&#165;_japan',
				),
				array(
					'label' => __( '🇨🇳 China - CNY', 'gs-books-showcase' ),
					'value' => '&#165;_china',
				),
				array(
					'label' => __( '🇧🇷 Brazil - BRL', 'gs-books-showcase' ),
					'value' => '&#82;&#36;',
				),
				array(
					'label' => __( '🇿🇦 South Africa - ZAR', 'gs-books-showcase' ),
					'value' => '&#82;',
				),
				array(
					'label' => __( '🇸🇬 Singapore - SGD', 'gs-books-showcase' ),
					'value' => '&#36;_sin', // Dollar
				),
				array(
					'label' => __( '🇳🇿 New Zealand - NZD', 'gs-books-showcase' ),
					'value' => '&#36;_new', // Dollar
				),
				array(
					'label' => __( '🇷🇺 Russia - RUB', 'gs-books-showcase' ),
					'value' => '&#8381;', // Russian Ruble
				),
				array(
					'label' => __( '🇰🇷 South Korea - KRW', 'gs-books-showcase' ),
					'value' => '&#8361;', // South Korean Won
				),
				array(
					'label' => __( '🇲🇽 Mexico - MXN', 'gs-books-showcase' ),
					'value' => '&#36;_mex', // Peso
				),
				array(
					'label' => __( '🇵🇰 Pakistan - PKR', 'gs-books-showcase' ),
					'value' => '&#8360;', // Pakistani Rupee
				),
				array(
					'label' => __( '🇹🇭 Thailand - THB', 'gs-books-showcase' ),
					'value' => '&#3647;', // Thai Baht
				),
				array(
					'label' => __( '🇲🇾 Malaysia - MYR', 'gs-books-showcase' ),
					'value' => '&#82;&#77;', // Malaysian Ringgit
				),
				array(
					'label' => __( '🇵🇭 Philippines - PHP', 'gs-books-showcase' ),
					'value' => '&#8369;', // Philippine Peso
				),
				array(
					'label' => __( '🇻🇳 Vietnam - VND', 'gs-books-showcase' ),
					'value' => '&#8363;', // Vietnamese Dong
				),
				array(
					'label' => __( '🇸🇦 Saudi Arabia - SAR', 'gs-books-showcase' ),
					'value' => '&#65020;', // Saudi Riyal
				),
				array(
					'label' => __( '🇦🇪 United Arab Emirates - AED', 'gs-books-showcase' ),
					'value' => '&#1583;&#46;&#1573;', // UAE Dirham
				),
				array(
					'label' => __( '🇹🇷 Turkey - TRY', 'gs-books-showcase' ),
					'value' => '&#8378;', // Turkish Lira
				),
				array(
					'label' => __( '🇮🇩 Indonesia - IDR', 'gs-books-showcase' ),
					'value' => '&#82;&#112;', // Indonesian Rupiah
				),
				array(
					'label' => __( '🇳🇬 Nigeria - NGN', 'gs-books-showcase' ),
					'value' => '&#8358;', // Nigerian Naira
				),
			),
			'store_display'             => $this->store_display(),
			'gs_link_target'  => array(
				array(
					'label' => __( 'New Tab', 'gs-books-showcase' ),
					'value' => 'new_tab',
				),
				array(
					'label' => __( 'Same Window', 'gs-books-showcase' ),
					'value' => 'same_window',
				),
			),
			'gs_font_weight'  => array(
				array(
					'label' => __( 'Normal', 'gs-books-showcase' ),
					'value' => 'normal',
				),
				array(
					'label' => __( 'Bold', 'gs-books-showcase' ),
					'value' => 'bold',
				),
				array(
					'label' => __( 'Lighter', 'gs-books-showcase' ),
					'value' => 'lighter',
				),
			),
			'gs_font_style'   => array(
				array(
					'label' => __( 'Normal', 'gs-books-showcase' ),
					'value' => 'normal_font',
				),
				array(
					'label' => __( 'Bold', 'gs-books-showcase' ),
					'value' => 'bold_font',
				),
				array(
					'label' => __( 'Lighter', 'gs-books-showcase' ),
					'value' => 'lighter_font',
				),
			),

		);
	}

	public function store_display() {

			$free_stores = [
				array(
					'label' => __( 'Button', 'gs-books-showcase' ),
					'value' => 'btn',
				)
			];

			$pro_stores = [
				array(
					'label' => __( 'Image', 'gs-books-showcase' ),
					'value' => 'image',
				),
				array(
					'label' => __( 'Both', 'gs-books-showcase' ),
					'value' => 'both',
				)
			];

			if ( ! is_pro_active() || ! is_pro_valid() ) {
				$pro_stores = array_map( function( $item ) {
					$item['pro'] = true;
					return $item;
				}, $pro_stores );
			}
	
			return array_merge( $free_stores, $pro_stores );
	} 

	public function get_single_pages() {

		$free_singles = [
			array(
				'label' => __( 'Default', 'gs-books-showcase' ),
				'value' => 'default',
			)
		];

		$pro_singles = [
			array(
				'label' => __( 'Style 01', 'gs-books-showcase' ),
				'value' => 'style_one',
			),
			array(
				'label' => __( 'Style 02', 'gs-books-showcase' ),
				'value' => 'style_two',
			),
			array(
				'label' => __( 'Style 03', 'gs-books-showcase' ),
				'value' => 'style_three',
			),
			array(
				'label' => __( 'Style 04', 'gs-books-showcase' ),
				'value' => 'style_four',
			)
		];

		if ( ! is_pro_active() || ! is_pro_valid() ) {
			$pro_singles = array_map( function( $item ) {
				$item['pro'] = true;
				return $item;
			}, $pro_singles );
		}

		return array_merge( $free_singles, $pro_singles );
	} 

	public function get_author_singles() {

		$free_authors = [];

		$pro_authors  = [
			array(
				'label' => __( 'Style 01', 'gs-books-showcase' ),
				'value' => 'at_style_one',
			),
			array(
				'label' => __( 'Style 02', 'gs-books-showcase' ),
				'value' => 'at_style_two',
			),
			array(
				'label' => __( 'Style 03', 'gs-books-showcase' ),
				'value' => 'at_style_three',
			)
		];

		if ( ! is_pro_active() || ! is_pro_valid() ) {
			$pro_authors = array_map( function( $item ) {
				$item['pro'] = true;
				return $item;
			}, $pro_authors );
		}
		
		return array_merge( $free_authors, $pro_authors );
	}


	public function validate_shortcode_prefs(array $settings)
	{

		foreach ($settings as $setting_key => $setting_val) {
			if ($setting_key == 'gs_book_showcase_custom_css') {
				$settings[$setting_key] = wp_strip_all_tags($setting_val);
			} else {
				$settings[$setting_key] = sanitize_text_field($setting_val);
			}
		}
		return $settings;
	}


	public function _save_shortcode_pref( $settings, $is_ajax ) {

		if ( empty($settings) ) $settings = [];
		
        $settings = $this->validate_shortcode_prefs( $settings );

		// Maybe add validation?
		update_option( $this->option_name, $settings, 'yes' );

		// Clean permalink flush
		delete_option( 'GS_book_permalinks_flushed' );

		do_action( 'gs_books_preference_update' );

		do_action( 'gsp_preference_update' );

		if ( $is_ajax ) {
			wp_send_json_success( __( 'Preference saved', 'gs-books-showcase' ) );
		}
	}

	public function save_shortcode_pref() {

		check_ajax_referer( '_gs_books_admin_nonce_gs_' );
		$prefs = isset( $_POST['prefs'] )?  wp_unslash( $_POST['prefs'] ) : '';

		if ( empty($prefs ) ) {
			wp_send_json_error( __( 'No preference provided', 'gs-books-showcase' ), 400 );
		}

		$this->_save_shortcode_pref( $prefs, true );
	}

	public function _get_shortcode_pref( $is_ajax ) {

		$pref = get_option( $this->option_name );

		if ( empty( $pref ) ) {
			$pref = $this->get_shortcode_default_prefs();
			$this->_save_shortcode_pref( $pref, false );
		}

		$pref = shortcode_atts( $this->get_shortcode_default_prefs(), $pref );

		if ( $is_ajax ) {
			wp_send_json_success( $pref );
		}

		return $pref;
	}

	public function get_shortcode_pref() {

		$this->_get_shortcode_pref( wp_doing_ajax() );
	}

	public function _get_localization( $is_ajax ) {

		$localization = (array) get_option( $this->level_option_name, [] );

		$localization = shortcode_atts( $this->get_shortcode_default_localization(), $localization );

		if ( $is_ajax ) wp_send_json_success( $localization );

		return $localization;
	}

	public function _save_localization( $settings, $is_ajax ) {

		// Sanitize
		$settings = stripslashes_deep( $settings );
		
		// Merge with default
		$settings = shortcode_atts( $this->get_shortcode_default_localization(), $settings );
		
		// Maybe add validation?
		update_option( $this->level_option_name, $settings );

		// Clean permalink flush
		delete_option( 'GS_Books_plugin_permalinks_flushed' );

		if ( $is_ajax ) {
			wp_send_json_success( __( 'Localization saved', 'gs-books-showcase' ) );
		}
	}

	public function save_localization(  ) {

		if(!check_ajax_referer( '_gs_books_admin_nonce_gs_' )){
			wp_send_json_error( __( 'Unauthorized Request!', 'gs-books-showcase' ), 400 );
		}

		$localization = isset( $_POST['localization'] )?  wp_unslash( $_POST['localization'] ) : '';

		if ( empty( $localization ) ) {
			wp_send_json_error( __( 'No localization provided', 'gs-books-showcase' ), 400 );
		}

		$this->_save_localization(  $localization, true );
	}

	static function maybe_create_shortcodes_table() {

		global $wpdb;

		$gs_books_db_version = '1.0';

		if ( get_option( "{$wpdb->prefix}gs_books_db_version" ) == $gs_books_db_version ) {
			return; // vail early
		}

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		$sql = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}gs_books (
            id BIGINT(20) unsigned NOT NULL AUTO_INCREMENT,
            shortcode_name TEXT NOT NULL,
            shortcode_settings LONGTEXT NOT NULL,
            created_at DATETIME NOT NULL DEFAULT '0000-00-00 00:00:00',
            updated_at DATETIME NOT NULL DEFAULT '0000-00-00 00:00:00',
            PRIMARY KEY (id)
        )" . $wpdb->get_charset_collate() . ';';

		if ( get_option( "{$wpdb->prefix}gs_books_db_version" ) < $gs_books_db_version ) {
			dbDelta( $sql );
		}

		update_option( "{$wpdb->prefix}gs_books_db_version", $gs_books_db_version );
	}

	public function create_dummy_shortcodes() {

		$request = wp_remote_get( GS_BOOKS_PLUGIN_URI . '/includes/demo-data/gs-books-dummy-shortcodes.json', array( 'sslverify' => false ) );

		if ( is_wp_error( $request ) ) {
			return false;
		}

		$shortcodes = wp_remote_retrieve_body( $request );
		$shortcodes = json_decode( $shortcodes, true );

		$wpdb = $this->gs_books_get_wpdb();

		if ( ! $shortcodes || ! count( $shortcodes ) ) {
			return;
		}

		foreach ( $shortcodes as $shortcode ) {

			$shortcode['shortcode_settings']                      = json_decode( $shortcode['shortcode_settings'], true );
			$shortcode['shortcode_settings']['gsbooks-demo_data'] = true;

			$data = array(
				'shortcode_name'     => $shortcode['shortcode_name'],
				'shortcode_settings' => json_encode( $shortcode['shortcode_settings'] ),
				'created_at'         => current_time( 'mysql' ),
				'updated_at'         => current_time( 'mysql' ),
			);
			 // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
			$wpdb->insert( "{$wpdb->prefix}gs_books", $data, $this->get_gs_books_shortcode_db_columns() );

		}
	}

	public function delete_dummy_shortcodes() {

		$wpdb   = $this->gs_books_get_wpdb();
		$needle = 'gsbooks-demo_data';
		$wpdb->query( "DELETE FROM {$wpdb->prefix}gs_books WHERE shortcode_settings like '%$needle%'" );
	}
}
