<?php

namespace GS_BOOKS;

defined( 'ABSPATH' ) || exit;

/**
 * Responsible for containing helper methods.
 *
 * @since 1.2.11
 */
class Helpers {

	static function get_option( $option, $default = '' ) {
		$options = get_option( 'gs_bookshowcase_settings' );
		if ( isset( $options[ $option ] ) ) {
			return $options[ $option ];
		}
		return $default;
	}

	static function get_label_option( $option, $default = '' ) {
		$options = get_option( 'gs_bookshowcase_level_settings' );
		if ( isset( $options[ $option ] ) ) {
			return $options[ $option ];
		}
		return $default;
	}

	static function get_translation( $translation_name ) {

		$translations = plugin()->builder->get_shortcode_default_localization();

		if ( ! array_key_exists( $translation_name, $translations ) ) {
			return '';
		}

		$localizations = plugin()->builder->_get_localization( false );

		if ( wp_validate_boolean( $localizations['gs_member_enable_multilingual'] ) ) {
			return $translations[ $translation_name ];
		}

		return $localizations[ $translation_name ];
	}

	static function isPreview() {
		return isset( $_REQUEST['gs_books_shortcode_preview'] ) && ! empty( $_REQUEST['gs_books_shortcode_preview'] );
	}

	static function gs_books_getoption( $option, $default = '' ) {
		$options = get_option( 'gs_books_slider_shortcode_prefs' );

		if ( isset( $options[ $option ] ) ) {
			return $options[ $option ];
		}

		return $default;
	}

	static function isFromShortcodeBuilder( $atts ) {
		return ! empty( $atts['id'] );
	}

	static function wp_kses( $content ) {

		$allowed_tags = wp_kses_allowed_html( 'post' );

		$input_common_atts = array(
			'class'       => true,
			'id'          => true,
			'style'       => true,
			'novalidate'  => true,
			'name'        => true,
			'width'       => true,
			'height'      => true,
			'data'        => true,
			'title'       => true,
			'placeholder' => true,
			'value'       => true,
		);

		$allowed_tags = array_merge_recursive(
			$allowed_tags,
			array(
				'select' => $input_common_atts,
				'input'  => array_merge(
					$input_common_atts,
					array(
						'type'    => true,
						'checked' => true,
					)
				),
				'option' => array(
					'class'    => true,
					'id'       => true,
					'selected' => true,
					'data'     => true,
					'value'    => true,
				),
			)
		);

		echo wp_kses( stripslashes_deep( $content ), $allowed_tags );
	}

	static function echo_return( $content, $echo = false ) {
		if ( $echo ) {
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		    self::wp_kses( $content );
		} else {
			return $content;
		}
	}

	static function showUpgradeToProMessage() {
		printf(
			'<h4 style="text-align: center;">%s <a href="%s" target="_blank">%s</a> %s<br><a href="%s" target="_blank">%s</a></h4>',
			esc_html_e( 'Select correct Theme or Upgrade to', 'gs-books-showcase' ),
			esc_url( 'https://www.gsplugins.com/product/wordpress-books-showcase-plugin' ),
			esc_html_e( 'Pro version', 'gs-books-showcase' ),
			esc_html_e( 'for more Options', 'gs-books-showcase' ),
			esc_url( 'http://bookshowcase.gsplugins.com' ),
			esc_html_e( 'Chcek available demos', 'gs-books-showcase' )
		);
	}

	static function checkUserCanEditPost( $postId ) {
		return current_user_can( 'edit_post', $postId );
	}

	static function gs_book_getoption( $option, $default = '' ) {

		$options = get_option( 'gs_bookshowcase_settings' );

		if ( isset( $options[ $option ] ) ) {
			return $options[ $option ];
		}

		return $default;
	}

	static function get_post_meta( $meta_key, $post_id = null, $single_mode = true ) {
		if ( empty( $post_id ) || ! is_numeric( $post_id ) ) {
			$post_id = get_the_ID();
		}
		if ( empty( $post_id ) ) return;
		return get_post_meta( $post_id, $meta_key, $single_mode );
	}

	// has post meta
	static function has_post_meta( $meta_key, $post_id = null ) {
		if ( empty( $post_id ) || ! is_numeric( $post_id ) ) {
			$post_id = get_the_ID();
		}
		if ( empty( $post_id ) ) return false;
		return metadata_exists( 'post', $post_id, $meta_key );
	}

	// has author taxonomy or author meta
	static function has_author_info( $post_id = null ) {
		if ( empty( $post_id ) || ! is_numeric( $post_id ) ) {
			$post_id = get_the_ID();
		}
		if ( empty( $post_id ) ) return false;

		// Checked for author taxonomy
		$author_term  = wp_get_post_terms( get_the_id(), 'gsb_author' );
		if ( ! is_wp_error( $author_term ) && ! empty( $author_term ) ) return true;

		// Checked for author meta
		return self::has_post_meta( '_gsbks_author_image_id', $post_id );
	}

	// Get related books query
	static function get_related_books_query( $post_id = null ) {
		
		if ( empty( $post_id ) || ! is_numeric( $post_id ) ) {
			$post_id = get_the_ID();
		}
		if ( empty( $post_id ) ) return false;

		$terms    = get_the_terms( $post_id, 'bookshowcase_group' );
		$term_ids = wp_list_pluck( $terms, 'term_id' );
		
		$args = array(
			'posts_per_page' => -1,
			'post_type'      => 'gs_bookshowcase',
			// phpcs:ignore WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in
			'post__not_in'   => array( $post_id ),
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
			'tax_query'      => array(
				array(
					'taxonomy' => 'bookshowcase_group',
					'field'    => 'term_id',
					'terms'    => $term_ids,
				),
			),
		);
		
		return new \WP_Query( $args );
	}

	static function gs_get_thumbnail_id( $post_id ) {

		$thumb_id = get_post_meta( $post_id, '_thumbnail_id', true );

		if ( empty( $thumb_id ) ) {
			global $wpdb;
			// Fallback direct DB query
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
			$thumb_id = $wpdb->get_var( $wpdb->prepare(
				"SELECT meta_value FROM $wpdb->postmeta WHERE post_id = %d AND meta_key = '_thumbnail_id' LIMIT 1",
				$post_id
			) );
		}

		return $thumb_id;
	}

	static function gs_render_thumbnail( $post_id, $size = 'full', $extra_classes = array(), $echo = false ) {
		$thumbnail_id = self::gs_get_thumbnail_id( $post_id );

		// Build image or fallback
		if ( $thumbnail_id ) {
			$thumbnail = wp_get_attachment_image(
				$thumbnail_id,
				$size,
				false,
				array_merge(
					array(
						'class'    => implode( ' ', $extra_classes ),
						'alt'      => get_the_title( $post_id ),
						'itemprop' => 'image',
					),
					array_filter( apply_filters( 'gs_book_thumbnail_attrs', array(), $post_id ) )
				)
			);
		} else {
			$thumbnail = sprintf(
				'<img src="%s" alt="%s" itemprop="image"/>',
				esc_url( GS_BOOKS_PLUGIN_URI . '/assets/img/no_img.png' ),
				esc_attr( get_the_title( $post_id ) )
			);
		}

		$thumbnail = apply_filters( 'gs_book_thumbnail_html', $thumbnail, $post_id );

		return self::echo_return( $thumbnail, $echo );
	}

	static function gs_book_thumbnail( $echo = false, $size = 'full' ) {
		$book_id = get_the_ID();
		return self::gs_render_thumbnail( $book_id, $size, array(), $echo );
	}

	static function post_thumbnail( $echo = false, $size = 'large' ) {
		$book_id = get_the_ID();

		$disable_lazy_load = self::get_option( 'disable_lazy_load', 'off' );
		$lazy_load_class   = self::get_option( 'lazy_load_class', 'skip-lazy' );

		$classes = array( 'gs_book--image' );

		if ( $disable_lazy_load && ! empty( $lazy_load_class ) ) {
			$classes[] = $lazy_load_class;
		}

		$classes = (array) apply_filters( 'gs_book_thumbnail_classes', $classes );

		$thumbnail = self::gs_render_thumbnail( $book_id, $size, $classes, false );

		// Wrap with div
		$thumbnail = '<div class="gsbooks--book-thumb">' . $thumbnail . '</div>';

		return self::echo_return( $thumbnail, $echo );
	}

	static function gs_book_thumbnail_with_link( $size = 'large', $link_type = 'single_page', $extra_link_class = '', $popup_style = 'style_one' ) {

		$image_html = self::post_thumbnail( false, $size );

		$before = $after = '';

		if ( $link_type == 'single_page' ) {

			$before = sprintf( '<a class="%s" href="%s">', esc_attr( $extra_link_class ), get_the_permalink() );

		} elseif ( $link_type == 'popup' ) {

			$before = sprintf( '<a class="gs_book_pop open-popup-link %s" data-mfp-src="#gs_book_popup_%s" data-theme="%s" href="#">', esc_attr( $extra_link_class ), get_the_ID(), 'gs-book-popup--' . esc_attr( $popup_style ) );

		}

		$after = '</a>';
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		echo $before . $image_html . $after;
	}

	static function post_title( $link_type = 'single_page', $popup_style = 'style_one' ) {

		$title = get_the_title();

		if ( $link_type == 'single_page' ) {

			$title = sprintf( '<a href="%s">%s</a>', get_the_permalink(), esc_html( $title ) );

		} elseif ( $link_type == 'popup' ) {

			$title = sprintf( '<a class="gs_book_pop open-popup-link" data-mfp-src="#gs_book_popup_%s" href="#" data-theme="%s">%s</a>', get_the_ID(), 'gs-book-popup--' . esc_attr( $popup_style ), esc_html( $title ) );

		}

		echo wp_kses_post( $title );
	}

	static function post_content( $max_length = 100, $is_excerpt = true, $echo = true, $link_type = 'single_page', $popup_style = 'style_one' ) {		

		$description = $is_excerpt ? get_the_excerpt() : get_the_content();

		$more_text = self::get_translation( 'gsb_more_text_modify' );

		$more_link = '';

		if ( $link_type == 'single_page' ) {

			$more_link = sprintf( '...<a href="%s">%s</a>', get_the_permalink(), $more_text );

		} elseif ( $link_type == 'popup' ) {

			$more_link = sprintf( '...<a class="gs_book_pop open-popup-link" data-mfp-src="#gs_book_popup_%s" href="#" data-theme="%s">%s</a>', get_the_ID(), 'gs-book-popup--' . esc_attr( $popup_style ), esc_html( $more_text ) );

		}

		// Reduce the description length
		if ( $max_length > 0 && strlen( $description ) > $max_length ) {
			$description = substr( $description, 0, $max_length ) . $more_link;
		}

		return self::echo_return( $description, $echo );
	}

	static function post_authors( $is_link = false, $separator = ', ' ) {

		$term_mode = true;
		$authors   = get_the_terms( get_the_ID(), 'gsb_author' );

		if ( is_wp_error( $authors ) || empty( $authors ) ) {
			return '';
		}

		if ( ! $term_mode ) {
			return $authors;
		}

		if ( ! $is_link ) {
			$authors = array_map(
				function ( $author ) {
					return sprintf( '<span>%s</span>', $author->name );
				},
				$authors
			);
		} else {
			$authors = array_map(
				function ( $author ) {
					return sprintf( '<a href="%s">%s</a>', get_term_link( $author, 'gsb_author' ), $author->name );
				},
				$authors
			);
		}

		return implode( $separator, $authors );
	}

	static function book_publisher( $echo = false ) {
		$publishers = get_the_term_list( get_the_ID(), 'gsb_publishers', '<span>', ', ', '</span>' );
		return self::echo_return( $publishers, $echo );
	}

	static function get_book_groups( $post_id, $term_name = 'bookshowcase_group' ) {

		$groups = get_the_terms( $post_id, $term_name );

		if ( is_wp_error( $groups ) || empty( $groups ) ) {
			return;
		}

		$groups = array_map(
			function ( $group ) {
				return $group->slug;
			},
			$groups
		);

		return implode( ' ', $groups );
	}

	static function get_store_links( $args = [] ) {

		$args = shortcode_atts(
			array(
				'show_url_field' => true,
				'link_target'    => '_blank',
				'link_rel'       => 'noreferrer noopener',
				'separator'      => ', ',
				'echo'           => false,
			),
			$args
		);

		$_urls = array();

		// if ( $args['show_url_field'] ) {
		// 	$url = self::get_post_meta( '_gsbks_url' );
		// 	if ( ! empty( $url ) ) {
		// 		$_urls[] = array(
		// 			'name' => 'Buy',
		// 			'url'  => $url,
		// 		);
		// 	}
		// }

		$store = self::get_post_meta( 'gs_repeatable_fields' );

		if ( ! empty( $store ) ) {
			$_urls = array_merge( $_urls, $store );
		}

		if ( empty( $_urls ) ) {
			return;
		}

		$urls = array_map(
			function ( $_url ) use ( $args ) {
				return sprintf(
					'<span class="%s"><a href="%s" target="%s" rel="%s">%s</a></span>',
					'gs-book--store-' . sanitize_title( $_url['name'] ),					
					esc_url( $_url['url'] ),
					esc_attr( $args['link_target'] ),
					esc_attr( $args['link_rel'] ),
					esc_html( $_url['name'] )
				);
			},
			$_urls
		);

		if ( empty( $args['separator'] ) ) {
			$separator = '';
		} else {
			$separator = sprintf( '<span class="gs-book--sitem_sep">%s</span>', esc_html( $args['separator'] ) );
		}

		$links = sprintf( '<span class="gs-book--store_urls">%s</span>', implode( $separator, $urls ) );

		return self::echo_return( $links, $args['echo'] );
	}

	static function get_col_classes( $desktop = '3', $tablet = '4', $mobile_portrait = '6', $mobile = '12' ) {
		return sprintf( 'gs-col-lg-%s gs-col-md-%s gs-col-sm-%s gs-col-xs-%s', $desktop, $tablet, $mobile_portrait, $mobile );
	}

	static function get_all_termss( $post_id ) {

		$post_type  = get_post_type( $post_id );
		$taxonomies = get_object_taxonomies( $post_type );
		$all_tax    = [];

		if ( ! empty( $taxonomies ) ) {
			
			foreach ( $taxonomies as $taxonomy ) {
				$terms 		= get_the_terms( $post_id, $taxonomy );
				$terms_slug = wp_list_pluck( $terms, 'name' );
				array_push( $all_tax, implode( ' ', $terms_slug ) );
			}
		}

		return implode( ' ', $all_tax );
	}

	static function get_all_terms($post_id) {
		$post_type  = get_post_type($post_id);
		$taxonomies = get_object_taxonomies($post_type);
		$all        = [];

		foreach ((array) $taxonomies as $taxonomy) {
			$terms = get_the_terms($post_id, $taxonomy);
			if (empty($terms) || is_wp_error($terms)) {
				continue;
			}

			foreach ($terms as $term) {
				$name = $term->name;

				// Fallback: if no name, use slug
				if ($name === '' && !empty($term->slug)) {
					// use the slug, but normalize first
					$name = urldecode($term->slug);
				}

				$name = gsb_text_to_class($name); // transliterate & slugify

				if ($name !== '') {
					$all[] = $name;
				}
			}
		}

		// De-duplicate
		$all = array_values(array_unique($all));

		// Return as one string with space between
		return implode(' ', $all);
	}

	/* * Check if a field is visible based on the visibility settings.
	 *
	 * @param string $field The field to check visibility for.
	 * @param string $device The device type (desktop, tablet, mobile_landscape, mobile).
	 * @return bool True if the field is visible, false otherwise.
	 *
	 * Example fields:
	 * - book_name
	 * - book_details
	 * - book_store
	 * - book_author
	 * - book_publish_date
	 * - book_publisher
	 * - book_author_name
	 * - book_author_img
	 * - book_rating
	 * - book_availability
	 * - book_preorder_available
	 * - book_accessibility_features
	 */
	static function is_visible( $field, $device = '' ) {

		if ( empty( $device ) ) {
			return $field['desktop'] || $field['tablet'] || $field['mobile_landscape'] || $field['mobile'];
		}

		if ( in_array( $device, ['desktop', 'tablet', 'mobile_landscape', 'mobile'] ) ) {
			return isset( $field[ $device ] ) ? $field[ $device ] : false;
		}

		return false;
	}

	static function get_visible_classes( $field, $additional_class = '' ) {

		$devices = [
			'desktop'          => 'gs-books--hide-md',
			'tablet'           => 'gs-books--hide-sm',
			'mobile_landscape' => 'gs-books--hide-xs',
			'mobile'           => 'gs-books--hide-xxs',
		];

		$classes = [];

		if ( !empty( $additional_class ) ) {
			$classes[] = $additional_class;
		}

		foreach ( $devices as $device => $class ) {
			if ( ! self::is_visible( $field, $device ) ) {
				$classes[] = $class;
			}
		}

		return $classes;
	}

	static function print_visible_classes( $field, $additional_class = '' ) {
		$classes = self::get_visible_classes( $field, $additional_class );
		echo esc_attr( implode( ' ', $classes ) );
	}
	
}