<?php

namespace GS_BOOKS;

use function GS_BOOKS_PRO\is_pro_valid;

defined( 'ABSPATH' ) || exit;

function is_pro_active() {
    require_once ABSPATH . 'wp-admin/includes/plugin.php';
    return defined( "GS_BOOKS_PRO_VERSION" ) && is_plugin_active( GS_BOOKS_PRO_PLUGIN );
}

function gs_appsero_init() {

	if ( ! class_exists( 'GSBookAppSero\Client' ) ) {
		require_once GS_BOOKS_PLUGIN_DIR . '/includes/appsero/Client.php';
	}

	$client = new \GSBookAppSero\Client(
		'f8265887-01c2-4841-9716-d45eed199345',
		__( 'GS Books Showcase', 'gs-books-showcase' ),
		GS_BOOKS_PLUGIN_FILE
	);

	// Active insights
	$client->insights()->init();
	
} 

function gsbooks_get_shortcode_params( $settings ) {

    $params = [];

    foreach( $settings as $key => $val ) {
        if ( gettype($val) == 'array' ) {
            $val = implode(',', $val);
        }
        $params[] = $key.'="'.$val.'"';
    }

    return implode( ' ', $params );

}

function get_settings( $atts = array() ) {
    return shortcode_atts( apply_filters( 'gs-books-showcase_shortcode_attributes', plugin()->builder->get_shortcode_default_settings() ), $atts );
}

function get_shortcodes() {
	return plugin()->builder->_get_shortcodes( null, false, true );
}

function get_carousel_settings($settings) {

	$getDatas                           = array();
	$getDatas['speed']                  = intval( $settings['speed'] );
	$getDatas['isAutoplay']             = wp_validate_boolean( $settings['isAutoplay'] );
	$getDatas['autoplay_delay']         = intval( $settings['autoplay_delay'] );
	$getDatas['loop']        		    = wp_validate_boolean( $settings['infinite_loop']  );
	$getDatas['pause_on_hover']         = wp_validate_boolean( $settings['pause_on_hover'] );
	$getDatas['navs']                   = wp_validate_boolean( $settings['gsb_slider_navs'] );
	$getDatas['dots']                   = wp_validate_boolean( $settings['gsb_slider_dots'] );
	$getDatas['desktop_columns']        = intval( $settings['columns'] );
	$getDatas['tablet_columns']         = intval( $settings['columns_tablet'] );
	$getDatas['mobile_columns']         = intval( $settings['columns_mobile_portrait'] );
	$getDatas['columns_small_mobile']   = intval( $settings['columns_mobile'] );
	$getDatas['carousel_navs_style']    = sanitize_key( $settings['gsb_navs_style'] );
	$getDatas['carousel_dots_style']    = sanitize_key( $settings['gsb_dots_style'] );
	$getDatas['gsb_navs_pos']           = sanitize_key( $settings['gsb_navs_pos'] );
	$getDatas['reverseDirection'] 		= wp_validate_boolean( $settings['reverse_direction'] );
	$getDatas['slidesPerGroup'] 		= intval( $settings['slides_per_group'] );

	return $getDatas;
}

function get_ticker_settings($settings) {
	return [
		'mode'                 => 'horizontal',
		'speed'                => intval($settings['ticker_speed']),
		'pauseOnHover'         => wp_validate_boolean($settings['pause_on_hover']),
		'slideSpace'           => 0,
		'desktopLogos'         => columns_to_number( $settings['columns'] ),
		'tabletLogos'          => columns_to_number( $settings['columns_tablet'] ),
		'largeMobileLogos'     => columns_to_number( $settings['columns_mobile_portrait'] ),
		'mobileLogos'          => columns_to_number( $settings['columns_mobile'] ),
		'reverseDirection'     => wp_validate_boolean($settings['reverse_direction']),
	];
}

function columns_to_number( $columns ) {
	$columns = str_replace('_', '.', $columns);
	return 12 / floatval( $columns );
}

function is_divi_active() {
	if ( ! defined( 'ET_BUILDER_PLUGIN_ACTIVE' ) || ! ET_BUILDER_PLUGIN_ACTIVE ) {
		return false;
	}
	return et_core_is_builder_used_on_current_request();
}

function is_divi_editor() {
	if ( empty( $_POST['action'] ) || empty( $_POST['module_type'] ) ) {
		return false;
	}

	$action      = sanitize_key( wp_unslash( $_POST['action'] ) );
	$module_type = sanitize_key( wp_unslash( $_POST['module_type'] ) );

	if ( 'et_pb_process_computed_property' === $action && 'gs_testimonial_slider' === $module_type ) {
		return true;
	}
	return false;
}

function get_all_groups( array $exclude_group, $settings ) {

	$args = array(
		'taxonomy'   => 'bookshowcase_group',
		'orderby'    => $settings['group_order_by'] ?? 'name',
		'order'      => $settings['group_order'] ?? 'DESC',
		'hide_empty' => true,
	);

	if ( is_pro_active() && is_pro_valid() ) {

		$fn = function ( $orderby, $args ) {
			if ( isset( $args['orderby'] ) && $args['orderby'] === 'term_order' ) {
				return 'tt.order';
			}
			return $orderby;
		};
		
		add_filter( 'get_terms_orderby', $fn, 10, 2 );
		
		$groups = get_terms($args);
		
		remove_filter( 'get_terms_orderby', $fn );

	} else {

		$groups = get_terms($args);

	}

	$all_groups = wp_list_pluck( $groups, 'slug' );
	$all_groups = array_diff( $all_groups, $exclude_group ); 
	return $all_groups;
}

function get_all_tags( $term_name, array $exclude_group ) {
	
	$tags = get_terms(
		array(
			'taxonomy'   => $term_name,
			'orderby'    => 'name',
			'order'      => 'ASC',
			'hide_empty' => true,
		)
	);	

	$all_tags = wp_list_pluck( $tags, 'slug' );
	$all_tags = array_diff( $all_tags, $exclude_group ); 

	return $all_tags;
}

function add_tax_query( &$queryArgs, $taxonomy, $terms, $operator = 'IN' ) {

	// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
	if ( ! isset($queryArgs['tax_query']) ) $queryArgs['tax_query'] = [];
	
    if (!empty($terms)) {
        $queryArgs['tax_query'][] = array(
            'taxonomy' => $taxonomy,
            'field'    => 'slug',
            'terms'    => $terms,
            'operator' => $operator,
        );
    }
}

function minimize_css_simple( $css ) {
    // https://datayze.com/howto/minify-css-with-php
    $css = preg_replace('/\/\*((?!\*\/).)*\*\//', '', $css); // negative look ahead
    $css = preg_replace('/\s{2,}/', ' ', $css);
    $css = preg_replace('/\s*([:;{}])\s*/', '$1', $css);
    $css = preg_replace('/;}/', '}', $css);
    return $css;
}

function gs_wp_kses( $content ) {

	$allowed_tags = wp_kses_allowed_html( 'post' );

	$input_common_atts = array(
		'class'       => true,
		'id'          => true,
		'style'       => true,
		'novalidate'  => true,
		'name'        => true,
		'width'       => true,
		'height'      => true,
		'data'        => true,
		'title'       => true,
		'placeholder' => true,
		'value'       => true,
	);

	$allowed_tags = array_merge_recursive(
		$allowed_tags,
		array(
			'select' => $input_common_atts,
			'input'  => array_merge(
				$input_common_atts,
				array(
					'type'    => true,
					'checked' => true,
				)
			),
			'option' => array(
				'class'    => true,
				'id'       => true,
				'selected' => true,
				'data'     => true,
				'value'    => true,
			),
		)
	);

	return wp_kses( stripslashes_deep( $content ), $allowed_tags );
}

function select_builder( $name, $options, $selected = '', $selecttext = '', $class = '', $optionvalue = 'value' ) {

	if ( is_array( $options ) ) {

		$select_html = sprintf( '<select name="%1$s" class="%2$s">', esc_attr( $name ), esc_attr( $class ) );

		if ( $selecttext ) {
			$select_html .= sprintf( '<option value="">%s</option>', esc_html( $selecttext ) );
		}

		foreach ( $options as $key => $option ) {
			$value        = $optionvalue == 'value' ? $option : $key;
			$is_selected  = $value == $selected ? 'selected="selected"' : '';
			$select_html .= sprintf( '<option value="%s" %s>%s</option>', esc_attr( $value ), $is_selected, esc_html( $option ) );
		}

		$select_html .= '</select>';
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		echo gs_wp_kses( $select_html );
	}
}

function getoption( $option, $default = '' ) {
    $prefs = plugin()->builder->_get_shortcode_pref( false );
    return isset($prefs[$option]) ? $prefs[$option] : $default;
}

/**
 * Plugins action links
 */
function add_pro_link( $links ) {
    if ( ! is_pro_active() ) {
        $links[] = '<a style="color: red; font-weight: bold;" class="gs-pro-link" href="https://www.gsplugins.com/product/gs-books-showcase/" target="_blank">Go Pro!</a>';
    }
    $links[] = '<a href="https://www.gsplugins.com/wordpress-plugins" target="_blank">GS Plugins</a>';
    return $links;
}

/**
 * On Activation
 */
function on_activation() {
	if ( is_pro_compatible( true ) ) {
		delete_option( 'GS_book_permalinks_flushed' );
	}
}

/**
 * On Deactivation
 */
function on_deactivation() {
}

/**
 * Compatibility check with Pro plugin
 */
function is_pro_compatible( $silent = false ) {
    require_once ABSPATH . 'wp-admin/includes/plugin.php';
    if ( defined('GS_BOOKS_PRO_VERSION') && is_plugin_active( GS_BOOKS_PRO_PLUGIN ) ) {
        if ( version_compare( GS_BOOKS_PRO_VERSION, GS_BOOKS_MIN_PRO_VERSION, '<' ) ) {
			if ( ! $silent ) add_action( 'admin_notices', 'GS_BOOKS\pro_compatibility_notice' );
            return false;
        }
    }
    return true;
}

/**
 * Get an instance of WP_Filesystem.
 *
 * @return WP_Filesystem_Base|false
 */
function get_wp_filesystem() {
    global $wp_filesystem;

    if ( empty( $wp_filesystem ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        WP_Filesystem();
    }

    return $wp_filesystem ?: false;
}


/**
 * Upgrade notice if compatibility fails
 */
function pro_compatibility_notice() {

    $screen = get_current_screen();
    
    if ( isset( $screen->parent_file ) && 'plugins.php' === $screen->parent_file && 'update' === $screen->id ) return;
    if ( 'update' === $screen->base && 'update' === $screen->id ) return;

    if ( ! current_user_can( 'update_plugins' ) ) return;

    $upgrade_url = wp_nonce_url( self_admin_url( 'update.php?action=upgrade-plugin&plugin=' . GS_BOOKS_PRO_PLUGIN ), 'upgrade-plugin_' . GS_BOOKS_PRO_PLUGIN );

	?>

	<div class="error">
		<p>
			<p><?php esc_html_e( 'GS Books Showcase is not working because you need to upgrade the GS Books Showcase Pro plugin to latest version.', 'gs-books-showcase' ); ?></p>
			<p><a href="<?php echo esc_url( $upgrade_url ); ?>" class="button-primary"><?php esc_html_e( 'Upgrade GS Books Showcase Pro Now', 'gs-books-showcase' ); ?></a></p>
		</p>
	</div>

	<?php
    
}

/**
 * Plugins Load Text Domain
 */
function gs_load_textdomain() {
    load_plugin_textdomain( 'gs-books-showcase', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
}

function wp_get_attachment_image_no_srcset($attachment_id, $size = 'thumbnail', $icon = false, $attr = '') {
    // add a filter to return null for srcset
    add_filter( 'wp_calculate_image_srcset_meta', '__return_null' );
    // get the srcset-less img html
    $html = wp_get_attachment_image($attachment_id, $size, $icon, $attr);
    // remove the above filter
    remove_filter( 'wp_calculate_image_srcset_meta', '__return_null' );
    return $html;
}

function get_post_type_title() {
    return __( 'Books', 'gs-books-showcase' );
}

function gs_get_post_type_archive_title() {
    $archive_title = getoption( 'archive_page_title', get_post_type_title() );
    if ( empty($archive_title) ) return get_post_type_title();
    return $archive_title;
}

function gsb_text_to_class($text) {
	
    $original = (string)$text;

    // Quick exit if already ASCII
    if (preg_match('/^[\x00-\x7F]+$/', $original)) {
        $s = strtolower($original);
        $s = preg_replace('/[^a-z0-9]+/', '-', $s);
        return trim($s, '-') ?: substr(md5($original), 0, 8);
    }

    // Normalize whitespace and remove control characters
    $s = preg_replace('/[\p{C}\s]+/u', ' ', $original);
    $s = trim($s);

    // Transliterate to Latin
    if (function_exists('transliterator_transliterate')) {
        $s = transliterator_transliterate('Any-Latin; Latin-ASCII; Lower()', $s);
    } else {
        // Fallback for Arabic
        $map_ar = [
            'ا'=>'a','أ'=>'a','إ'=>'i','آ'=>'a','ب'=>'b','ت'=>'t','ث'=>'th','ج'=>'j','ح'=>'h','خ'=>'kh',
            'د'=>'d','ذ'=>'dh','ر'=>'r','ز'=>'z','س'=>'s','ش'=>'sh','ص'=>'s','ض'=>'d','ط'=>'t','ظ'=>'z',
            'ع'=>'a','غ'=>'gh','ف'=>'f','ق'=>'q','ك'=>'k','ل'=>'l','م'=>'m','ن'=>'n','ه'=>'h','و'=>'w',
            'ؤ'=>'u','ي'=>'y','ئ'=>'i','ة'=>'a','ى'=>'a','ﻻ'=>'la','لا'=>'la'
        ];
        $s = strtr($s, $map_ar);
    }

    // Make class-safe
    $s = strtolower($s);
    $s = preg_replace('/[^a-z0-9]+/', '-', $s); // only letters, numbers, hyphens
    $s = preg_replace('/^-+|-+$/', '', $s); // trim leading/trailing hyphens
	$s = strtolower($s);

    return $s ?: substr(md5($original), 0, 8);
}