<?php
/**
 * GrowthNation
 *
 * @package     GrowthNation
 * @author      GrowthNation
 * @copyright   2025 GrowthNation
 * @license     GPL v2 or later
 *
 * @wordpress-plugin
 * Plugin Name:       GrowthNation
 * Description:       This plugin enables you to push blog posts to your WordPress site from the GrowthNation platform. Posts can be created as drafts or published directly.
 * Version:           1.0.5
 * Author:            GrowthNation
 * Author URI:        https://growthnation.ai
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       growthnation
 */

// Prevent direct file access
if (!defined('ABSPATH')) {
    exit;
}

class GrowthNation
{
    public function __construct() 
    {
        add_action('rest_api_init', array($this, 'register_endpoints'));
    }

    public function register_endpoints() 
    {
        // Test connection endpoint
        register_rest_route('growthnation/v1', '/test', array(
            'methods' => 'POST',
            'callback' => array($this, 'test_connection'),
            'permission_callback' => array($this, 'check_permission')
        ));

        // Create post endpoint (preferred for new integrations)
        register_rest_route('growthnation/v1', '/post-blog', array(
            'methods' => 'POST',
            'callback' => array($this, 'create_post'),
            'permission_callback' => array($this, 'check_permission')
        ));

        // Legacy endpoint (kept for backwards compatibility)
        register_rest_route('growthnation/v1', '/post-draft-blog', array(
            'methods' => 'POST',
            'callback' => array($this, 'create_post'),
            'permission_callback' => array($this, 'check_permission')
        ));
    }

    /**
     * Permission callback for REST API endpoints
     * 
     * This validates that the request contains valid WordPress credentials
     * and that the user has the required capabilities to create posts.
     * 
     * @param WP_REST_Request $request The REST request object
     * @return bool|WP_Error True if permission granted, WP_Error otherwise
     */
    public function check_permission($request) 
    {
        // Get credentials from request
        $username = $request->get_param('username');
        $password = $request->get_param('password');

        // Check if credentials are provided
        if (empty($username) || empty($password)) {
            return new WP_Error(
                'missing_credentials',
                'Username and password are required',
                array('status' => 401)
            );
        }

        // Authenticate the user
        $user = wp_authenticate($username, $password);

        // Check if authentication was successful
        if (is_wp_error($user)) {
            return new WP_Error(
                'authentication_failed',
                'Invalid credentials',
                array('status' => 401)
            );
        }

        // Check if user has permission to edit posts
        if (!$user->has_cap('edit_posts')) {
            return new WP_Error(
                'insufficient_permissions',
                'User does not have permission to create or edit posts',
                array('status' => 403)
            );
        }

        // Store authenticated user in request for use in callbacks
        $request->set_param('authenticated_user', $user);

        return true;
    }

    public function test_connection($request) 
    {
        // Get the authenticated user from the permission callback
        $user = $request->get_param('authenticated_user');

        // Get user capabilities
        $capabilities = array();
        if ($user->has_cap('edit_posts')) {
            $capabilities[] = 'edit_posts';
        }
        if ($user->has_cap('publish_posts')) {
            $capabilities[] = 'publish_posts';
        }

        return array(
            'success' => true,
            'message' => 'Connection successful',
            'site_info' => array(
                'site_name' => esc_html(get_bloginfo('name')),
                'site_url' => esc_url(get_bloginfo('url')),
                'wp_version' => esc_html(get_bloginfo('version')),
                'user_display_name' => esc_html($user->display_name),
                'user_capabilities' => $capabilities
            )
        );
    }

    public function create_post($request)
    {
        // Get the authenticated user from the permission callback
        $user = $request->get_param('authenticated_user');

        // Get and sanitize other parameters
        $title = sanitize_text_field($request->get_param('title'));
        $content = wp_kses_post($request->get_param('content'));
        $excerpt = sanitize_text_field($request->get_param('excerpt'));
        $categories = $request->get_param('categories');
        $tags = $request->get_param('tags');
        $featured_image_url = esc_url_raw($request->get_param('featured_image_url'));
        $status = sanitize_text_field($request->get_param('status'));
        $post_id = $request->get_param('post_id');

        // Validate and normalize status parameter
        $allowed_statuses = array('draft', 'publish');
        if (empty($status) || !in_array($status, $allowed_statuses)) {
            $status = 'draft'; // Default to draft if not specified or invalid
        }

        // Check publish capability if trying to publish
        if ($status === 'publish' && !$user->has_cap('publish_posts')) {
            return new WP_Error(
                'insufficient_permissions',
                'User does not have permission to publish posts. Post will be saved as draft.',
                array('status' => 403)
            );
        }

        // Validate post ownership when updating an existing post (security critical)
        if (!empty($post_id)) {
            $existing_post = get_post($post_id);
            if (!$existing_post) {
                return new WP_Error(
                    'post_not_found',
                    'Post not found',
                    array('status' => 404)
                );
            }
            if ($existing_post->post_author != $user->ID) {
                return new WP_Error(
                    'insufficient_permissions',
                    'Cannot edit posts by other users',
                    array('status' => 403)
                );
            }
        }

        // Validate required fields
        if (empty($title)) {
            return new WP_Error(
                'missing_title',
                'Post title is required',
                array('status' => 400)
            );
        }

        if (empty($content)) {
            return new WP_Error(
                'missing_content',
                'Post content is required',
                array('status' => 400)
            );
        }

        // Create post data
        $post_data = array(
            'post_title' => $title,
            'post_content' => $content,
            'post_status' => $status,
            'post_author' => $user->ID,
        );

        if (!empty($excerpt)) {
            $post_data['post_excerpt'] = $excerpt;
        }

        // Insert or update the post
        if (!empty($post_id)) {
            // Update existing post
            $post_data['ID'] = $post_id;
            $result = wp_update_post($post_data, true);
        } else {
            // Create new post
            $result = wp_insert_post($post_data, true);
        }

        if (is_wp_error($result)) {
            return new WP_Error(
                'post_error',
                $result->get_error_message(),
                array('status' => 500)
            );
        }
        $created_post_id = $result;

        // Handle categories
        if (!empty($categories) && is_array($categories)) {
            $category_ids = array();
            foreach ($categories as $category) {
                $category = sanitize_text_field($category);
                $cat = get_category_by_slug($category);
                if ($cat) {
                    $category_ids[] = $cat->term_id;
                } else {
                    // Create category if it doesn't exist
                    $new_cat = wp_create_category($category);
                    if (!is_wp_error($new_cat)) {
                        $category_ids[] = $new_cat;
                    }
                }
            }
            if (!empty($category_ids)) {
                wp_set_post_categories($created_post_id, $category_ids);
            }
        }

        // Handle tags
        if (!empty($tags) && is_array($tags)) {
            $sanitized_tags = array_map('sanitize_text_field', $tags);
            wp_set_post_tags($created_post_id, $sanitized_tags);
        }

        // Handle featured image
        $featured_image_result = array('status' => 'skipped');
        if (!empty($featured_image_url)) {
            try {
                $featured_image_result = $this->download_and_set_featured_image($created_post_id, $featured_image_url, $title);
            } catch (Exception $e) {
                // Catch any unexpected errors
                $featured_image_result = array(
                    'status' => 'error',
                    'message' => 'Unexpected error: ' . $e->getMessage()
                );
            }
        }

        // Build response message based on status and action
        $status_label = ($status === 'publish') ? 'Published' : 'Draft';
        $action = !empty($post_id) ? 'updated' : 'created';
        $message = $status_label . ' post ' . $action . ' successfully';

        $response = array(
            'success' => true,
            'post_id' => (int) $created_post_id,
            'post_status' => $status,
            'edit_url' => esc_url(admin_url('post.php?post=' . $created_post_id . '&action=edit')),
            'message' => $message,
            'featured_image' => $featured_image_result
        );

        // Add view URL for published posts
        if ($status === 'publish') {
            $response['view_url'] = esc_url(get_permalink($created_post_id));
        }

        return $response;
    }

    /**
     * Download an image from URL and set it as featured image for a post
     *
     * @param int    $post_id    The post ID to attach the image to
     * @param string $image_url  The URL of the image to download
     * @param string $post_title The post title, used for image filename
     * 
     * @return array Status array with 'status', 'message', and optionally 'attachment_id'
     */
    private function download_and_set_featured_image($post_id, $image_url, $post_title) 
    {
        // Ensure WordPress admin functions are available FIRST
        if (!function_exists('wp_tempnam') || !function_exists('media_handle_sideload')) {
            require_once(ABSPATH . 'wp-admin/includes/file.php');
            require_once(ABSPATH . 'wp-admin/includes/media.php');
            require_once(ABSPATH . 'wp-admin/includes/image.php');
        }

        // Basic URL validation
        if (!filter_var($image_url, FILTER_VALIDATE_URL)) {
            return array(
                'status' => 'error',
                'message' => 'Invalid image URL format'
            );
        }

        // Validate URL protocol (only allow HTTP/HTTPS)
        $parsed_url = wp_parse_url($image_url);
        if (!isset($parsed_url['scheme']) || !in_array(strtolower($parsed_url['scheme']), array('http', 'https'))) {
            return array(
                'status' => 'error',
                'message' => 'Only HTTP and HTTPS protocols are allowed'
            );
        }

        // WordPress URL validation
        if (!wp_http_validate_url($image_url)) {
            return array(
                'status' => 'error',
                'message' => 'URL failed WordPress security validation'
            );
        }

        // Download the image
        $response = wp_remote_get($image_url, array(
            'timeout' => 30,
            'user-agent' => 'GrowthNation WordPress Plugin/1.0.0'
        ));

        if (is_wp_error($response)) {
            return array(
                'status' => 'error', 
                'message' => 'Failed to download image: ' . $response->get_error_message()
            );
        }

        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            return array(
                'status' => 'error',
                'message' => 'Image download failed with HTTP ' . $response_code
            );
        }

        // Check Content-Type header
        $content_type = wp_remote_retrieve_header($response, 'content-type');
        $allowed_mime_types = array('image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp');
        $is_valid_type = false;
        
        foreach ($allowed_mime_types as $mime_type) {
            if (strpos($content_type, $mime_type) === 0) {
                $is_valid_type = true;
                break;
            }
        }
        
        if (!$is_valid_type) {
            return array(
                'status' => 'error',
                'message' => 'Invalid content type: ' . $content_type . '. Only JPEG, PNG, GIF, and WebP images are allowed.'
            );
        }

        $image_data = wp_remote_retrieve_body($response);
        if (empty($image_data)) {
            return array(
                'status' => 'error',
                'message' => 'Downloaded image is empty'
            );
        }

        // Check file size (default WordPress limit is usually 2MB for most hosts)
        $max_size = wp_max_upload_size();
        if (strlen($image_data) > $max_size) {
            return array(
                'status' => 'error',
                'message' => 'Image file size exceeds maximum allowed size of ' . size_format($max_size)
            );
        }

        // Get file extension from URL or Content-Type
        $url_parts = wp_parse_url($image_url);
        $path_info = pathinfo($url_parts['path']);
        $extension = isset($path_info['extension']) ? strtolower($path_info['extension']) : '';
        
        // Map content types to extensions if URL doesn't have extension
        if (empty($extension)) {
            $mime_to_ext = array(
                'image/jpeg' => 'jpg',
                'image/jpg' => 'jpg',
                'image/png' => 'png',
                'image/gif' => 'gif',
                'image/webp' => 'webp'
            );
            
            foreach ($mime_to_ext as $mime => $ext) {
                if (strpos($content_type, $mime) === 0) {
                    $extension = $ext;
                    break;
                }
            }
            
            // Default to png if still no extension
            if (empty($extension)) {
                $extension = 'png';
            }
        }
        
        // Validate extension
        $allowed_extensions = array('jpg', 'jpeg', 'png', 'gif', 'webp');
        if (!in_array($extension, $allowed_extensions)) {
            return array(
                'status' => 'error',
                'message' => 'Invalid file extension: ' . $extension
            );
        }
        
        // Create a temporary file with fallback
        $tmp_name = wp_tempnam($image_url);
        if (!$tmp_name) {
            // Fallback to PHP's native tempnam if wp_tempnam fails
            $tmp_name = tempnam(sys_get_temp_dir(), 'growthnation_image_');
            if (!$tmp_name) {
                return array(
                    'status' => 'error',
                    'message' => 'Failed to create temporary file (both wp_tempnam and tempnam failed)'
                );
            }
        }

        // Write image data to temporary file
        if (file_put_contents($tmp_name, $image_data) === false) {
            wp_delete_file($tmp_name); // Clean up on failure
            return array(
                'status' => 'error',
                'message' => 'Failed to write image data'
            );
        }

        // Prepare file array for WordPress media handling
        $file_array = array(
            'name' => sanitize_file_name($post_title . '.' . $extension),
            'tmp_name' => $tmp_name
        );

        // Check file type using WordPress function
        $wp_filetype = wp_check_filetype_and_ext($tmp_name, $file_array['name']);
        if (!$wp_filetype['type']) {
            wp_delete_file($tmp_name);
            return array(
                'status' => 'error',
                'message' => 'File type is not allowed'
            );
        }

        // Admin functions already loaded at function start

        // Upload to WordPress media library
        $attachment_id = media_handle_sideload($file_array, $post_id);

        // Clean up temporary file
        if (file_exists($tmp_name)) {
            wp_delete_file($tmp_name);
        }

        if (is_wp_error($attachment_id)) {
            return array(
                'status' => 'error',
                'message' => 'Failed to upload to media library: ' . $attachment_id->get_error_message()
            );
        }

        // Set as featured image
        $set_result = set_post_thumbnail($post_id, $attachment_id);
        if (!$set_result) {
            return array(
                'status' => 'error',
                'message' => 'Failed to set featured image',
                'attachment_id' => $attachment_id
            );
        }

        return array(
            'status' => 'success',
            'attachment_id' => $attachment_id,
            'message' => 'Featured image set successfully'
        );
    }

    private function authenticate_user($username, $password) 
    {
        if (empty($username) || empty($password)) {
            return new WP_Error('missing_credentials', 'Username and password are required');
        }

        $user = wp_authenticate($username, $password);

        if (is_wp_error($user)) {
            return $user;
        }

        // Check if user has permission to create posts
        if (!$user->has_cap('edit_posts')) {
            return new WP_Error('insufficient_permissions', 'User does not have permission to create posts');
        }

        return $user;
    }
}

// Initialize the plugin
new GrowthNation();