<?php
/**
 * Plugin Name: Growffinity CRM for WooCommerce
 * Plugin URI: https://growffinity.com
 * Description: Connect your WooCommerce store to Growffinity CRM. Sync customers, orders, and manage your business from one powerful dashboard.
 * Version: 1.3.2
 * Author: Growffinity
 * Author URI: https://growffinity.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: growffinity-crm-for-woocommerce
 * Domain Path: /languages
 * Requires at least: 5.8
 * Tested up to: 6.9
 * Requires PHP: 7.4
 * WC requires at least: 5.0
 * WC tested up to: 10.4.3
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('GROWFFINITY_VERSION', '1.3.2');
define('GROWFFINITY_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('GROWFFINITY_PLUGIN_URL', plugin_dir_url(__FILE__));
define('GROWFFINITY_API_URL', 'https://woocommerce-crm-backend.onrender.com/api');

/**
 * Main Growffinity CRM Class
 */
class Growffinity_CRM {

    /**
     * Instance of this class
     */
    private static $instance = null;

    /**
     * Get instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Declare HPOS compatibility
        add_action('before_woocommerce_init', array($this, 'declare_hpos_compatibility'));

        // Check if WooCommerce is active
        add_action('admin_init', array($this, 'check_woocommerce'));

        // Admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));

        // Admin scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));

        // AJAX handlers
        add_action('wp_ajax_growffinity_connect_store', array($this, 'ajax_connect_store'));
        add_action('wp_ajax_growffinity_disconnect_store', array($this, 'ajax_disconnect_store'));
        add_action('wp_ajax_growffinity_sync_now', array($this, 'ajax_sync_now'));

        // Automatic sync hooks
        add_action('woocommerce_new_order', array($this, 'sync_order'), 10, 1);
        add_action('woocommerce_update_order', array($this, 'sync_order'), 10, 1);
        add_action('woocommerce_created_customer', array($this, 'sync_customer'), 10, 1);
        add_action('woocommerce_update_customer', array($this, 'sync_customer'), 10, 1);

        // Activation/deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }

    /**
     * Declare HPOS compatibility
     */
    public function declare_hpos_compatibility() {
        if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
            \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
        }
    }

    /**
     * Check if WooCommerce is active
     */
    public function check_woocommerce() {
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', array($this, 'woocommerce_missing_notice'));
            deactivate_plugins(plugin_basename(__FILE__));
        }
    }

    /**
     * WooCommerce missing notice
     */
    public function woocommerce_missing_notice() {
        ?>
        <div class="error">
            <p><?php esc_html_e('Growffinity CRM requires WooCommerce to be installed and active.', 'growffinity-crm-for-woocommerce'); ?></p>
        </div>
        <?php
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            __('Growffinity CRM', 'growffinity-crm-for-woocommerce'),
            __('Growffinity CRM', 'growffinity-crm-for-woocommerce'),
            'manage_woocommerce',
            'growffinity-crm-for-woocommerce',
            array($this, 'admin_page'),
            'dashicons-chart-area',
            56
        );
    }

    /**
     * Admin scripts and styles
     */
    public function admin_scripts($hook) {
        if ('toplevel_page_growffinity-crm-for-woocommerce' !== $hook) {
            return;
        }

        wp_enqueue_style('growffinity-admin', GROWFFINITY_PLUGIN_URL . 'assets/css/admin.css', array(), GROWFFINITY_VERSION);
        wp_enqueue_script('growffinity-admin', GROWFFINITY_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), GROWFFINITY_VERSION, true);

        wp_localize_script('growffinity-admin', 'growffinityData', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('growffinity_nonce'),
            'isConnected' => $this->is_connected(),
            'apiKey' => get_option('growffinity_api_key', ''),
        ));
    }

    /**
     * Admin page
     */
    public function admin_page() {
        require_once GROWFFINITY_PLUGIN_DIR . 'includes/admin-page.php';
    }

    /**
     * Check if store is connected
     */
    public function is_connected() {
        $api_key = get_option('growffinity_api_key', '');
        $store_id = get_option('growffinity_store_id', '');
        return !empty($api_key) && !empty($store_id);
    }

    /**
     * AJAX: Connect store
     */
    public function ajax_connect_store() {
        check_ajax_referer('growffinity_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }

        // Validate and sanitize input with isset checks and wp_unslash
        if (!isset($_POST['email']) || !isset($_POST['password'])) {
            wp_send_json_error(array('message' => 'Email and password are required'));
        }

        $email = sanitize_email(wp_unslash($_POST['email']));
        $password = sanitize_text_field(wp_unslash($_POST['password']));

        if (empty($email) || empty($password)) {
            wp_send_json_error(array('message' => 'Email and password are required'));
        }

        // Authenticate with Growffinity API
        $response = $this->authenticate_with_api($email, $password);

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
        }

        // Save API key and store ID
        update_option('growffinity_api_key', $response['token']);
        update_option('growffinity_user_id', $response['userId']);
        update_option('growffinity_user_email', $email);

        // Create or connect store
        $store_response = $this->connect_store_to_api($response['token']);

        if (is_wp_error($store_response)) {
            wp_send_json_error(array('message' => $store_response->get_error_message()));
        }

        update_option('growffinity_store_id', $store_response['storeId']);
        update_option('growffinity_connected_at', current_time('mysql'));

        wp_send_json_success(array(
            'message' => 'Successfully connected to Growffinity!',
            'storeId' => $store_response['storeId']
        ));
    }

    /**
     * AJAX: Disconnect store
     */
    public function ajax_disconnect_store() {
        check_ajax_referer('growffinity_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }

        delete_option('growffinity_api_key');
        delete_option('growffinity_store_id');
        delete_option('growffinity_user_id');
        delete_option('growffinity_user_email');
        delete_option('growffinity_connected_at');
        delete_option('growffinity_consumer_key');
        delete_option('growffinity_consumer_secret');

        wp_send_json_success(array('message' => 'Successfully disconnected'));
    }

    /**
     * AJAX: Sync now
     */
    public function ajax_sync_now() {
        check_ajax_referer('growffinity_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }

        if (!$this->is_connected()) {
            wp_send_json_error(array('message' => 'Store not connected'));
        }

        $result = $this->perform_full_sync();

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        $customers_synced = isset($result['customers']) ? $result['customers'] : 0;
        $orders_synced = isset($result['orders']) ? $result['orders'] : 0;
        $products_synced = isset($result['products']) ? $result['products'] : 0;

        wp_send_json_success(array(
            'message' => sprintf('Sync completed! %d customers, %d orders, and %d products synced to Growffinity.', $customers_synced, $orders_synced, $products_synced),
            'stats' => $result
        ));
    }

    /**
     * Authenticate with Growffinity API
     */
    private function authenticate_with_api($email, $password) {
        $response = wp_remote_post(GROWFFINITY_API_URL . '/auth/login', array(
            'body' => json_encode(array(
                'email' => $email,
                'password' => $password
            )),
            'headers' => array(
                'Content-Type' => 'application/json'
            ),
            'timeout' => 15
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code !== 200) {
            return new WP_Error('auth_failed', $body['error'] ?? 'Authentication failed');
        }

        return $body;
    }

    /**
     * Get or create WooCommerce API credentials for dashboard sync
     */
    private function get_or_create_api_credentials() {
        global $wpdb;

        // Check if we already have stored credentials
        $consumer_key = get_option('growffinity_consumer_key');
        $consumer_secret = get_option('growffinity_consumer_secret');

        if ($consumer_key && $consumer_secret) {
            return array(
                'consumer_key' => $consumer_key,
                'consumer_secret' => $consumer_secret
            );
        }

        // Generate new API credentials
        $user = wp_get_current_user();
        $description = 'Growffinity CRM - Auto-generated for dashboard sync';

        // Generate keys
        $consumer_key = 'ck_' . wc_rand_hash();
        $consumer_secret = 'cs_' . wc_rand_hash();

        // Insert into WooCommerce API keys table
        // Note: 'read_write' permission is needed to create webhooks for real-time sync
        $data = array(
            'user_id'         => $user->ID,
            'description'     => $description,
            'permissions'     => 'read_write',
            'consumer_key'    => wc_api_hash($consumer_key),
            'consumer_secret' => $consumer_secret,
            'truncated_key'   => substr($consumer_key, -7),
        );

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- WooCommerce does not provide an API to create REST API keys programmatically
        $wpdb->insert(
            $wpdb->prefix . 'woocommerce_api_keys',
            $data,
            array('%d', '%s', '%s', '%s', '%s', '%s')
        );

        if ($wpdb->last_error) {
            return new WP_Error('api_key_creation_failed', 'Failed to create API keys: ' . $wpdb->last_error);
        }

        // Store the credentials
        update_option('growffinity_consumer_key', $consumer_key);
        update_option('growffinity_consumer_secret', $consumer_secret);

        return array(
            'consumer_key' => $consumer_key,
            'consumer_secret' => $consumer_secret
        );
    }

    /**
     * Connect store to API
     */
    private function connect_store_to_api($token) {
        // Generate or retrieve WooCommerce REST API credentials
        $api_credentials = $this->get_or_create_api_credentials();

        if (is_wp_error($api_credentials)) {
            return $api_credentials;
        }

        $store_data = array(
            'name' => get_bloginfo('name'),
            'url' => get_site_url(),
            'woocommerce_version' => WC()->version,
            'wordpress_version' => get_bloginfo('version'),
            'consumer_key' => $api_credentials['consumer_key'],
            'consumer_secret' => $api_credentials['consumer_secret'],
        );

        $response = wp_remote_post(GROWFFINITY_API_URL . '/stores', array(
            'body' => json_encode($store_data),
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $token
            ),
            'timeout' => 15
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code !== 200 && $status_code !== 201) {
            return new WP_Error('connection_failed', $body['error'] ?? 'Failed to connect store');
        }

        return array('storeId' => $body['id']);
    }

    /**
     * Perform full sync
     */
    private function perform_full_sync() {
        $api_key = get_option('growffinity_api_key');
        $store_id = get_option('growffinity_store_id');

        if (!$api_key || !$store_id) {
            return new WP_Error('not_connected', 'Store not connected');
        }

        $results = array(
            'customers' => 0,
            'orders' => 0,
            'products' => 0
        );

        // Sync customers
        $customer_result = $this->sync_customers_bulk($api_key, $store_id);
        if (!is_wp_error($customer_result)) {
            $results['customers'] = $customer_result;
        }

        // Sync orders
        $order_result = $this->sync_orders_bulk($api_key, $store_id);
        if (!is_wp_error($order_result)) {
            $results['orders'] = $order_result;
        }

        // Sync products
        $product_result = $this->sync_products_bulk($api_key, $store_id);
        if (!is_wp_error($product_result)) {
            $results['products'] = $product_result;
        }

        return $results;
    }

    /**
     * Sync all customers in bulk
     */
    private function sync_customers_bulk($api_key, $store_id) {
        if (!class_exists('WC_Customer')) {
            return new WP_Error('wc_not_active', 'WooCommerce is not active');
        }

        // Get all WooCommerce customers (limit to 500 for performance)
        $customers = get_users(array(
            'role' => 'customer',
            'number' => 500
        ));


        $customer_data = array();

        foreach ($customers as $user) {
            try {
                $customer = new WC_Customer($user->ID);

                $customer_data[] = array(
                    'id' => $user->ID,
                    'first_name' => $customer->get_first_name(),
                    'last_name' => $customer->get_last_name(),
                    'email' => $customer->get_email(),
                    'billing' => array(
                        'phone' => $customer->get_billing_phone()
                    ),
                    'total_spent' => $customer->get_total_spent(),
                    'orders_count' => $customer->get_order_count()
                );
            } catch (Exception $e) {
                continue;
            }
        }


        if (empty($customer_data)) {
            return 0;
        }

        // Send to Growffinity API
        $response = wp_remote_post(GROWFFINITY_API_URL . '/sync/customers/' . $store_id . '/bulk', array(
            'body' => json_encode(array('customers' => $customer_data)),
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $api_key
            ),
            'timeout' => 60
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code !== 200) {
            return new WP_Error('sync_failed', $body['error'] ?? 'Failed to sync customers');
        }

        return $body['inserted'] + $body['updated'];
    }

    /**
     * Sync all orders in bulk
     */
    private function sync_orders_bulk($api_key, $store_id) {
        if (!function_exists('wc_get_orders')) {
            return new WP_Error('wc_not_active', 'WooCommerce is not active');
        }

        // Get recent WooCommerce orders (limit to 500 for performance)
        $orders = wc_get_orders(array(
            'limit' => 500,
            'orderby' => 'date',
            'order' => 'DESC',
            'type' => 'shop_order' // Only get regular orders, not refunds
        ));


        $order_data = array();

        foreach ($orders as $order) {
            try {
                // Skip if not a WC_Order object (safety check)
                if (!is_a($order, 'WC_Order')) {
                    continue;
                }

                // Get line items for this order
                $line_items = array();
                foreach ($order->get_items() as $item_id => $item) {
                    $product = $item->get_product();
                    $line_items[] = array(
                        'product_id' => $item->get_product_id(),
                        'name' => $item->get_name(),
                        'sku' => $product ? $product->get_sku() : '',
                        'quantity' => $item->get_quantity(),
                        'subtotal' => $item->get_subtotal(),
                        'total' => $item->get_total(),
                        'total_tax' => $item->get_total_tax()
                    );
                }

                $order_data[] = array(
                    'id' => $order->get_id(),
                    'number' => $order->get_order_number(),
                    'customer_id' => $order->get_customer_id(),
                    'total' => $order->get_total(),
                    'status' => $order->get_status(),
                    'date_created' => $order->get_date_created()->date('Y-m-d H:i:s'),
                    'payment_method' => $order->get_payment_method(),
                    'payment_method_title' => $order->get_payment_method_title(),
                    'currency' => $order->get_currency(),
                    'billing' => array(
                        'first_name' => $order->get_billing_first_name(),
                        'last_name' => $order->get_billing_last_name(),
                        'email' => $order->get_billing_email(),
                        'phone' => $order->get_billing_phone(),
                        'address_1' => $order->get_billing_address_1(),
                        'address_2' => $order->get_billing_address_2(),
                        'city' => $order->get_billing_city(),
                        'state' => $order->get_billing_state(),
                        'postcode' => $order->get_billing_postcode(),
                        'country' => $order->get_billing_country()
                    ),
                    'shipping' => array(
                        'first_name' => $order->get_shipping_first_name(),
                        'last_name' => $order->get_shipping_last_name(),
                        'address_1' => $order->get_shipping_address_1(),
                        'address_2' => $order->get_shipping_address_2(),
                        'city' => $order->get_shipping_city(),
                        'state' => $order->get_shipping_state(),
                        'postcode' => $order->get_shipping_postcode(),
                        'country' => $order->get_shipping_country()
                    ),
                    'line_items' => $line_items
                );
            } catch (Exception $e) {
                continue;
            }
        }


        if (empty($order_data)) {
            return 0;
        }

        // Send to Growffinity API
        $response = wp_remote_post(GROWFFINITY_API_URL . '/sync/orders/' . $store_id . '/bulk', array(
            'body' => json_encode(array('orders' => $order_data)),
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $api_key
            ),
            'timeout' => 60
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);


        if ($status_code !== 200) {
            return new WP_Error('sync_failed', $body['error'] ?? 'Failed to sync orders');
        }

        $synced = $body['inserted'] + $body['updated'];

        return $synced;
    }

    /**
     * Sync all products in bulk
     */
    private function sync_products_bulk($api_key, $store_id) {
        if (!function_exists('wc_get_products')) {
            return new WP_Error('wc_not_active', 'WooCommerce is not active');
        }

        // Get all WooCommerce products (limit to 500 for performance)
        // This gets parent products (simple and variable)
        $products = wc_get_products(array(
            'limit' => 500,
            'status' => 'publish',
            'orderby' => 'date',
            'order' => 'DESC',
            'type' => array('simple', 'variable') // Get both simple and variable products
        ));


        $product_data = array();

        foreach ($products as $product) {
            try {
                // Get product categories
                $categories = array();
                $terms = get_the_terms($product->get_id(), 'product_cat');
                if ($terms && !is_wp_error($terms)) {
                    foreach ($terms as $term) {
                        $categories[] = array('name' => $term->name);
                    }
                }

                // Get product images
                $images = array();
                $image_id = $product->get_image_id();
                if ($image_id) {
                    $image_url = wp_get_attachment_url($image_id);
                    if ($image_url) {
                        $images[] = array('src' => $image_url);
                    }
                }

                // Add parent product
                $product_data[] = array(
                    'id' => $product->get_id(),
                    'name' => $product->get_name(),
                    'sku' => $product->get_sku(),
                    'price' => $product->get_regular_price(),
                    'stock_quantity' => $product->get_stock_quantity(),
                    'stock_status' => $product->get_stock_status(),
                    'categories' => $categories,
                    'images' => $images,
                    'parent_id' => 0 // Parent products have parent_id = 0
                );

                // If this is a variable product, fetch and add its variations
                if ($product->is_type('variable')) {
                    $variations = $product->get_children();

                    foreach ($variations as $variation_id) {
                        try {
                            $variation = wc_get_product($variation_id);

                            if (!$variation) {
                                continue;
                            }

                            // Get variation attributes (e.g., {"Size": "Large", "Color": "Red"})
                            $attributes = array();
                            $variation_attributes = $variation->get_variation_attributes();

                            foreach ($variation_attributes as $attr_name => $attr_value) {
                                // Remove 'attribute_' prefix from attribute name
                                $clean_name = str_replace('attribute_', '', $attr_name);
                                // Convert pa_size to Size, pa_color to Color, etc.
                                $clean_name = str_replace('pa_', '', $clean_name);
                                $clean_name = ucwords(str_replace(array('-', '_'), ' ', $clean_name));

                                $attributes[] = array(
                                    'name' => $clean_name,
                                    'option' => $attr_value
                                );
                            }

                            // Add variation to product data
                            $product_data[] = array(
                                'id' => $variation->get_id(),
                                'name' => $variation->get_name(),
                                'sku' => $variation->get_sku(),
                                'price' => $variation->get_regular_price() ?: $variation->get_price(),
                                'stock_quantity' => $variation->get_stock_quantity(),
                                'stock_status' => $variation->get_stock_status(),
                                'categories' => $categories, // Inherit from parent
                                'images' => $images, // Inherit from parent
                                'parent_id' => $product->get_id(), // Link to parent product
                                'attributes' => $attributes
                            );
                        } catch (Exception $e) {
                            // Skip this variation if there's an error
                            continue;
                        }
                    }
                }
            } catch (Exception $e) {
                continue;
            }
        }


        if (empty($product_data)) {
            return 0;
        }

        // Send to Growffinity API
        $response = wp_remote_post(GROWFFINITY_API_URL . '/sync/products/' . $store_id . '/bulk', array(
            'body' => json_encode(array('products' => $product_data)),
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $api_key
            ),
            'timeout' => 60
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);


        if ($status_code !== 200) {
            return new WP_Error('sync_failed', $body['error'] ?? 'Failed to sync products');
        }

        $synced = $body['inserted'] + $body['updated'];
        return $synced;
    }

    /**
     * Sync single order
     */
    public function sync_order($order_id) {
        if (!$this->is_connected()) {
            return;
        }

        // Implementation will sync order to API
    }

    /**
     * Sync single customer
     */
    public function sync_customer($customer_id) {
        if (!$this->is_connected()) {
            return;
        }

        // Implementation will sync customer to API
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Set default options
        add_option('growffinity_auto_sync', 'yes');
        add_option('growffinity_sync_frequency', 'realtime');
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Cleanup if needed
    }
}

// Initialize the plugin
function growffinity_crm_init() {
    return Growffinity_CRM::get_instance();
}

// Start the plugin
add_action('plugins_loaded', 'growffinity_crm_init');
