<?php
/**
 * Plugin Name: GrL'Ca Search for WooCommerce
 * Plugin URI: https://grlca.com/grlca-wordpress-plugins/grlca-search-for-woocommerce/
 * Description: Enhanced search functionality for WooCommerce with autocomplete suggestions
 * Version: 1.0.0
 * Author: GrL'Ca
 * Author URI: https://grlca.com
 * License: GPL v2 or later
 * Text Domain: grlca-search-for-woocommerce
 * Requires Plugins: woocommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

if (!in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) {
    add_action('admin_notices', function() {
        echo '<div class="error"><p>' . esc_html__('GrL\'Ca Search for WooCommerce requires WooCommerce to be installed and active.', 'grlca-search-for-woocommerce') . '</p></div>';
    });
    return;
}

require_once plugin_dir_path(__FILE__) . 'includes/grlca-dashboard.php';

class GrLca_Search_For_WooCommerce {
    
    public function __construct() {
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_grlca_search_suggestions', array($this, 'ajax_search_suggestions'));
        add_action('wp_ajax_nopriv_grlca_search_suggestions', array($this, 'ajax_search_suggestions'));
        add_shortcode('grlca_search', array($this, 'search_shortcode'));
        add_filter('get_product_search_form', array($this, 'enhance_search_form'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('init', array($this, 'init'));
        add_action('template_redirect', array($this, 'verify_search_submission'));
    }
    
    public function init() {
    }
    
    public function add_admin_menu() {
        add_submenu_page(
            'grlca-dashboard',
            __('Search Settings', 'grlca-search-for-woocommerce'),
            __('Search', 'grlca-search-for-woocommerce'),
            'manage_options',
            'grlca-search-settings',
            array($this, 'display_settings_page')
        );
    }
    
    public function display_settings_page() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'grlca-search-for-woocommerce'));
        }
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('GrL\'Ca Search for WooCommerce Settings', 'grlca-search-for-woocommerce'); ?></h1>
            
            <div class="card">
                <h2><?php echo esc_html__('Search Features', 'grlca-search-for-woocommerce'); ?></h2>
                <p><?php echo esc_html__('The GrL\'Ca Search for WooCommerce plugin enhances your WooCommerce store with:', 'grlca-search-for-woocommerce'); ?></p>
                <ul>
                    <li><?php echo esc_html__('Autocomplete search suggestions', 'grlca-search-for-woocommerce'); ?></li>
                    <li><?php echo esc_html__('Product images and prices in suggestions', 'grlca-search-for-woocommerce'); ?></li>
                    <li><?php echo esc_html__('Category suggestions', 'grlca-search-for-woocommerce'); ?></li>
                    <li><?php echo esc_html__('Keyboard navigation support', 'grlca-search-for-woocommerce'); ?></li>
                    <li><?php echo esc_html__('Mobile-responsive design', 'grlca-search-for-woocommerce'); ?></li>
                </ul>
            </div>
            
            <div class="card">
                <h2><?php echo esc_html__('How to Use', 'grlca-search-for-woocommerce'); ?></h2>
                
                <h3><?php echo esc_html__('Automatic Enhancement:', 'grlca-search-for-woocommerce'); ?></h3>
                <p><?php echo esc_html__('All existing WooCommerce search forms are automatically enhanced with autocomplete functionality.', 'grlca-search-for-woocommerce'); ?></p>
                
                <h3><?php echo esc_html__('Shortcode:', 'grlca-search-for-woocommerce'); ?></h3>
                <p><?php echo esc_html__('Use the shortcode to add search forms anywhere:', 'grlca-search-for-woocommerce'); ?></p>
                <code>[grlca_search]</code>
                
                <h4><?php echo esc_html__('Custom Shortcode:', 'grlca-search-for-woocommerce'); ?></h4>
                <code>[grlca_search placeholder="Your custom placeholder" class="your-custom-class"]</code>
            </div>
            
            <div class="card">
                <h2><?php echo esc_html__('Search Preview', 'grlca-search-for-woocommerce'); ?></h2>
                <p><?php echo esc_html__('Here\'s how the search form looks with autocomplete functionality:', 'grlca-search-for-woocommerce'); ?></p>
                <div style="max-width: 400px; margin: 20px 0;">
                    <?php echo wp_kses_post($this->search_shortcode(array())); ?>
                </div>
                <p><em><?php echo esc_html__('Start typing to see autocomplete suggestions with product images and prices.', 'grlca-search-for-woocommerce'); ?></em></p>
            </div>
            
            <div class="card">
                <h2><?php echo esc_html__('Plugin Information', 'grlca-search-for-woocommerce'); ?></h2>
                <p><strong><?php echo esc_html__('Version:', 'grlca-search-for-woocommerce'); ?></strong> 1.0.0</p>
                <p><strong><?php echo esc_html__('Text Domain:', 'grlca-search-for-woocommerce'); ?></strong> grlca-search-for-woocommerce</p>
                <p><strong><?php echo esc_html__('Requirements:', 'grlca-search-for-woocommerce'); ?></strong> WooCommerce</p>
            </div>
        </div>
        <?php
    }
    
    public function enqueue_scripts() {
        wp_enqueue_script('jquery');
        
        wp_enqueue_script(
            'grlca-search',
            plugin_dir_url(__FILE__) . 'assets/js/grlca-search.js',
            array('jquery'),
            '1.0.0',
            true
        );
        
        wp_enqueue_style(
            'grlca-search',
            plugin_dir_url(__FILE__) . 'assets/css/grlca-search.css',
            array(),
            '1.0.0'
        );
        
        wp_localize_script('grlca-search', 'grlca_search_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('grlca_search_nonce'),
            'loading_text' => __('Loading...', 'grlca-search-for-woocommerce'),
            'no_results_text' => __('No products found', 'grlca-search-for-woocommerce'),
            'error_text' => __('Error loading suggestions', 'grlca-search-for-woocommerce'),
            'category_text' => __('Category', 'grlca-search-for-woocommerce'),
            'products_text' => __('products', 'grlca-search-for-woocommerce'),
            'sku_text' => __('SKU', 'grlca-search-for-woocommerce')
        ));
    }
    
    public function enhance_search_form($form) {
        $form = preg_replace('/<button type="submit"[^>]*>.*?<\/button>/', '', $form);
        $nonce_field = wp_nonce_field('grlca_search_action', 'grlca_search_nonce', true, false);
        $form = str_replace('</form>', $nonce_field . '<div class="grlca-search-suggestions"></div></form>', $form);
        $form = '<div class="grlca-search-wrapper">' . $form . '</div>';
        return $form;
    }
    
    public function verify_search_submission() {
        if (is_search() && isset($_GET['post_type']) && $_GET['post_type'] === 'product' && isset($_GET['s'])) {
            if (!isset($_GET['grlca_search_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['grlca_search_nonce'])), 'grlca_search_action')) {
                wp_die(esc_html__('Security verification failed for search request.', 'grlca-search-for-woocommerce'));
            }
        }
    }
    
    public function ajax_search_suggestions() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'grlca_search_nonce')) {
            wp_die('Security check failed');
        }
        
        if (!isset($_POST['search_term']) || empty($_POST['search_term'])) {
            wp_die();
        }
        
        $search_term = sanitize_text_field(wp_unslash($_POST['search_term']));
        $suggestions = $this->get_search_suggestions($search_term);
        
        wp_send_json_success($suggestions);
    }
    
    private function get_search_suggestions($search_term) {
        $suggestions = array();
        
        $products = new WP_Query(array(
            'post_type' => 'product',
            'posts_per_page' => 5,
            's' => $search_term,
            'post_status' => 'publish'
        ));
        
        foreach ($products->posts as $product_post) {
            $product = wc_get_product($product_post->ID);
            $image = wp_get_attachment_image_src(get_post_thumbnail_id($product_post->ID), 'thumbnail');
            
            $suggestions[] = array(
                'type' => 'product',
                'title' => $product_post->post_title,
                'url' => get_permalink($product_post->ID),
                'price' => $product->get_price_html(),
                'image' => $image ? $image[0] : wc_placeholder_img_src(),
                'sku' => $product->get_sku()
            );
        }
        
        $categories = get_terms(array(
            'taxonomy' => 'product_cat',
            'name__like' => $search_term,
            'number' => 3,
            'hide_empty' => true
        ));
        
        foreach ($categories as $category) {
            $suggestions[] = array(
                'type' => 'category',
                'title' => $category->name,
                'url' => get_term_link($category),
                'count' => $category->count
            );
        }
        
        return $suggestions;
    }
    
    public function search_shortcode($atts = array()) {
        $atts = shortcode_atts(array(
            'placeholder' => __('Search products...', 'grlca-search-for-woocommerce'),
            'class' => ''
        ), $atts);
        
        ob_start();
        ?>
        <div class="grlca-search-shortcode <?php echo esc_attr($atts['class']); ?>">
            <form role="search" method="get" action="<?php echo esc_url(home_url('/')); ?>">
                <input type="hidden" name="post_type" value="product" />
                <?php wp_nonce_field('grlca_search_action', 'grlca_search_nonce'); ?>
                <input type="search" 
                       class="search-field" 
                       placeholder="<?php echo esc_attr($atts['placeholder']); ?>" 
                       value="<?php echo get_search_query(); ?>" 
                       name="s" 
                       autocomplete="off" />
                <div class="grlca-search-suggestions"></div>
            </form>
        </div>
        <?php
        return ob_get_clean();
    }
}

new GrLca_Search_For_WooCommerce();