(function($) {
    'use strict';

    let currentRequest = null;

    function initSearch() {
        $('.grlca-search-wrapper input[name="s"], .grlca-search-shortcode input[name="s"]').each(function() {
            const $input = $(this);
            const $suggestions = $input.closest('form').find('.grlca-search-suggestions');
            
            if ($input.data('grlca-initialized')) {
                return;
            }
            $input.data('grlca-initialized', true);

            $input.on('input', function() {
                const searchTerm = $(this).val().trim();
                
                if (currentRequest) {
                    currentRequest.abort();
                }

                if (searchTerm.length < 2) {
                    $suggestions.hide().empty();
                    return;
                }

                $suggestions.html('<div class="grlca-search-loading">' + grlca_search_ajax.loading_text + '</div>').show();

                currentRequest = $.ajax({
                    url: grlca_search_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'grlca_search_suggestions',
                        search_term: searchTerm,
                        nonce: grlca_search_ajax.nonce
                    },
                    success: function(response) {
                        if (response.success && response.data.length > 0) {
                            displaySuggestions($suggestions, response.data);
                        } else {
                            $suggestions.html('<div class="grlca-search-no-results">' + grlca_search_ajax.no_results_text + '</div>').show();
                        }
                    },
                    error: function() {
                        $suggestions.html('<div class="grlca-search-error">' + grlca_search_ajax.error_text + '</div>').show();
                    }
                });
            });

            $input.on('keydown', function(e) {
                const $suggestions = $(this).closest('form').find('.grlca-search-suggestions');
                const $items = $suggestions.find('.grlca-search-item');
                const $active = $items.filter('.active');

                if (e.keyCode === 38) {
                    e.preventDefault();
                    navigateSuggestions($items, $active, -1);
                } else if (e.keyCode === 40) {
                    e.preventDefault();
                    navigateSuggestions($items, $active, 1);
                } else if (e.keyCode === 13 && $active.length) {
                    e.preventDefault();
                    window.location.href = $active.find('a').attr('href');
                } else if (e.keyCode === 27) {
                    $suggestions.hide().empty();
                }
            });

            $(document).on('click', function(e) {
                if (!$(e.target).closest('.grlca-search-wrapper, .grlca-search-shortcode').length) {
                    $('.grlca-search-suggestions').hide().empty();
                }
            });
        });
    }

    function navigateSuggestions($items, $active, direction) {
        let $next;

        if (!$active.length) {
            $next = direction === 1 ? $items.first() : $items.last();
        } else {
            const currentIndex = $items.index($active);
            let nextIndex = currentIndex + direction;
            
            if (nextIndex < 0) nextIndex = $items.length - 1;
            if (nextIndex >= $items.length) nextIndex = 0;
            
            $next = $items.eq(nextIndex);
        }

        $items.removeClass('active');
        $next.addClass('active');
    }

    function displaySuggestions($container, suggestions) {
        let html = '<div class="grlca-search-suggestions-list">';
        
        suggestions.forEach(function(item) {
            if (item.type === 'product') {
                html += `
                    <div class="grlca-search-item grlca-search-product">
                        <a href="${item.url}">
                            <img src="${item.image}" alt="${item.title}" class="grlca-search-product-image">
                            <div class="grlca-search-product-info">
                                <div class="grlca-search-product-title">${item.title}</div>
                                <div class="grlca-search-product-price">${item.price}</div>
                                ${item.sku ? '<div class="grlca-search-product-sku">' + grlca_search_ajax.sku_text + ': ' + item.sku + '</div>' : ''}
                            </div>
                        </a>
                    </div>
                `;
            } else if (item.type === 'category') {
                html += `
                    <div class="grlca-search-item grlca-search-category">
                        <a href="${item.url}">
                            <div class="grlca-search-category-info">
                                <div class="grlca-search-category-title">${item.title}</div>
                                <div class="grlca-search-category-count">${grlca_search_ajax.category_text} (${item.count} ${grlca_search_ajax.products_text})</div>
                            </div>
                        </a>
                    </div>
                `;
            }
        });
        
        html += '</div>';
        $container.html(html).show();
    }

    $(document).ready(function() {
        initSearch();
    });

    $(document).ajaxComplete(function() {
        initSearch();
    });

})(jQuery);