<?php
/**
 * Gravyware Quick Links
 *
 * @package       ADMLNKLIST
 * @author        David Garthe
 * @license       gplv2-or-later
 * @version       1.81
 *
 * @wordpress-plugin
 * Plugin Name:   Gravyware Quick Links
 * Plugin URI:    https://gravyware.com
 * Description:   Easily create a list of helpful and often used links that will appear on your WordPress dashboard. You no longer have to jump back and forth to your bookmarks all the time - your common links will be on your dashboard!
 * Version:       1.81
 * Author:        Gravyware
 * Author URI:    https://gravyware.com/gravyware-quick-links
 * Text Domain:   gravyware-quick-links
 * Domain Path:   /languages
 * License:       GPLv2 or later
 * License URI:   https://www.gnu.org/licenses/gpl-2.0.html
 *
 * You should have received a copy of the GNU General Public License
 * along with Gravyware Quick Links. If not, see <https://www.gnu.org/licenses/gpl-2.0.html/>.
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) exit;

wp_register_script('gravyware-quick-links-admin-scripts', plugin_dir_url(__FILE__) . 'admin-scripts.js', ['jquery-ui-sortable'], '1.0', array(
    'strategy'  => 'defer',
));

// Add the menu page for the plugin
function gravquli_links_add_admin_menu() {
    add_menu_page(
        'Gravyware\'s Quick Links Manager',
        'Quick Links',
        'manage_options',
        'gravyware-link-manager',
        'gravquli_links_display_admin_page',
        'dashicons-admin-links',
        20
    );
}
add_action('admin_menu', 'gravquli_links_add_admin_menu');

// Display the admin page
function gravquli_links_display_admin_page() {
    $links = get_option('gravquli_links_links', []);

    // Nonce verification for edit_index
    $editing_index = null;
    $editing_link = null;
    if (
        isset($_GET['edit_index'], $_GET['gravquli_links_edit_nonce']) &&
        wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['gravquli_links_edit_nonce'])), 'gravquli_links_edit_link')
    ) {
        $editing_index = absint($_GET['edit_index']);
        $editing_link = isset($links[$editing_index]) ? $links[$editing_index] : null;
    }

    $plugin_url = plugin_dir_url(__FILE__);
    $image_url = $plugin_url . 'gravyware_logo-b_224x40.png';

    ?>
    <div class="wrap container">
        <a href="https://gravyware.com" target="_blank"><img src="<?php echo esc_url($image_url); ?>" alt="Gravyware.com"></a>
        <h1 class="text-center mb-4">Dashboard Quick Links Manager</h1>

        <div class="row">
            <div class="col">

                <form method="post" action="">
                    <?php wp_nonce_field('gravquli_links_save_links', 'gravquli_links_nonce'); ?>
                    <input type="hidden" name="edit_index" value="<?php echo $editing_index !== null ? esc_attr($editing_index) : ''; ?>">
                    <table class="form-table">
                        <tr>
                            <td><label for="gravquli_links_link_name"><strong>Link Name</strong></label></td>
                            <td><input type="text" name="gravquli_links_link_name" id="gravquli_links_link_name" class="regular-text" value="<?php echo esc_attr($editing_link['name'] ?? ''); ?>" required></td>
                        </tr>
                        <tr>
                            <td><label for="gravquli_links_link_url"><strong>Link URL</strong></label></td>
                            <td><input type="url" name="gravquli_links_link_url" id="gravquli_links_link_url" class="regular-text" value="<?php echo esc_url($editing_link['url'] ?? ''); ?>" required></td>
                        </tr>
                    </table>
                    <p class="submit">
                        <input type="submit" name="gravquli_links_submit" id="gravquli_links_submit" class="button button-primary" value="<?php echo $editing_index !== null ? 'Update Link' : 'Add Link'; ?>">
                    </p>
                </form>

            </div>
        </div>

        <h2>Saved Links</h2>
        <p>Re-order each item by dragging the icon on the left to the desired location</p>
        <table class="widefat fixed">
            <thead>
            <tr>
                <th><strong>Name</strong></th>
                <th><strong>URL</strong></th>
                <th><strong>Actions</strong></th>
            </tr>
            </thead>

            <tbody id="sortable">
            <?php if (!empty($links)) : ?>
                <?php foreach ($links as $index => $link) : ?>
                    <tr>
                        <td class="drag-handle"><span class="dashicons dashicons-menu"></span> <?php echo esc_html($link['name']); ?></td>
                        <td><a href="<?php echo esc_url($link['url']); ?>" target="_blank"><?php echo esc_html($link['url']); ?></a></td>
                        <td>
                            <?php
                            $edit_url = wp_nonce_url(
                                admin_url('admin.php?page=gravyware-link-manager&edit_index=' . $index),
                                'gravquli_links_edit_link',
                                'gravquli_links_edit_nonce'
                            );
                            ?>
                            <a href="<?php echo esc_url($edit_url); ?>" class="button">Edit</a>
                            <?php
                            $delete_url = wp_nonce_url(
                                admin_url('admin.php?page=gravyware-link-manager&delete_index=' . $index),
                                'gravquli_links_delete_link',
                                'gravquli_links_delete_nonce'
                            );
                            ?>
                            <a href="<?php echo esc_url($delete_url); ?>" class="button" onclick="return confirm('Are you sure you want to delete this link?');">Delete</a>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php else : ?>
                <tr><td colspan="3">No links saved yet.</td></tr>
            <?php endif; ?>
            </tbody>
        </table>
    </div>
    <?php
}

// Handle form submissions for adding and editing
function gravquli_links_handle_form_submission() {
    $nonce = isset($_POST['gravquli_links_nonce']) ? sanitize_text_field(wp_unslash($_POST['gravquli_links_nonce'])) : '';
    if (
        isset($_POST['gravquli_links_submit']) &&
        $nonce &&
        wp_verify_nonce($nonce, 'gravquli_links_save_links')
    ) {
        $link_name = isset($_POST['gravquli_links_link_name']) ? sanitize_text_field(wp_unslash($_POST['gravquli_links_link_name'])) : '';
        $link_url = isset($_POST['gravquli_links_link_url']) ? esc_url_raw(wp_unslash($_POST['gravquli_links_link_url'])) : '';

        // Validate values
        if ((!esc_url_raw($link_url) === $link_url) || empty($link_url) || empty($link_name)) {
            exit;
        }

        $edit_index = isset($_POST['edit_index']) && $_POST['edit_index'] !== '' ? absint(wp_unslash($_POST['edit_index'])) : null;

        $links = get_option('gravquli_links_links', []);

        if (!empty($link_name) && !empty($link_url)) {
            if ($edit_index !== null && isset($links[$edit_index])) {
                // Update existing link
                $links[$edit_index] = ['name' => $link_name, 'url' => $link_url];
            } else {
                // Add new link
                $links[] = ['name' => $link_name, 'url' => $link_url];
            }

            update_option('gravquli_links_links', $links);
            wp_redirect(admin_url('admin.php?page=gravyware-link-manager'));
            exit;
        }
    }
}
add_action('admin_init', 'gravquli_links_handle_form_submission');

// Handle link deletion
function gravquli_links_handle_link_deletion() {
    $nonce = isset($_GET['gravquli_links_delete_nonce']) ? sanitize_text_field(wp_unslash($_GET['gravquli_links_delete_nonce'])) : '';
    if (
        isset($_GET['delete_index']) &&
        $nonce &&
        current_user_can('manage_options') &&
        wp_verify_nonce($nonce, 'gravquli_links_delete_link')
    ) {
        $delete_index = absint(wp_unslash($_GET['delete_index']));
        $links = get_option('gravquli_links_links', []);
        if (isset($links[$delete_index])) {
            unset($links[$delete_index]);
            update_option('gravquli_links_links', array_values($links));
        }
        wp_redirect(admin_url('admin.php?page=gravyware-link-manager'));
        exit;
    }
}
add_action('admin_init', 'gravquli_links_handle_link_deletion');

// Add a widget to the dashboard
function gravquli_links_add_dashboard_widget() {
    wp_add_dashboard_widget(
        'gravquli_links_dashboard_widget',
        'Gravyware Quick Links',
        'gravquli_links_display_dashboard_widget'
    );
}
add_action('wp_dashboard_setup', 'gravquli_links_add_dashboard_widget');

// Display the dashboard widget
function gravquli_links_display_dashboard_widget() {
    $links = get_option('gravquli_links_links', []);
    echo '<ul style="list-style-type: square;list-style-position: inside;">';
    if (!empty($links)) {
        foreach ($links as $link) {
            echo '<li><a href="' . esc_url($link['url']) . '" target="_blank">' . esc_html($link['name']) . '</a></li>';
        }
    } else {
        echo '<li>No links saved yet.</li>';
    }
    echo '</ul>';
}

// Enqueue admin scripts and styles for drag-and-drop functionality
function gravquli_links_enqueue_admin_scripts($hook) {
    wp_enqueue_style('gravquli_links_admin_style', plugin_dir_url(__FILE__) . 'admin-style.css', array(), '1.0');
    if ($hook === 'toplevel_page_gravyware-link-manager') {
        wp_enqueue_script('jquery-ui-sortable');
        wp_enqueue_script('gravquli_links_admin_scripts', plugin_dir_url(__FILE__) . 'admin-scripts.js', ['jquery-ui-sortable'], '1.0', true);
    }
}
add_action('admin_enqueue_scripts', 'gravquli_links_enqueue_admin_scripts');

// Handle AJAX request for saving the new order
function gravquli_links_save_sort_order() {
    $nonce = isset($_POST['gravquli_links_nonce']) ? sanitize_text_field(wp_unslash($_POST['gravquli_links_nonce'])) : '';
    if (
        isset($_POST['order']) &&
        $nonce &&
        wp_verify_nonce($nonce, 'gravquli_links_save_links')
    ) {
        $order = array_map('absint', explode(',', sanitize_text_field(wp_unslash($_POST['order']))));
        $links = get_option('gravquli_links_links', []);
        $new_links = [];
        foreach ($order as $index) {
            if (isset($links[$index])) {
                $new_links[] = $links[$index];
            }
        }
        update_option('gravquli_links_links', $new_links);
        wp_send_json_success();
    }
    wp_send_json_error();
}
add_action('wp_ajax_gravquli_links_save_sort_order', 'gravquli_links_save_sort_order');
