<?php
/**
 * Plugin Name: WP Asynchronous Tasks
 * Version: 1.1
 * Description: Crée une classe abstraite pour exécuter des tâches asynchrones
 * Author: 10up, Eric Mann, Luke Gedeon, John P. Bloch
 * License: MIT
 *
 * ---
 * Modifié par David DJIAN @ Graphcomment :
 *   - pour prendre en compte la constante 'SSL_VERIFY'
 */

if ( ! class_exists( 'WP_Async_Task' ) ) {
    abstract class WP_Async_Task {

        const LOGGED_IN  = 1;
        const LOGGED_OUT = 2;
        const BOTH       = 3;

        /**
         * Nombre d'arguments pour l'action principale
         *
         * @var int
         */
        protected $argument_count = 20;

        /**
         * Priorité pour déclencher l'action intermédiaire
         *
         * @var int
         */
        protected $priority = 10;

        /**
         * Nom de l'action
         *
         * @var string
         */
        protected $action;

        /**
         * Données du corps de la requête
         *
         * @var array
         */
        protected $_body_data;

        /**
         * Constructeur pour connecter les actions nécessaires
         *
         * @throws Exception Si la propriété $action n'est pas définie
         *
         * @param int $auth_level Le niveau d'authentification à utiliser
         */
        public function __construct( $auth_level = self::BOTH ) {
            if ( empty( $this->action ) ) {
                throw new Exception( 'Action non définie pour la classe ' . __CLASS__ );
            }

            add_action( $this->action, array( $this, 'launch' ), (int) $this->priority, (int) $this->argument_count );

            if ( $auth_level & self::LOGGED_IN ) {
                add_action( "admin_post_wp_async_$this->action", array( $this, 'handle_postback' ) );
            }
            if ( $auth_level & self::LOGGED_OUT ) {
                add_action( "admin_post_nopriv_wp_async_$this->action", array( $this, 'handle_postback' ) );
            }
        }

        /**
         * Ajouter l'action de fermeture pour lancer le postback réel si aucune exception n'est levée par prepare_data().
         */
        public function launch() {
            GcLogger::getLogger()->debug( 'WP_Async_Task: launch' );
            $data = func_get_args();

            try {
                $data = $this->prepare_data( $data );
            } catch ( Exception $e ) {
                GcLogger::getLogger()->error( 'WP_Async_Task: exception on prepare_data' );
                return;
            }

            $data['action'] = "wp_async_$this->action";
            $data['_nonce'] = $this->create_async_nonce();

            $this->_body_data = $data;

            if ( ! has_action( 'shutdown', array( $this, 'launch_on_shutdown' ) ) ) {
                add_action( 'shutdown', array( $this, 'launch_on_shutdown' ) );
            }
        }

        /**
         * Lancer la requête lors du hook de fermeture de WordPress
         */
        public function launch_on_shutdown() {
            if ( ! empty( $this->_body_data ) ) {
                $cookies = array();
                foreach ( $_COOKIE as $name => $value ) {
                    $cookies[] = "$name=" . urlencode( is_array( $value ) ? serialize( $value ) : $value );
                }

                // Utiliser la constante 'SSL_VERIFY' si elle est définie, sinon true par défaut
                $sslverify = defined( 'SSL_VERIFY' ) ? SSL_VERIFY : true;

                $request_args = array(
                    'timeout'   => 0.01,
                    'blocking'  => false,
                    'sslverify' => $sslverify,
                    'body'      => $this->_body_data,
                    'headers'   => array(
                        'cookie' => implode( '; ', $cookies ),
                    ),
                );

                GcLogger::getLogger()->debug( "before post wp_async_$this->action exist ? " . ( has_action( "wp_async_$this->action" ) ? 'true' : 'false' ) );
                $url = admin_url( 'admin-post.php' );
                wp_remote_post( $url, $request_args );
            }
        }

        /**
         * Vérifier que le postback est valide, puis exécuter les actions planifiées.
         */
        public function handle_postback() {
            GcLogger::getLogger()->debug( 'WP_Async_Task: handle postback' );

            if ( isset( $_POST['_nonce'] ) && $this->verify_async_nonce( $_POST['_nonce'] ) ) {
                if ( ! is_user_logged_in() ) {
                    $this->action = "nopriv_$this->action";
                }
                GcLogger::getLogger()->debug( "handle_postback wp_async_$this->action exist ? " . ( has_action( "wp_async_$this->action" ) ? 'true' : 'false' ) );
                $this->run_action();
            }

            // Éviter toute sortie supplémentaire
            add_filter( 'wp_die_handler', array( $this, 'disable_wp_die' ) );
            wp_die();
        }

        /**
         * Désactiver la sortie de wp_die lors du postback asynchrone.
         */
        public function disable_wp_die() {
            die();
        }

        /**
         * Créer un jeton aléatoire à usage unique.
         */
        protected function create_async_nonce() {
            $action = $this->get_nonce_action();
            $i      = wp_nonce_tick();

            return substr( wp_hash( $i . $action . get_class( $this ), 'nonce' ), -12, 10 );
        }

        /**
         * Vérifier que le nonce correct a été utilisé dans le délai imparti.
         *
         * @param string $nonce Le nonce à vérifier
         *
         * @return bool|int Résultat de la vérification
         */
        protected function verify_async_nonce( $nonce ) {
            $action = $this->get_nonce_action();
            $i      = wp_nonce_tick();

            // Nonce généré il y a 0-12 heures
            if ( substr( wp_hash( $i . $action . get_class( $this ), 'nonce' ), -12, 10 ) === $nonce ) {
                return 1;
            }

            // Nonce généré il y a 12-24 heures
            if ( substr( wp_hash( ( $i - 1 ) . $action . get_class( $this ), 'nonce' ), -12, 10 ) === $nonce ) {
                return 2;
            }

            // Nonce invalide
            return false;
        }

        /**
         * Obtenir une action nonce basée sur la propriété $action de la classe
         */
        protected function get_nonce_action() {
            $action = $this->action;
            if ( strpos( $action, 'nopriv_' ) === 0 ) {
                $action = substr( $action, 7 );
            }
            return "wp_async_$action";
        }

        /**
         * Préparer les données à passer au postback asynchrone
         *
         * @param array $data Les données brutes reçues par la méthode launch
         *
         * @return array Les données préparées
         */
        abstract protected function prepare_data( $data );

        /**
         * Exécuter la fonction do_action pour le postback asynchrone.
         */
        abstract protected function run_action();

    }
}
