<?php
/**
 * Plugin Name: GraphComment
 * Plugin URI: http://graphcomment.com/
 * Description: Allow the support of GraphComment into your WordPress blog.
 * Author: GraphComment <contact@graphcomment.com>
 * Author URI: https://graphcomment.com/
 * Version: 4.0.3
 * License: GPL
 * Text Domain: graphcomment-comment-system
 * Domain Path: /lang
 */

if (!defined('ABSPATH')) {
    exit; // Empêcher l'accès direct
}

define('GRAPHCOMMENT_VERSION', '4.0.3');

require_once ABSPATH . 'wp-admin/includes/plugin.php';
require_once __DIR__ . '/config.php';

// Augmenter max_execution_time pour l'importation
if (function_exists('ini_get') && function_exists('ini_set')) {
    $max_execution_time = ini_get('max_execution_time');
    if (!$max_execution_time || $max_execution_time < 5000) {
        ini_set('max_execution_time', 5000);
    }
}


/**
 * Définition des constantes pour l'API GraphComment
 */
define('AVATAR_URL', GRAPHCOMMENT_URL . '/avatar');
define('API_URL', GRAPHCOMMENT_URL . '/api');
define('API_URL_ME', API_URL . '/users/me');
define('API_URL_GET_WEBSITES', GRAPHCOMMENT_URL . '/api/website');
define('API_URL_GET_WEBSITE_API_KEYS', GRAPHCOMMENT_URL . '/api/website/:websiteId/apiKeys');
define('API_URL_TOGGLE_SSO', API_URL . '/website/:websiteId/activeSso');
define('API_URL_DEACTIVATE', API_URL . '/plugins/deactivate');
define('API_URL_SYNC_COMMENTS', API_URL . '/pub/export-comments/pubkey/:pubkey/key/:key');
define('ADMIN_URL_LOGIN', ADMIN_URL . '/#/login');
define('ADMIN_URL_LOGOUT', ADMIN_URL . '/#/logout');
define('REGISTER_URL', ADMIN_URL . '/#/register');
define('API_URL_IMPORT', IMPORT_URL . '/website/import');
define('API_URL_IMPORT_INIT', IMPORT_URL . '/website/import/init');
define('API_URL_IMPORT_RESTART', IMPORT_URL . '/website/import/restart');
define('API_URL_IMPORT_STOP', IMPORT_URL . '/website/import/cancel');
define('API_URL_IMPORT_FINISH', IMPORT_URL . '/website/import/end');

define('API_URL_OAUTH_CLIENT', API_URL . '/oauth');
define('API_URL_OAUTH_CLIENT_CREATE', API_URL_OAUTH_CLIENT . '/client');
define('API_URL_OAUTH_CLIENT_ME', API_URL_OAUTH_CLIENT_CREATE . '/me');
define('API_URL_OAUTH_CLIENT_CREATE_TOKEN', API_URL_OAUTH_CLIENT . '/token');

define('DEFAULT_LANGUAGE', 'en');

define('GC_LOG', true);

/**
 * Définition des temps de synchronisation
 */
define('SYNC_TIME_FIRST', 2);
define('SYNC_TIME_OTHER', (1 * HOUR_IN_SECONDS));
define('NUMBER_COMMENTS_IMPORT_PARTS', 30);
define('IMPORT_COMMENTS_TIMEOUT_SEC', 25);
define('SSLVERIFY', true);

require_once __DIR__ . '/class/services/gc_logger.class.php';
require_once __DIR__ . '/class/services/gc_seo_helper.class.php';
require_once __DIR__ . '/class/services/gc_comment_pairing_dao.class.php';
require_once __DIR__ . '/class/services/gc_comment_builder.class.php';
require_once __DIR__ . '/class/services/gc_params_service.class.php';

/**
 * Inclusion des fichiers du plugin.
 */
require_once __DIR__ . '/lib/sso.php';
require_once __DIR__ . '/options.php';

/**
 * Fonction pour remplacer le template des commentaires.
 *
 * @param string $value
 * @return string
 */
function graphcomment_comments_template($value) {
    if (graphcomment_should_apply()) {
        return __DIR__ . '/comments.php';
    }
    return $value;
}

/**
 * Chargement du domaine de texte pour la traduction.
 */
function graphcomment_load_textdomain() {
    load_plugin_textdomain('graphcomment-comment-system', false, dirname(plugin_basename(__FILE__)) . '/lang/');
}
add_action('plugins_loaded', 'graphcomment_load_textdomain');

/**
 * Gestion du fallback de la langue vers DEFAULT_LANGUAGE.
 *
 * @param string $mofile
 * @param string $domain
 * @return string
 */
function graphcomment_smarter_load_textdomain($mofile, $domain) {
    if ($domain == 'graphcomment-comment-system' && !is_readable($mofile)) {
        $mofile = preg_replace(
            '/graphcomment-comment-system-[a-z_A-Z]+\.mo$/',
            'graphcomment-comment-system-' . DEFAULT_LANGUAGE . '.mo',
            $mofile
        );
    }
    return $mofile;
}
add_filter('load_textdomain_mofile', 'graphcomment_smarter_load_textdomain', 10, 2);

/**
 * Vérifie si GraphComment doit être appliqué sur ce post.
 *
 * @return bool
 */
function graphcomment_should_apply() {
    $post_type = get_post_type();

    return !gcSeoHelper::request_is_bot()
        && comments_open()
        && get_option('gc_activated') === 'true'
        && (
            get_option('gc_activated_all')
            || get_option('gc_activated_from') < get_the_time('Y-m-d')
        )
        && $post_type !== 'product'; // Exclure les produits WooCommerce
}

/**
 * Ajuste le timeout des requêtes HTTP.
 *
 * @param int $time
 * @return int
 */
function graphcomment_request_timeout($time) {
    return IMPORT_COMMENTS_TIMEOUT_SEC;
}
add_filter('http_request_timeout', 'graphcomment_request_timeout');

/**
 * Supprime le bloc de commentaires par défaut dans les thèmes basés sur les blocs.
 *
 * @param string $pre_render
 * @param array $parsed_block
 * @return string
 */
function graphcomment_remove_comment_block($pre_render, $parsed_block) {
    if ('core/comments' === $parsed_block['blockName'] && graphcomment_should_apply()) {
        ob_start();
        include __DIR__ . '/comments.php';
        return ob_get_clean();
    }
    return $pre_render;
}

// Vérification du type de thème et ajout des filtres appropriés
if (function_exists('wp_is_block_theme') && wp_is_block_theme()) {
    add_filter('pre_render_block', 'graphcomment_remove_comment_block', 10, 2);
} else {
    add_filter('comments_template', 'graphcomment_comments_template');
}

/*--------------------------------------------------------------------------------------------*
 * Activation/Désactivation
 *--------------------------------------------------------------------------------------------*/

/**
 * Fonction de désactivation du plugin.
 */
function graphcomment_deactivate() {
    wp_clear_scheduled_hook('graphcomment_cron_task_sync_comments_action');
    delete_option('gc_sync_last_success');
    delete_option('gc-sync-error');
    delete_option('gc-msg');
    update_option('gc_sync_comments', 'false');
    delete_option('gc_notif_comments');
}
register_deactivation_hook(__FILE__, 'graphcomment_deactivate');

/**
 * Fonction d'activation du plugin.
 */
function graphcomment_activation() {
    add_option('Activated_Plugin', 'graphcomment');
    // Code supplémentaire à l'activation
}
register_activation_hook(__FILE__, 'graphcomment_activation');

/**
 * Chargement du plugin après activation.
 */
function graphcomment_load_plugin() {
    if (is_admin() && get_option('Activated_Plugin') == 'graphcomment') {
        delete_option('Activated_Plugin');

        if (GcCommentPairingDao::createPairingTable() === false) {
            exit;
        }

        delete_option('gc-sync-error');
        delete_option('gc-msg');
        wp_clear_scheduled_hook('graphcomment_cron_task_sync_comments_action');

        if (get_option('gc_seo_activated') === false) {
            update_option('gc_seo_activated', 'true');
        }

        delete_option('gc_notif_comments');
    }
}
add_action('admin_init', 'graphcomment_load_plugin');
