<?php
/**
 * Classe GcLogger
 * Usage :
 *   - GcLogger::get_logger()->debug('it works !');
 *   - GcLogger::get_logger()->log('my super info');
 *   - GcLogger::get_logger()->error('my super error');
 */

if ( ! class_exists( 'GcLogger' ) ) {
    class GcLogger {

        private static $_instance = null;

        private $log_file_path;
        private $prefix;

        /**
         * Initialise le répertoire de logs
         *
         * @return string|false Chemin du répertoire ou false en cas d'échec
         */
        private static function bootstrap_log_dir() {
            $uploads = wp_upload_dir();
            $my_upload_dir = trailingslashit( $uploads['basedir'] ) . 'graphcomment';

            if ( ! is_dir( $my_upload_dir ) ) {
                // Le répertoire n'existe pas, on le crée avec les permissions appropriées
                if ( ! wp_mkdir_p( $my_upload_dir ) ) {
                    error_log( 'GcLogger: Impossible de créer le répertoire de logs.' );
                    return false;
                }
            }

            return $my_upload_dir;
        }

        /**
         * Constructeur privé pour le singleton
         */
        private function __construct() {
            $log_dir = self::bootstrap_log_dir();

            if ( ! $log_dir ) {
                $this->log_file_path = false;
                return;
            }

            $this->log_file_path = trailingslashit( $log_dir ) . 'wp_graphcomment.log';
            $this->prefix        = '[' . date( 'Y-m-d H:i:s' ) . '] - ';
        }

        /**
         * Récupère l'instance du logger (singleton)
         *
         * @return GcLogger
         */
        public static function getLogger() {
            if ( is_null( self::$_instance ) ) {
                self::$_instance = new self();
            }

            return self::$_instance;
        }

        /**
         * Récupère les dernières lignes du fichier de log
         *
         * @return string
         */
        public function debugLogs() {
            if ( ! $this->log_file_path || ! file_exists( $this->log_file_path ) ) {
                return '';
            }

            $data = file( $this->log_file_path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES );

            if ( ! is_array( $data ) ) {
                return '';
            }

            $data = array_reverse( array_slice( $data, -500 ) );
            return implode( PHP_EOL, $data );
        }

        /**
         * Enregistre un message de débogage
         *
         * @param string $message
         */
        public function debug( $message ) {
            $this->write( '[debug]' . $this->prefix . $message, false );
        }

        /**
         * Enregistre un message d'information
         *
         * @param string $message
         */
        public function log( $message ) {
            $this->write( '[info]' . $this->prefix . $message, false );
        }

        /**
         * Enregistre un message d'erreur
         *
         * @param string $message
         */
        public function error( $message ) {
            $this->write( '[error]' . $this->prefix . $message, true );
        }

        /**
         * Écrit un message dans le fichier de log
         *
         * @param string $message
         * @param bool   $force
         */
        private function write( $message, $force ) {
            if ( ! $this->log_file_path ) {
                error_log( 'GcLogger: Chemin du fichier de log non défini.' );
                return;
            }

            if ( GcParamsService::getInstance()->graphcommentDebugIsActivated() || $force ) {
                $result = file_put_contents( $this->log_file_path, $message . PHP_EOL, FILE_APPEND | LOCK_EX );

                if ( false === $result ) {
                    error_log( 'GcLogger: Impossible d\'écrire dans le fichier de log.' );
                }
            }
        }
    }
}
