<?php

/**
 * Class GcCommentBuilder
 *
 * Build a WordPress comment from a GraphComment one.
 */
class GcCommentBuilder {

    private $comment;

    public function __construct( $comment ) {
        if ( is_array( $comment ) ) {
            $this->comment = (object) $comment;
        } else {
            $this->comment = $comment;
        }
        $this->convertStatusGraphCommentToWordpress();
    }

    private function convertStatusGraphCommentToWordpress() {
        if ( isset( $this->comment->spam ) && $this->comment->spam === true ) {
            $new_status = 'spam';
        } elseif ( $this->comment->status === 'refused' || $this->comment->status === 'deleted' ) {
            $new_status = 'trash';
        } elseif ( $this->comment->status === 'approved' ) {
            $new_status = '1';  // Approved status should be an integer
        } elseif ( $this->comment->status === 'pending' ) {
            $new_status = '0';  // Pending status should be an integer
        } else {
            $new_status = '0';  // Default to pending if no match
        }
        // Assign the new status to the comment
        $this->comment->status = $new_status;
    }

    private function getCreationDatetime() {
        if ( empty( $this->comment->created_at ) ) {
            return current_time( 'mysql' );
        }
        return date( 'Y-m-d H:i:s', strtotime( $this->comment->created_at ) );
    }

    public function __toString() {
        return '(' . $this->comment->_id . ') ' . $this->comment->content;
    }

    public function isFirstLevel() {
        return ! isset( $this->comment->parent_id ) || $this->comment->parent_id === null;
    }

    public function getGraphCommentId() {
        return $this->comment->_id;
    }

    public function getWordPressId() {
        return $this->comment->comment_ID;
    }

    public function setWordpressId( $id ) {
        $this->comment->comment_ID = intval( $id );
    }

    public function getParentId() {
        return $this->comment->parent_id;
    }

    public function setParent( $id ) {
        $this->comment->parent_id = intval( $id );
    }

    public function setPostId( $post_ID ) {
        $this->comment->post_ID = intval( $post_ID );
    }

    public function updateCommentInDatabase() {

        $email = $this->authorMeta( 'email' );
        if ( empty( $email ) ) {
            $email = $this->getGraphCommentId() . '@graphcomment.com';
            $this->comment->cached_author['email'] = $email;
        }

        $data = array(
            'comment_ID'           => intval( $this->comment->comment_ID ),
            'comment_author_email' => sanitize_email( $email ),
            'comment_content'      => wp_kses_post( $this->comment->content ),
            'comment_approved'     => $this->comment->status,
        );

        $up_res = wp_update_comment( $data );

        if ( $up_res === false ) {
            GcLogger::getLogger()->error( 'GcCommentBuilder::updateCommentInDatabase() - Update error ( request: ' . $this->getCommentDataAsString() . ' )' );
            return false;
        }

        return true;
    }

    public function maybeEncodeEmoji( $string ) {
        if ( function_exists( 'wp_encode_emoji' ) ) {
            return wp_encode_emoji( $string );
        }
        return $string;
    }

    public function insertCommentInDatabase() {

        $email = $this->authorMeta( 'email' );
        if ( empty( $email ) ) {
            $email = $this->getGraphCommentId() . '@graphcomment.com';
            $this->comment->cached_author['email'] = $email;
        }

        $data = array(
            'comment_post_ID'      => intval( $this->comment->post_ID ),
            'comment_author'       => sanitize_user($this->authorMeta('username')),
            'comment_author_email' => sanitize_email( $email ),
            'comment_author_IP'    => sanitize_text_field( $this->authorMeta( 'ip' ) ),
            'comment_content'      => wp_kses_post( $this->maybeEncodeEmoji( $this->comment->content ) ),
            'comment_parent'       => intval( $this->comment->parent_id ),
            'comment_date'         => $this->getCreationDatetime(),
            'comment_approved'     => $this->comment->status,
        );

        // Vérifier si un commentaire similaire existe déjà (doublon)
        $existing_comment = get_comments(array(
            'post_id' => intval($this->comment->post_ID),
            'author_email' => sanitize_email($email),
            'content' => wp_kses_post( $this->maybeEncodeEmoji( $this->comment->content ) ),
            'number' => 1,
        ));

        if (!empty($existing_comment)) {
            GcLogger::getLogger()->error('GcCommentBuilder::insertCommentInDatabase() - Duplicate comment detected');
            // Retourner 'true' même en cas de doublon
            return true;
        }

        // Insertion du commentaire
        $comment_id = wp_new_comment( $data, true );

        if ( is_wp_error( $comment_id ) ) {
            $error_message = $comment_id->get_error_message();

            // Vérification d'une erreur de doublon possible (dépend du message d'erreur)
            if (strpos($error_message, 'Duplicate entry') !== false) {
                GcLogger::getLogger()->error('GcCommentBuilder::insertCommentInDatabase() - Duplicate comment error');
                // Retourner 'true' en cas de doublon
                return true;
            } else {
                GcLogger::getLogger()->error('GcCommentBuilder::insertCommentInDatabase() - Error: ' . $error_message);
                return false;
            }
        }

        $this->setWordpressId( $comment_id );

        return true;
    }

    private function authorMeta( $property, $default = '' ) {
        // Check if the property exists in the cached_author array and return it
        if ( isset( $this->comment->cached_author[$property] ) ) {
            return $this->comment->cached_author[$property];
        }
        // Return default if the property doesn't exist
        return $default;
    }

    public function getCommentDataAsString() {
        $email = $this->authorMeta( 'email' );
        if ( empty( $email ) ) {
            $email = $this->getGraphCommentId() . '@graphcomment.com';
            $this->comment->cached_author['email'] = $email;
        }

        return '{' .
            'comment_author: \'' . sanitize_user( $this->authorMeta( 'username' ) ) . '\',' .
            'comment_author_email: \'' . $email . '\',' .
            'comment_author_ip: \'' . sanitize_text_field( $this->authorMeta( 'ip' ) ) . '\',' .
            'comment_content: \'' . wp_kses_post( $this->maybeEncodeEmoji( $this->comment->content ) ) . '\',' .
            'comment_parent: \'' . ( isset( $this->comment->parent_id ) ? intval( $this->comment->parent_id ) : 0 ) . '\',' .
            'comment_date: \'' . $this->getCreationDatetime() . '\',' .
            'comment_approved: \'' . $this->comment->status . '\'' .
            '}';
    }
}
