<?php

require_once(__DIR__ . '/../services/gc_logger.class.php');
require_once(__DIR__ . '/../services/gc_import_comment.class.php');
require_once('gc_abstract_controller.class.php');
require_once(__DIR__ . '/../services/gc_import_service.class.php');

/**
 * @controller GcImportController
 *
 * Gère l'onglet d'importation.
 * Lance l'importation des anciens commentaires WordPress dans GraphComment.
 *
 * Le constructeur doit être appelé après l'événement 'plugins_loaded' de WordPress.
 */

if ( ! class_exists( 'GcImportController' ) ) {
    class GcImportController extends GcController {

        private $action = 'import_comment';
        protected $post;

        public function __construct( $post_data = null ) {
            if ( $post_data === null ) {
                $this->post = $_POST;
            } else {
                $this->post = $post_data;
            }
        }

        public function handleOptionForm() {
            GcLogger::getLogger()->debug( 'GcImportController::handleOptionForm() (max execution time: ' . ini_get( 'max_execution_time' ) . ' seconds)' );

            $gc_import_service = new GcImportService();
            $msg = null;

            $gc_public_key = GcParamsService::getInstance()->graphcommentGetWebsite();
            if ( is_null( $gc_public_key ) ) {
                GcLogger::getLogger()->error( 'GcImportController::handleOptionForm() - No gc_public_key' );

                // L'utilisateur n'a pas encore défini la clé publique
                update_option( 'gc-msg', json_encode( array( 'type' => 'danger', 'content' => __( 'No Public Key', 'graphcomment-comment-system' ), 'active_tab' => 'importation' ) ) );
                wp_redirect( admin_url( 'admin.php?page=graphcomment-settings' ) );
                exit;
            }

            // La clé publique est définie

            // L'importation est en attente et l'utilisateur souhaite l'arrêter
            if ( isset( $this->post['gc-import-stop'] ) && $this->post['gc-import-stop'] === 'true' ) {
                GcLogger::getLogger()->debug( 'GcImportController::handleOptionForm() - Action: Stop import' );

                if ( ( $msg = $gc_import_service->stopImportation() ) !== true ) {
                    GcLogger::getLogger()->error( 'GcImportController::handleOptionForm() - Stop import failed' );

                    update_option( 'gc-msg', json_encode( array( 'active_tab' => 'importation' ) ) );
                    $gc_import_service->setImportErrorMsg( $msg );
                } else {
                    GcLogger::getLogger()->debug( 'GcImportController::handleOptionForm() - Stop import success' );

                    update_option( 'gc-msg', json_encode( array( 'type' => 'warning', 'content' => __( 'Import Cancel Success', 'graphcomment-comment-system' ), 'active_tab' => 'importation' ) ) );
                }
                wp_redirect( admin_url( 'admin.php?page=graphcomment-settings' ) );
                exit;
            }

            // L'utilisateur a relancé l'importation suite à une erreur
            if ( isset( $this->post['gc-import-restart'] ) && $this->post['gc-import-restart'] === 'true' ) {
                GcLogger::getLogger()->debug( 'GcImportController::handleOptionForm() - Action: Restart import' );

                if ( ( $msg = $gc_import_service->restartImportationInit() ) !== true ) {
                    GcLogger::getLogger()->error( 'GcImportController::handleOptionForm() - Restart import failed' );

                    update_option( 'gc-msg', json_encode( array( 'active_tab' => 'importation' ) ) );
                    $gc_import_service->setImportErrorMsg( $msg );
                } else {
                    GcLogger::getLogger()->debug( 'GcImportController::handleOptionForm() - Restart import success' );

                    update_option( 'gc-msg', json_encode( array( 'type' => 'success', 'content' => __( 'Import Restart Success', 'graphcomment-comment-system' ), 'active_tab' => 'importation' ) ) );

                    // Lancer l'action d'importation liée de manière asynchrone à wp_async_import_comment
                    do_action( $this->action, $this->post );
                }
                wp_redirect( admin_url( 'admin.php?page=graphcomment-settings' ) );
                exit;
            }

            GcLogger::getLogger()->debug( 'GcImportController::handleOptionForm() - Init import' );

            if ( ( $msg = $gc_import_service->importInit() ) !== true ) {
                GcLogger::getLogger()->error( 'GcImportController::handleOptionForm() - Init import failed' );
                update_option( 'gc-msg', json_encode( array( 'active_tab' => 'importation' ) ) );
                if ( $msg !== 'skip' ) {
                    $gc_import_service->setImportationError( $msg );
                }
                wp_redirect( admin_url( 'admin.php?page=graphcomment-settings' ) );
                exit;
            }

            // Rediriger l'utilisateur directement vers l'onglet d'importation
            update_option( 'gc-msg', json_encode( array( 'active_tab' => 'importation' ) ) );

            // Lancer l'action d'importation liée de manière asynchrone à wp_async_import_comment
            do_action( $this->action, $this->post );

            wp_redirect( admin_url( 'admin.php?page=graphcomment-settings' ) );
            exit;
        }

        public static function registerAsyncAction() {
            add_action( 'wp_async_import_comment', array( __CLASS__, 'importComments' ) );
        }

        public static function importComments() {
            $gc_import_service = new GcImportService();
            while ( $gc_import_service->importNextComments() ) {
                // Continue l'importation des commentaires
            }
        }
    }
}
