<?php
namespace GrUpnPf;

if (!defined('ABSPATH'))
    exit;

class WC_Settings_GrUpnPf extends \WC_Settings_Page
{
    public function __construct()
    {
        $this->id = "grupnpf_settings";
        $this->label = __("UPN Nastavitve", "gr-upn-payment-form");

        add_filter("woocommerce_settings_tabs_array", [$this, "add_settings_tab"], 50);
        add_action("woocommerce_settings_" . $this->id, [$this, "output"]);
        add_action("woocommerce_settings_save_" . $this->id, [$this, "save"]);
        add_action('admin_init', [$this, 'register_hooks']);
    }

    public function register_hooks()
    {
        add_filter('woocommerce_admin_settings_sanitize_option_grupnpf_iban_prejemnika', [$this, 'grupnpf_sanitize_iban_for_save'], 10, 3);
        add_filter('woocommerce_admin_settings_get_option_grupnpf_iban_prejemnika', [$this, 'grupnpf_format_iban_for_display'], 10, 2);
    }

    public function grupnpf_format_iban_for_display($value, $option)
    {
        $sanitized_value = preg_replace('/[^a-zA-Z0-9]/', '', $value);
        $sanitized_value = substr($sanitized_value, 0, 19);
        return trim(chunk_split($sanitized_value, 4, ' '));
    }

    public function grupnpf_sanitize_iban_for_save($value, $option, $raw_value)
    {
        $raw_value = strtoupper(str_replace(' ', '', $raw_value));

        // Validate Slovenian IBAN format (SI + 17 digits)
        if (!preg_match('/^SI\d{17}$/', $raw_value)) {
            \add_settings_error(
                $option['id'],
                'invalid_iban',
                __('Neveljaven IBAN format. Uporabi obliko: SI56 0000 0000 0000 000', 'gr-upn-payment-form'),
                'error'
            );
            return get_option($option['id']); // revert to previous valid value
        }

        // check IBAN checksum
        if (!$this->validate_iban_checksum($raw_value)) {
            \add_settings_error(
                $option['id'],
                'invalid_iban_checksum',
                __('IBAN ni veljaven (napačen nadzorni vsoti).', 'gr-upn-payment-form'),
                'error'
            );
            return get_option($option['id']);
        }

        // Format to groups of 4 for display
        return trim(chunk_split($raw_value, 4, ' '));
    }

    // Simple IBAN checksum validation
    private function validate_iban_checksum($iban)
    {
        $iban = strtolower(str_replace(' ', '', $iban));
        $iban = substr($iban, 4) . substr($iban, 0, 4);
        $iban = preg_replace_callback('/[a-z]/', function ($m) {
            return ord($m[0]) - 87;
        }, $iban);
        $mod = intval($iban[0]);
        for ($i = 1, $len = strlen($iban); $i < $len; $i++) {
            $mod = ($mod * 10 + intval($iban[$i])) % 97;
        }
        return $mod === 1;
    }


    public function add_settings_tab($tabs)
    {
        $tabs[$this->id] = $this->label;
        return $tabs;
    }

    public function get_settings()
    {
        $gateways = \WC_Payment_Gateways::instance();
        $availableGateways = $gateways->get_available_payment_gateways();

        if (isset($availableGateways["bacs"]) && !empty($availableGateways["bacs"]->account_details)) {
            $bacs_account = (object) $availableGateways["bacs"]->account_details[0];
            // Format IBAN for display
            $bacs_account->iban = preg_replace('/(\w{4})/', '$1 ', preg_replace('/[^a-zA-Z0-9]/', '', $bacs_account->iban));
        } else {
            // BACS not available or not configured. Create a dummy object with empty values.
            $bacs_account = (object) [
                'sort_code' => '',
                'iban' => '',
                'account_name' => '',
            ];
            // Display a warning notice instead of a fatal error.
            add_action("admin_notices", function () {
                echo '<div class="notice notice-warning is-dismissible"><p>' .
                    esc_html__('Direktno bančno nakazilo (BACS) ni nastavljeno. Podatke za UPN plačila boste morali vnesti ročno.', 'gr-upn-payment-form') .
                    '</p></div>';
            });
        }

        $pro_notice = [];
        if (!defined('GRUPNPF_PRO_ACTIVE')) {
            $pro_notice[] = [
                'title' => __('Nadgradnja na PRO', 'gr-upn-payment-form'),
                'type' => 'title',
                'desc' => __('Za dodatne funkcioalnosti si zagotovite PRO različico. <a href="https://gregarednak.si/gr-upn-payment-form" target="_blank">Kupi sedaj</a>.', 'gr-upn-payment-form'),
                'id' => 'grupnpf_pro_notice'
            ];
        }

        $is_pro_active = defined('GRUPNPF_PRO_ACTIVE');
        $pro_settings_disabled = !$is_pro_active ? ['disabled' => 'disabled'] : [];
        $pro_settings_desc = !$is_pro_active ? __('To je PRO funkcionalnost.', 'gr-upn-payment-form') . ' <a href="https://gregarednak.si/gr-upn-payment-form" target="_blank">' . __('Kupi sedaj', 'gr-upn-payment-form') . '</a>' : '';

        $main_settings = [
            [
                "title" => __("UPN Nastavitve", "gr-upn-payment-form"),
                "type" => "title",
                "desc" => __("Nastavitve za UPN QR plačila.", "gr-upn-payment-form"),
                "id" => "grupnpf_settings_options",
            ],
            [
                "title" => __("Namen plačila", "gr-upn-payment-form"),
                "id" => "grupnpf_namen_placila",
                "type" => "text",
                "default" => "Plačilo naročila",
                "desc_tip" => true,
                'custom_attributes' => ['required' => 'required'],
            ],
            [
                "title" => __("Koda namena", "gr-upn-payment-form"),
                "id" => "grupnpf_koda_namena",
                "type" => "text",
                "default" => $bacs_account->sort_code,
                "desc_tip" => true,
                'custom_attributes' => ['required' => 'required'],
            ],
            [
                "title" => __("IBAN prejemnika", "gr-upn-payment-form"),
                "id" => "grupnpf_iban_prejemnika",
                "type" => "text",
                "default" => $bacs_account->iban,
                "desc_tip" => true,
                'custom_attributes' => [
                    'required' => 'required',
                    'placeholder' => 'SI56 0000 0000 0000 000',
                    'pattern' => 'SI[0-9 ]{17,22}', // basic client-side check
                    'title' => __('Vnesite veljaven IBAN (npr. SI56 0000 0000 0000 000)', 'gr-upn-payment-form'),
                ],
            ],
            [
                "title" => __("Referenca prejemnika", "gr-upn-payment-form"),
                "id" => "grupnpf_referenca_prejemnika",
                "type" => "text",
                "default" => "SI00",
                "desc_tip" => true,
                'custom_attributes' => ['required' => 'required'],
            ],
            [
                "title" => __("Ime prejemnika", "gr-upn-payment-form"),
                "id" => "grupnpf_ime_prejemnika",
                "type" => "text",
                "default" => $bacs_account->account_name,
                "desc_tip" => true,
                'custom_attributes' => ['required' => 'required'],
            ],
            [
                "title" => __("Ulica prejemnika", "gr-upn-payment-form"),
                "id" => "grupnpf_ulica_prejemnika",
                "type" => "text",
                "default" => WC()->countries->get_base_address(),
                "desc_tip" => true,
                'custom_attributes' => ['required' => 'required'],
            ],
            [
                "title" => __("Kraj prejemnika", "gr-upn-payment-form"),
                "id" => "grupnpf_kraj_prejemnika",
                "type" => "text",
                "default" => WC()->countries->get_base_city(),
                "desc_tip" => true,
                'custom_attributes' => ['required' => 'required'],
            ],
            [
                "title" => __("Poštna številka prejemnika", "gr-upn-payment-form"),
                "id" => "grupnpf_posta_prejemnika",
                "type" => "text",
                "default" => WC()->countries->get_base_postcode(),
                "desc_tip" => true,
                'custom_attributes' => ['required' => 'required'],
            ],
            [
                "title" => __("Predpona imena UPN slike", "gr-upn-payment-form"),
                "id" => "grupnpf_predpona_upn_slike",
                "type" => "text",
                "default" => "qr-upn",
                "desc_tip" => true,
                'custom_attributes' => ['required' => 'required'],
            ],
            [
                'type' => 'sectionend',
                'id' => 'grupnpf_settings_options'
            ],
            [
                'title' => __('Nastavitve za odpravljanje napak', 'gr-upn-payment-form'),
                'type' => 'title',
                'id' => 'grupnpf_debug_options'
            ],
            [
                'title' => __('Omogoči beleženje napak', 'gr-upn-payment-form'),
                'id' => 'grupnpf_enable_debug_logging',
                'type' => 'checkbox',
                'default' => 'no',
                'desc' => __('Če je omogočeno, bo vtičnik zapisoval podrobne informacije v datoteko wp-content/debug.log. Uporabno za diagnosticiranje težav.', 'gr-upn-payment-form'),
            ],
            [
                'type' => 'sectionend',
                'id' => 'grupnpf_debug_options_end'
            ],
            [
                'title' => __('Prikaz UPN (PRO)', 'gr-upn-payment-form'),
                'type' => 'title',
                'id' => 'grupnpf_display_options'
            ],
            [
                "title" => __("Dodaj UPN kot sliko v email sporočilo", "gr-upn-payment-form"),
                "id" => "grupnpf_email_vstavi_upn",
                "type" => "checkbox",
                "default" => "yes",
                "desc" => $pro_settings_desc,
                'custom_attributes' => $pro_settings_disabled,
            ],
            [
                "title" => __("Dodaj UPN v email priponko", "gr-upn-payment-form"),
                "id" => "grupnpf_email_priponka",
                "type" => "checkbox",
                "default" => "yes",
                "desc" => $pro_settings_desc,
                'custom_attributes' => $pro_settings_disabled,
            ],
            [
                "title" => __("Dodaj UPN na zahvalno stran", "gr-upn-payment-form"),
                "id" => "grupnpf_thankyou_page",
                "type" => "checkbox",
                "default" => "yes",
                "desc" => $pro_settings_desc,
                'custom_attributes' => $pro_settings_disabled,
            ],
            [
                "type" => "sectionend",
                "id" => "grupnpf_display_options_end",
            ],
        ];

        $all_settings = array_merge($pro_notice, $main_settings);

        return apply_filters("woocommerce_get_settings_" . $this->id, $all_settings);
    }
}