<?php
namespace GrUpnPf;

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class GrUpnPf_Main
{
    public function __construct()
    {
        add_action('woocommerce_init', [$this, 'init'], 20);
        add_action('wp_ajax_grupnpf_send_upn_to_customer', [$this, 'ajax_send_upn_to_customer']);

        if (is_admin()) {
            add_action('admin_init', [$this, 'admin_init']);
            add_action('admin_notices', [$this, 'admin_notices']);
            add_filter('woocommerce_get_settings_pages', [$this, 'add_upn_settings']);
            add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
            add_filter('woocommerce_admin_order_actions', [$this, 'add_order_list_send_upn_action'], 100, 2);
            add_action('admin_action_grupnpf_send_upn_from_column', [$this, 'handle_send_upn_from_column_action']);

            $plugin = plugin_basename(GRUPNPF_PLUGIN_FILE);
            add_filter("plugin_action_links_$plugin", [$this, 'add_plugin_settings_link']);
            add_filter('plugin_row_meta', [$this, 'add_plugin_row_meta'], 10, 2);
        }

        if (!wp_next_scheduled('grupnpf_daily_cleanup')) {
            wp_schedule_event(time(), 'daily', 'grupnpf_daily_cleanup');
        }
        add_action('grupnpf_daily_cleanup', [$this, 'cleanup_images']);
    }

    public function init()
    {
        new GrUpnPf_Logic_UPN();
    }

    public function ajax_send_upn_to_customer()
    {
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';

        if (empty($nonce) || !wp_verify_nonce($nonce, 'grupnpf_send_upn_nonce')) {
            wp_send_json_error(__('Neveljavna zahteva.', 'gr-upn-payment-form'));
            return;
        }

        $order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;
        if ($order_id > 0) {
            $order = wc_get_order($order_id);
            if ($order && current_user_can('edit_shop_orders')) {
                if ($this->send_upn_email($order_id)) {
                    wp_send_json_success(__('UPN uspešno poslan.', 'gr-upn-payment-form'));
                } else {
                    wp_send_json_error(__('Napaka pri pošiljanju UPN.', 'gr-upn-payment-form'));
                }
                return;
            }
        }

        wp_send_json_error(__('Napaka pri pošiljanju UPN.', 'gr-upn-payment-form'));
    }

    public function send_upn_email($order_id)
    {
        $order = wc_get_order($order_id);
        if (!$order || 'bacs' !== $order->get_payment_method()) {
            return false;
        }

        $UPN = new GrUpnPf_Logic_UPN();
        $png = $UPN->genUPN($order);
        if (!$png) {
            return false;
        }

        $upn_image_path = $UPN->saveUpnImage($png, $order->get_id());

        if ($upn_image_path) {
            $subject = __('UPN obrazec za naročilo #', 'gr-upn-payment-form') . $order_id;
            $message = __('V priponki je UPN obrazec za lažje plačilo na naš TRR.', 'gr-upn-payment-form');
            $headers = ['Content-Type: text/html; charset=UTF-8'];
            $recipient = $order->get_billing_email();
            $attachments = [$upn_image_path];

            $sent = wp_mail($recipient, $subject, $message, $headers, $attachments);

            if (file_exists($upn_image_path)) {
                wp_delete_file($upn_image_path);
            }
            return $sent;
        }
        return false;
    }

    public function admin_init()
    {
        if (wp_doing_ajax())
            return;
        if (!in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) {
            echo '<div class="notice notice-error"><p>' .
                esc_html__('GR UPN plačilni nalog - QR vtičnik potrebuje WooCommerce za delovanje!', 'gr-upn-payment-form') .
                '</p></div>';
        }
    }

    public function admin_notices()
    {
        if ($notice = get_transient('grupnpf_admin_notice')) {
            printf(
                '<div class="notice notice-%1$s is-dismissible"><p>%2$s</p></div>',
                esc_attr($notice['type']),
                esc_html($notice['message'])
            );
            delete_transient('grupnpf_admin_notice');
        }

        $required_options = [
            "grupnpf_ime_prejemnika",
            "grupnpf_ulica_prejemnika",
            "grupnpf_posta_prejemnika",
            "grupnpf_kraj_prejemnika",
            "grupnpf_iban_prejemnika",
            "grupnpf_koda_namena",
            "grupnpf_referenca_prejemnika",
            "grupnpf_namen_placila",
        ];

        $missing_settings = false;
        foreach ($required_options as $option) {
            if (empty(get_option($option))) {
                $missing_settings = true;
                break;
            }
        }

        if ($missing_settings) {
            $settings_url = admin_url('admin.php?page=wc-settings&tab=grupnpf_settings');
            $error_message = sprintf(
                /* translators: %s: URL to the plugin settings page */
                __('Manjkajoče UPN nastavitve. Prosimo, <a href="%s">preverite nastavitve</a> plačilnega naloga.', 'gr-upn-payment-form'),
                esc_url($settings_url)
            );

            echo '<div class="notice notice-error"><p>' . wp_kses_post($error_message) . '</p></div>';
        }
    }

    public function add_upn_settings($settings)
    {
        if (class_exists('WC_Settings_Page')) {
            $settings[] = new WC_Settings_GrUpnPf();
        }
        return $settings;
    }

    public function enqueue_scripts($hook)
    {
        global $post_type;
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $current_tab = isset($_GET['tab']) ? sanitize_text_field(wp_unslash($_GET['tab'])) : '';

        // Enqueue on settings page
        if ('woocommerce_page_wc-settings' === $hook && 'grupnpf_settings' === $current_tab) {
            $this->enqueue_assets();
            return;
        }

        // Enqueue on orders list page
        if (('edit.php' === $hook && 'shop_order' === $post_type) || $hook === 'woocommerce_page_wc-orders') {
            $this->enqueue_assets(true, false); // CSS only
            return;
        }

    }

    private function enqueue_assets($css = true, $js = true)
    {
        if ($css) {
            $css_path = plugin_dir_path(GRUPNPF_PLUGIN_FILE) . 'src/assets/css/main.css';
            $css_version = file_exists($css_path) ? filemtime($css_path) : '1.1.0';
            wp_register_style('grupnpf_main_css', plugin_dir_url(GRUPNPF_PLUGIN_FILE) . 'src/assets/css/main.css', [], $css_version);
            wp_enqueue_style('grupnpf_main_css');
        }

        if ($js) {
            wp_enqueue_script('grupnpf_admin_script', plugin_dir_url(GRUPNPF_PLUGIN_FILE) . 'src/assets/js/admin.js', ['jquery'], '1.1.0', true);
            wp_enqueue_script('grupnpf_admin_iban', plugin_dir_url(GRUPNPF_PLUGIN_FILE) . 'src/assets/js/admin-iban.js', [], '1.0.0', true);
            wp_localize_script('grupnpf_admin_script', 'grupnpf_ajax', ['ajaxurl' => admin_url('admin-ajax.php')]);
        }
    }

    public function add_plugin_settings_link($links)
    {
        $links[] = '<a href="admin.php?page=wc-settings&tab=grupnpf_settings">' . esc_html__('Nastavitve', 'gr-upn-payment-form') . '</a>';
        return $links;
    }

    public function add_plugin_row_meta($plugin_meta, $plugin_file)
    {
        if (strpos($plugin_file, 'gr-upn.php') !== false) {
            $plugin_meta[] = '<a href="admin.php?page=wc-settings&tab=grupnpf_settings">' . esc_html__('Nastavitve', 'gr-upn-payment-form') . '</a>';
        }
        return $plugin_meta;
    }

    public function cleanup_images()
    {
        $upload_dir = wp_upload_dir();
        $dir = $upload_dir['basedir'] . '/grupnpf-qrcodes';
        $files = glob($dir . '/*.png');
        foreach ($files as $file) {
            if (is_file($file) && time() - filemtime($file) > 86400) {
                wp_delete_file($file);
            }
        }
    }

    public function add_order_list_send_upn_action($actions, $order)
    {
        if ('bacs' === $order->get_payment_method()) {
            $url = wp_nonce_url(
                admin_url('admin.php?action=grupnpf_send_upn_from_column&order_id=' . $order->get_id()),
                'grupnpf_send_upn_from_column_nonce',
                'grupnpf_nonce'
            );
            $actions['grupnpf_send_upn'] = [
                'url' => $url,
                'name' => __('Pošlji UPN', 'gr-upn-payment-form'),
                'action' => 'grupnpf-send-upn-icon',
            ];
        }
        return $actions;
    }

    public function handle_send_upn_from_column_action()
    {
        $nonce = isset($_GET['grupnpf_nonce']) ? sanitize_text_field(wp_unslash($_GET['grupnpf_nonce'])) : '';
        $order_id = isset($_GET['order_id']) ? intval($_GET['order_id']) : 0;

        if (!$order_id || empty($nonce) || !wp_verify_nonce($nonce, 'grupnpf_send_upn_from_column_nonce')) {
            wp_die(esc_html__('Neveljavna zahteva.', 'gr-upn-payment-form'));
        }

        if (!current_user_can('edit_shop_orders')) {
            wp_die(esc_html__('Nimate dovoljenja za izvedbo tega dejanja.', 'gr-upn-payment-form'));
        }

        $sent = $this->send_upn_email($order_id);
        $message = $sent ? __('UPN e-pošta uspešno poslana.', 'gr-upn-payment-form') : __('Napaka pri pošiljanju UPN e-pošte.', 'gr-upn-payment-form');
        $notice_type = $sent ? 'success' : 'error';

        set_transient('grupnpf_admin_notice', ['message' => $message, 'type' => $notice_type], 5);

        wp_safe_redirect(wp_get_referer() ? wp_get_referer() : admin_url('edit.php?post_type=shop_order'));
        exit;
    }
}