<?php
/**
 * Tracker Class
 * Handles the pageview tracking functionality
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class GPTrends_Tracker {

    private $default_api_url = 'https://api.gptrends.io/v1/intake/pageview';

    public function __construct() {
        add_action( 'plugins_loaded', array( $this, 'start_timer' ) );
        add_action( 'shutdown', array( $this, 'track_pageview' ) );
    }

    public function start_timer() {
        if ( ! isset( $GLOBALS['gptrends_request_start'] ) ) {
            $GLOBALS['gptrends_request_start'] = microtime( true );
        }
    }

    public function track_pageview() {
        $options = get_option( 'gptrends_options', array() );

        if ( empty( $options['enabled'] ) || empty( $options['website_key'] ) ) {
            return;
        }

        if ( is_admin() || $this->is_static_resource() ) {
            return;
        }

        $user_agent = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';
        if ( empty( $user_agent ) ) {
            return;
        }

        $start_time = isset( $GLOBALS['gptrends_request_start'] ) ? $GLOBALS['gptrends_request_start'] : microtime( true );
        $start_time = isset( $GLOBALS['gptrends_request_start'] ) ? $GLOBALS['gptrends_request_start'] : microtime( true );
        $duration   = round( ( microtime( true ) - $start_time ) * 1000 );

        $data = array(
            'url'              => $this->get_current_url(),
            'userAgent'        => $user_agent,
            'ref'              => isset( $_SERVER['HTTP_REFERER'] ) ? esc_url_raw( wp_unslash( $_SERVER['HTTP_REFERER'] ) ) : '',
            'ip'               => $this->get_client_ip(),
            'websiteKey'       => sanitize_text_field( $options['website_key'] ),
            'method'           => isset( $_SERVER['REQUEST_METHOD'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_METHOD'] ) ) : 'GET',
            'status'           => http_response_code(),
            'duration'         => $duration,
            'integrationType'  => 'wordpress',
            'sdk'              => '@gptrends/wordpress',
            'sdkVersion'       => GPTRENDS_VERSION,
        );

        $this->send_tracking_request( $data );
    }

    private function get_current_url() {
        $protocol = isset( $_SERVER['HTTPS'] ) && 'on' === $_SERVER['HTTPS'] ? 'https' : 'http';
        $host     = isset( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : '';
        $uri      = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';

        return $protocol . '://' . $host . $uri;
    }

    private function is_static_resource() {
        $uri = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';

        $static_extensions = array( 'css', 'js', 'jpg', 'jpeg', 'png', 'gif', 'ico', 'svg', 'woff', 'woff2', 'ttf', 'eot' );
        $pattern           = '/\.(' . implode( '|', $static_extensions ) . ')$/i';

        if ( preg_match( $pattern, $uri ) ) {
            return true;
        }

        if ( strpos( $uri, 'wp-login' ) !== false ) {
            return true;
        }

        return false;
    }

    private function get_client_ip() {
        $ip_keys = array(
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_X_CLUSTER_CLIENT_IP',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR',
        );

        foreach ( $ip_keys as $key ) {
            if ( ! empty( $_SERVER[ $key ] ) ) {
                $ips = explode( ',', sanitize_text_field( wp_unslash( $_SERVER[ $key ] ) ) );
                $ip  = trim( $ips[0] );

                if ( filter_var( $ip, FILTER_VALIDATE_IP ) ) {
                    return $ip;
                }
            }
        }

        return '0.0.0.0';
    }

    private function send_tracking_request( $data ) {
        $options = get_option( 'gptrends_options', array() );

        $api_url = ! empty( $options['api_endpoint'] ) ? $options['api_endpoint'] : $this->default_api_url;

        wp_remote_post(
            $api_url,
            array(
                'timeout'    => 5,
                'blocking'   => false,
                'sslverify'  => true,
                'user-agent' => 'GPTrends-WordPress-Plugin/' . GPTRENDS_VERSION,
                'headers'    => array(
                    'Content-Type' => 'application/json',
                ),
                'body'       => wp_json_encode( $data ),
            )
        );
    }
}
