<?php
/**
 * Admin Class
 * Handles admin interface and settings
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class GPTrends_Admin {

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_menu' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_filter( 'plugin_action_links_' . plugin_basename( GPTRENDS_PLUGIN_FILE ), array( $this, 'add_action_links' ) );
    }

    public function add_menu() {
        add_options_page(
            __( 'GPTrends Analytics', 'gptrends-agent-analytics' ),
            __( 'GPTrends', 'gptrends-agent-analytics' ),
            'manage_options',
            'gptrends-analytics',
            array( $this, 'render_settings_page' )
        );
    }

    public function register_settings() {
        register_setting(
            'gptrends_options_group',
            'gptrends_options',
            array( $this, 'sanitize_options' )
        );
    }

    public function sanitize_options( $input ) {
        if ( ! isset( $_POST['gptrends_settings_nonce'] ) ||
            ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['gptrends_settings_nonce'] ) ), 'gptrends_save_settings' ) ) {
            add_settings_error(
                'gptrends_messages',
                'gptrends_nonce_error',
                __( 'Security check failed. Please try again.', 'gptrends-agent-analytics' ),
                'error'
            );
            return get_option( 'gptrends_options' );
        }

        $sanitized = array();

        $sanitized['enabled']      = ! empty( $input['enabled'] );
        $sanitized['website_key']  = isset( $input['website_key'] ) ? sanitize_text_field( $input['website_key'] ) : '';
        $sanitized['api_endpoint'] = isset( $input['api_endpoint'] ) ? esc_url_raw( $input['api_endpoint'] ) : '';

        return $sanitized;
    }

    public function add_action_links( $links ) {
        $settings_link = sprintf(
            '<a href="%s">%s</a>',
            admin_url( 'options-general.php?page=gptrends-analytics' ),
            __( 'Settings', 'gptrends-agent-analytics' )
        );

        array_unshift( $links, $settings_link );
        return $links;
    }

    public function render_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $options = get_option(
            'gptrends_options',
            array(
                'enabled'      => true,
                'website_key'  => '',
                'api_endpoint' => '',
            )
        );

        if ( isset( $_GET['settings-updated'] ) ) {
            add_settings_error(
                'gptrends_messages',
                'gptrends_message',
                __( 'Settings saved successfully.', 'gptrends-agent-analytics' ),
                'success'
            );
        }

        settings_errors( 'gptrends_messages' );
        ?>
        <div class="wrap">
            <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>

            <div class="card" style="max-width: 800px;">
                <h2><?php esc_html_e( 'About GPTrends', 'gptrends-agent-analytics' ); ?></h2>
                <p>
                    <?php esc_html_e( 'GPTrends helps companies understand how AI agents and bots crawl, reference, and interact with their content — turning AI visibility optimization from a black box into data-driven actions.', 'gptrends-agent-analytics' ); ?>
                </p>
                <p>
                    <strong><?php esc_html_e( 'Get your Website ID:', 'gptrends-agent-analytics' ); ?></strong>
                    <a href="https://gptrends.io/app/auth" target="_blank" rel="noopener noreferrer"><?php esc_html_e( 'Sign up at GPTrends.io', 'gptrends-agent-analytics' ); ?></a>
                </p>
            </div>

            <form method="post" action="options.php">
                <?php
                settings_fields( 'gptrends_options_group' );
                wp_nonce_field( 'gptrends_save_settings', 'gptrends_settings_nonce' );
                ?>

                <table class="form-table" role="presentation">
                    <tbody>
                    <tr>
                        <th scope="row">
                            <label for="gptrends_enabled">
                                <?php esc_html_e( 'Enable Tracking', 'gptrends-agent-analytics' ); ?>
                            </label>
                        </th>
                        <td>
                            <fieldset>
                                <label>
                                    <input
                                            type="checkbox"
                                            name="gptrends_options[enabled]"
                                            id="gptrends_enabled"
                                            value="1"
                                        <?php checked( ! empty( $options['enabled'] ) ); ?>
                                    />
                                    <?php esc_html_e( 'Track AI agents, bots & crawlers (disable to pause tracking)', 'gptrends-agent-analytics' ); ?>
                                </label>
                            </fieldset>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="gptrends_website_key">
                                <?php esc_html_e( 'Website ID', 'gptrends-agent-analytics' ); ?>
                            </label>
                        </th>
                        <td>
                            <input
                                    type="text"
                                    name="gptrends_options[website_key]"
                                    id="gptrends_website_key"
                                    value="<?php echo esc_attr( $options['website_key'] ); ?>"
                                    class="regular-text"
                                    placeholder="gptrends_secret_********************************"
                            />
                            <p class="description">
                                <?php
                                echo wp_kses_post(
                                    sprintf(
                                    /* translators: 1: opening link tag, 2: closing link tag */
                                        __( 'Enter your Website ID from your %1$sGPTrends dashboard%2$s', 'gptrends-agent-analytics' ),
                                        '<a href="https://gptrends.io" target="_blank" rel="noopener noreferrer">',
                                        '</a>'
                                    )
                                );
                                ?>
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="gptrends_api_endpoint">
                                <?php esc_html_e( 'API Endpoint', 'gptrends-agent-analytics' ); ?>
                            </label>
                        </th>
                        <td>
                            <input
                                    type="url"
                                    name="gptrends_options[api_endpoint]"
                                    id="gptrends_api_endpoint"
                                    value="<?php echo esc_attr( $options['api_endpoint'] ); ?>"
                                    class="regular-text"
                                    placeholder=""
                            />
                            <p class="description">
                                <?php esc_html_e( 'API Endpoint (for testing)', 'gptrends-agent-analytics' ); ?>
                            </p>
                        </td>
                    </tr>
                    </tbody>
                </table>

                <?php submit_button(); ?>
            </form>

            <div class="card" style="max-width: 800px; margin-top: 20px;">
                <h3><?php esc_html_e( 'Need Help?', 'gptrends-agent-analytics' ); ?></h3>
                <ul>
                    <li><a href="https://docs.gptrends.io/agent-analytics" target="_blank" rel="noopener noreferrer"><?php esc_html_e( 'Documentation', 'gptrends-agent-analytics' ); ?></a></li>
                </ul>
            </div>
        </div>
        <?php
    }
}
